/* Bijiben
 * Copyright (C) Pierre-Yves Luyten 2012 <py@luyten.fr>
 *
 * bijiben is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * bijiben is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib/gi18n.h>

#include "bjb-main-toolbar.h"
#include "bjb-rename-note.h"
#include "bjb-window-base.h"

/* select_mode + two different standard mode */
typedef enum
{
  BJB_TOOLBAR_0,
  BJB_TOOLBAR_STD_LIST,
  BJB_TOOLBAR_STD_ICON,
  BJB_TOOLBAR_SELECT,
  BJB_TOOLBAR_NUM
} BjbToolbarType ;

struct _BjbMainToolbarPrivate
{
  /* Controllers */
  GdMainView     *view;
  BjbToolbarType  type;
  BjbMainView    *parent;
  BjbController  *controller;

  /* Buttons */
  GtkWidget      *new;  
  GtkWidget      *list;
  GtkWidget      *grid;
  GtkWidget      *select;

  /* Signal Handlers */
  gulong         finish_sig;
  gulong         update_selection;
  gulong         search_handler;
};

/* GObject properties */

enum {
  PROP_0,
  PROP_VIEW,
  PROP_PARENT,
  PROP_CONTROLLER,
  NUM_PROPERTIES
};

static GParamSpec *properties[NUM_PROPERTIES] = { NULL, };

#define BJB_MAIN_TOOLBAR_GET_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE ((o), BJB_TYPE_MAIN_TOOLBAR, BjbMainToolbarPrivate))

G_DEFINE_TYPE (BjbMainToolbar, bjb_main_toolbar, GD_TYPE_MAIN_TOOLBAR);

/* Callbacks */

static void
on_new_note_clicked (GtkWidget *but, BjbMainView *view)
{
  BijiNoteObj *result ;
  BijiNoteBook *book ;

  /* append note to collection */
  book = bjb_window_base_get_book(bjb_main_view_get_window(view));
  result = biji_note_book_get_new_note_from_string (book, "");

  /* Go to that note */
  switch_to_note_view(view,result);
}

static void populate_main_toolbar(BjbMainToolbar *self);

void
on_selection_mode_changed (BjbMainToolbar *self)
{
  GtkStyleContext *context;
  GdMainView *view = self->priv->view;
  GtkWidget *widget = GTK_WIDGET(self);
  context = gtk_widget_get_style_context (widget);

  if (!gd_main_view_get_selection_mode (view))
    gtk_style_context_remove_class (context, "selection-mode");

  else
    gtk_style_context_add_class (context, "selection-mode");

  gtk_widget_reset_style (widget);
  populate_main_toolbar(self);
  return ;
}

static void
on_selection_mode_clicked (GtkWidget *button, BjbMainToolbar *self)
{
  if (gd_main_view_get_selection_mode (self->priv->view))
    gd_main_view_set_selection_mode (self->priv->view, FALSE);

  else
    gd_main_view_set_selection_mode (self->priv->view, TRUE);

  on_selection_mode_changed (self);
}

static gboolean
on_view_mode_clicked (GtkWidget *button, BjbMainToolbar *self)
{
  GdMainView *view = self->priv->view ;
  GdMainViewType current = gd_main_view_get_view_type(view);
    
  switch ( current )
  {
    case GD_MAIN_VIEW_ICON :
      gd_main_view_set_view_type ( view ,GD_MAIN_VIEW_LIST );
      break ;
    case GD_MAIN_VIEW_LIST :
      gd_main_view_set_view_type ( view ,GD_MAIN_VIEW_ICON );
      break ;
    default:
      gd_main_view_set_view_type ( view ,GD_MAIN_VIEW_ICON );
  }

  bjb_main_view_update_model (self->priv->parent);
  populate_main_toolbar (self);

  return TRUE;
}

/* Just makes toolbar draggable */
static gboolean
on_button_press (GtkWidget* widget,
                 GdkEventButton * event,
                 GdkWindowEdge edge)
{
  if (event->type == GDK_BUTTON_PRESS)
  {
    if (event->button == 1) {
      gtk_window_begin_move_drag (GTK_WINDOW (gtk_widget_get_toplevel (widget)),
                                  event->button,
                                  event->x_root,
                                  event->y_root,
                                  event->time);
    }
  }

  return FALSE;
}


static gboolean
update_selection_label (GdMainView *view, BjbMainToolbar *self)
{
  GList *selected;
  gint length;
  gchar *label;

  selected = gd_main_view_get_selection(view);
  length = g_list_length (selected);

  if (length == 0)
    label = g_strdup(_("Click on items to select them"));
  else
    label = g_strdup_printf (g_dngettext (GETTEXT_PACKAGE, "%d selected", "%d selected", length),length);

  gd_main_toolbar_set_labels (GD_MAIN_TOOLBAR (self), NULL, label);
  g_free (label);

  return TRUE;
}

static void
populate_bar_for_selection(BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;
  GtkStyleContext *context;

  priv->select = gd_main_toolbar_add_button (GD_MAIN_TOOLBAR (self),
                                             NULL,"Done", FALSE);
  context = gtk_widget_get_style_context (priv->select);
  gtk_style_context_add_class (context, "suggested-action");
  gtk_widget_reset_style (priv->select);

  g_signal_connect (priv->select, "clicked",
                    G_CALLBACK (on_selection_mode_clicked), self);

  priv->update_selection = g_signal_connect(priv->view,
                                            "view-selection-changed",
                                            G_CALLBACK(update_selection_label),
                                            self);

  update_selection_label(priv->view,self);
}

static void
update_label_for_standard (BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;
  gchar *needle = bjb_controller_get_needle (priv->controller);
  gchar *label ;
  
  if (needle && g_strcmp0 (needle, "") !=0)
    label = g_strdup_printf (_("Results for %s"), needle);

  else
    label = g_strdup (_("New and Recent"));

  gd_main_toolbar_set_labels (GD_MAIN_TOOLBAR (self), label, NULL);
  g_free (label);
}

static void
populate_bar_for_standard(BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;
  GtkWidget *bin = NULL;

  /* Label */
  update_label_for_standard (self);
  priv->search_handler = g_signal_connect_swapped (priv->controller,
         "search-changed", G_CALLBACK(update_label_for_standard), self);

  /* New Note button */
  priv->new = gd_main_toolbar_add_button(GD_MAIN_TOOLBAR (self),
                                         NULL,
                                         _("New"),
                                         TRUE);
  gtk_widget_set_size_request (priv->new, 70, -1);
  bin = gtk_bin_get_child (GTK_BIN (priv->new));

  if (bin)
  {
    gint y_padding = 0;
    gtk_misc_get_padding (GTK_MISC (bin), NULL, &y_padding);
    gtk_misc_set_padding (GTK_MISC (bin), 12, y_padding);
  }

  g_signal_connect(priv->new,"clicked",
                   G_CALLBACK(on_new_note_clicked),priv->parent);

  /* Go to selection mode */
  priv->select = gd_main_toolbar_add_button(GD_MAIN_TOOLBAR (self),
                                            "object-select-symbolic",
                                            NULL,
                                            FALSE);

  g_signal_connect (priv->select,"clicked",
                    G_CALLBACK(on_selection_mode_clicked),self);
}

static void
populate_bar_for_icon_view(BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;

  /* Switch to list */
  priv->list= gd_main_toolbar_add_button(GD_MAIN_TOOLBAR (self),
                                         "view-list-symbolic",
                                         NULL,
                                         FALSE);

  g_signal_connect (priv->list, "clicked",
                    G_CALLBACK(on_view_mode_clicked),self);

  populate_bar_for_standard(self);
}

static void
populate_bar_for_list_view(BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;

  /* Switch to icon view */
  priv->grid = gd_main_toolbar_add_button(GD_MAIN_TOOLBAR (self),
                                          "view-grid-symbolic",
                                          NULL,
                                          FALSE);

  g_signal_connect (priv->grid, "clicked",
                    G_CALLBACK(on_view_mode_clicked),self);

  populate_bar_for_standard(self);
}

static void
populate_bar_switch(BjbMainToolbar *self)
{
  switch (self->priv->type)
  {
    case BJB_TOOLBAR_SELECT:
      populate_bar_for_selection(self);
      break;

    case BJB_TOOLBAR_STD_ICON:
      populate_bar_for_icon_view(self);
      break;

    case BJB_TOOLBAR_STD_LIST:
      populate_bar_for_list_view(self);
      break;

    default:
      g_warning("Main Toolbar implementation is erroneous.\
                 Please fill in a bug report");
  }

  gtk_widget_show_all (GTK_WIDGET (self));
}

static void
populate_main_toolbar(BjbMainToolbar *self)
{
  BjbMainToolbarPrivate *priv = self->priv;
  BjbToolbarType to_be = BJB_TOOLBAR_0 ;

  if (gd_main_view_get_selection_mode(priv->view) == TRUE)
    to_be = BJB_TOOLBAR_SELECT;

  else if (gd_main_view_get_view_type(priv->view) == GD_MAIN_VIEW_ICON)
    to_be = BJB_TOOLBAR_STD_ICON;

  else if (gd_main_view_get_view_type(priv->view) == GD_MAIN_VIEW_LIST)
    to_be = BJB_TOOLBAR_STD_LIST;
  
  /* Simply clear then populate */
  if (to_be != priv->type)
  {
    priv->type = to_be;
    gd_main_toolbar_clear (GD_MAIN_TOOLBAR (self));

    if (priv->search_handler != 0)
    {
      g_signal_handler_disconnect (priv->controller, priv->search_handler);
      priv->search_handler = 0;
    }

    populate_bar_switch (self);
  }
}

static void
bjb_main_toolbar_constructed (GObject *obj)
{
  G_OBJECT_CLASS(bjb_main_toolbar_parent_class)->constructed(obj);
}

static void
bjb_main_toolbar_init (BjbMainToolbar *self)
{
  self->priv = BJB_MAIN_TOOLBAR_GET_PRIVATE(self);
  self->priv->type = BJB_TOOLBAR_0 ;
  g_signal_connect (self, "button-press-event", G_CALLBACK (on_button_press), NULL);
}

static void
bjb_main_toolbar_finalize (GObject *object)
{
  BjbMainToolbar *self = BJB_MAIN_TOOLBAR(object);
  BjbMainToolbarPrivate *priv = self->priv;

  if (priv->search_handler != 0)
  {
    g_signal_handler_disconnect (priv->controller, priv->search_handler);
    priv->search_handler = 0;
  }

  /* chain up */
  G_OBJECT_CLASS (bjb_main_toolbar_parent_class)->finalize (object);
}

static void
bjb_main_toolbar_get_property (GObject     *object,
                               guint       property_id,
                               GValue      *value,
                               GParamSpec  *pspec)
{
  BjbMainToolbar *self = BJB_MAIN_TOOLBAR (object);

  switch (property_id)
    {
    case PROP_VIEW:
      g_value_set_object (value, self->priv->view);
      break;
    case PROP_PARENT:
      g_value_set_object (value, self->priv->parent);
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
bjb_main_toolbar_set_property (GObject      *object,
                               guint         property_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
  BjbMainToolbar *self = BJB_MAIN_TOOLBAR (object);

  switch (property_id)
    {
    case PROP_VIEW:
      bjb_main_toolbar_set_view(self,g_value_get_object(value));
      break;
    case PROP_PARENT:
      self->priv->parent = g_value_get_object(value);
      break;
    case PROP_CONTROLLER:
      self->priv->controller = g_value_get_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
bjb_main_toolbar_class_init (BjbMainToolbarClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (BjbMainToolbarPrivate));

  object_class->get_property = bjb_main_toolbar_get_property;
  object_class->set_property = bjb_main_toolbar_set_property;
  object_class->constructed = bjb_main_toolbar_constructed;
  object_class->finalize = bjb_main_toolbar_finalize;

  properties[PROP_VIEW] = g_param_spec_object ("view",
                                               "View",
                                               "View",
                                               GD_TYPE_MAIN_VIEW,
                                               G_PARAM_READWRITE |
                                               G_PARAM_CONSTRUCT |
                                               G_PARAM_STATIC_STRINGS);

  properties[PROP_PARENT] = g_param_spec_object ("parent",
                                                 "Parent",
                                                 "Parent",
                                                 BJB_TYPE_MAIN_VIEW,
                                                 G_PARAM_READWRITE |
                                                 G_PARAM_CONSTRUCT |
                                                 G_PARAM_STATIC_STRINGS);

  properties[PROP_CONTROLLER] = g_param_spec_object ("controller",
                                "BjbController",
                                "Controller for notes model and search",
                                BJB_TYPE_CONTROLLER,
                                G_PARAM_READWRITE |
                                G_PARAM_CONSTRUCT |
                                G_PARAM_STATIC_STRINGS);

  g_object_class_install_properties (object_class, NUM_PROPERTIES, properties);
}

BjbMainToolbar *
bjb_main_toolbar_new (GdMainView *view,
                      BjbMainView *parent,
                      BjbController *controller)
{
  /* Since heriting GdMainToolbar, we populate bar _after_ construct */
  BjbMainToolbar *self;

  self = BJB_MAIN_TOOLBAR (g_object_new (BJB_TYPE_MAIN_TOOLBAR,
                                         "controller", controller,
                                         "parent", parent,
                                         "view", view,
                                         NULL));

  populate_main_toolbar(self);
  return self;
}

void
bjb_main_toolbar_set_view (BjbMainToolbar *self, GdMainView *view)
{
  self->priv->view = view ;
}
