/* This is -*- C -*- */
/* $Id: guppi-plug-in.c,v 1.7 2000/04/14 14:56:00 trow Exp $ */

/*
 * guppi-plug-in.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include "gmodule.h"
#include "guppi-splash.h"
#include "guppi-plug-in.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_plug_in_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plug_in_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plug_in_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_plug_in_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_plug_in_class_init(GuppiPlugInClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_plug_in_get_arg;
  object_class->set_arg = guppi_plug_in_set_arg;
  object_class->destroy = guppi_plug_in_destroy;
  object_class->finalize = guppi_plug_in_finalize;

}

static void
guppi_plug_in_init(GuppiPlugIn* obj)
{

}

GtkType
guppi_plug_in_get_type(void)
{
  static GtkType guppi_plug_in_type = 0;
  if (!guppi_plug_in_type) {
    static const GtkTypeInfo guppi_plug_in_info = {
      "GuppiPlugIn",
      sizeof(GuppiPlugIn),
      sizeof(GuppiPlugInClass),
      (GtkClassInitFunc)guppi_plug_in_class_init,
      (GtkObjectInitFunc)guppi_plug_in_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_plug_in_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_plug_in_info);
  }
  return guppi_plug_in_type;
}

GuppiPlugIn*
guppi_plug_in_new(void)
{
  return GUPPI_PLUG_IN(gtk_type_new(guppi_plug_in_get_type()));
}

/****************************************************************************/

static GHashTable* plug_in_table = NULL;

static gint
version_compare(const GuppiPlugIn* a, const GuppiPlugIn* b)
{
  if (a->major_version < b->major_version)
    return -1;
  else if (a->major_version > b->major_version)
    return +1;
  else if (a->minor_version < b->minor_version)
    return -1;
  else if (a->minor_version > b->minor_version)
    return +1;
  else if (a->micro_version < b->micro_version)
    return -1;
  else if (a->micro_version > b->micro_version)
    return +1;
  else
    return 0;
}

static void
register_plug_in(GuppiPlugIn* info)
{
  gpointer data;
  GuppiPlugIn* ai;
  GHashTable* type_table;
  gint rv;

  g_return_if_fail(info != NULL);

  g_return_if_fail(info->type != NULL);
  g_return_if_fail(info->external_name != NULL);
  g_return_if_fail(info->internal_name != NULL);

  if (plug_in_table == NULL) 
    plug_in_table = g_hash_table_new(g_str_hash, g_str_equal);

  type_table = (GHashTable*)g_hash_table_lookup(plug_in_table, info->type);
  if (type_table == NULL) {
    type_table = g_hash_table_new(g_str_hash, g_str_equal);
    g_hash_table_insert(plug_in_table, (gchar*)info->type, type_table);
  }

  data = g_hash_table_lookup(type_table, info->internal_name);
  if (data != NULL) {

    ai = GUPPI_PLUG_IN(data);
    g_assert(ai != NULL);
    rv = version_compare(ai, info);

    if (rv == -1) { 
      /* The already-loaded version is older */
      g_message("Replacing %s %d.%d.%d with %d.%d.%d",
                info->internal_name,
                info->major_version, info->minor_version, info->micro_version,
                ai->major_version, ai->minor_version, ai->micro_version);

      g_hash_table_remove(type_table, info->internal_name);
      
    } else {
      /* The already-loaded version is newer or the same */

      g_message("Skipping %s %d.%d.%d",
                info->internal_name,
                info->major_version, info->minor_version, info->micro_version);
      return;
    }
  }

  g_hash_table_insert(type_table, (gchar*)info->internal_name, info);
}

GuppiPlugIn*
guppi_plug_in_lookup(const gchar* type, const gchar* name)
{
  gpointer data;

  g_return_val_if_fail(type != NULL, NULL);
  g_return_val_if_fail(name != NULL, NULL);

  data = g_hash_table_lookup(plug_in_table, type);
  if (data == NULL)
    return NULL;
  data = g_hash_table_lookup((GHashTable*)data, name);
  if (data == NULL)
    return NULL;

  return GUPPI_PLUG_IN(data);
}

/*****************************************************************************/

GuppiPlugIn*
guppi_plug_in_load(const gchar* path)
{
  GModule* module;
  gboolean found_symbol;
  gpointer plug_in = NULL;
  GuppiPlugIn* pi;
  gchar msg_buffer[256];

  g_return_val_if_fail(path != NULL, NULL);

  module = g_module_open(path, G_MODULE_BIND_LAZY);
  if (module == NULL) {
    g_warning("Attempt to open plug-in %s failed: %s",
              path, g_module_error());
    return NULL;
  }

  found_symbol = g_module_symbol(module,"guppi_plug_in",&plug_in);
  if (! found_symbol) {
    g_warning("Can't find symbol guppi_plug_in in %s", path);
    g_module_close(module);
    return NULL;
  }

  if (plug_in == NULL) {
    g_warning("In %s, the symbol guppi_plug_in is NULL", path);
    g_module_close(module);
    return NULL;
  }

  pi = ((GuppiPlugIn* (*)(void))plug_in)();
  if (pi == NULL) {
    g_warning("In %s, guppi_plug_in() returned NULL", path);
    g_module_close(module);
    return NULL;
  }

  if (pi->magic_number != GUPPI_PLUG_IN_MAGIC_NUMBER) {
    g_warning("In %s, guppi_plug_in() returned a structure with a bad magic number.", path);
    g_module_close(module);
    return NULL;
  }

  pi->load_path = g_strdup(path);
  pi->reserved = module;

  register_plug_in(pi);
  g_snprintf(msg_buffer, 256, "Loaded plug-in: %s", pi->external_name);
  guppi_splash_message(msg_buffer);

  return pi;
}

void
guppi_plug_in_load_dir(const gchar* path_orig, gboolean recursive)
{
  DIR* dir;
  struct dirent* dirent;
  struct stat buf;
  gchar path[1024];
  gchar str[1024];
  gint i=0, j=0;

  /*
    If we are handed something like
    /home/foo/plug-ins:/bar/plug-ins:/usr/local/plug-ins
    we should do the right thing and load from all three.
  */
  while (path_orig[i]) {
    if (path_orig[i] == ':') {
      path[j] = '\0';
      guppi_plug_in_load_dir(path, recursive);
      j=0;
    } else {
      path[j] = path_orig[i];
      ++j;
    }
    ++i;
  }
  path[j]='\0';
  

  dir = opendir(path);
  if (dir == NULL) {
    /* If there is no such directory, just silently fail. */
    /* g_message("Couldn't load plug-ins from %s", path);*/
    return;
  }

  errno = 0;
  while ((dirent = readdir(dir)) != NULL) {

    if (recursive &&
        strcmp(dirent->d_name, ".") && 
        strcmp(dirent->d_name, "..")) {
      g_snprintf(str, 1023, "%s/%s", path, dirent->d_name);
      if (stat(str, &buf) < 0) {
        g_message("stat() on %s failed", str);
      } else if (S_ISDIR(buf.st_mode) && recursive) 
        guppi_plug_in_load_dir(str, recursive);
    }

    if (!strncmp(dirent->d_name, "lib", 3) &&
        !strcmp(g_extension_pointer(dirent->d_name), "so")) {
      g_snprintf(str, 1023, "%s/%s", path, dirent->d_name);
      guppi_plug_in_load(str);
    }

    errno = 0;
  }
  if (errno) {
    g_message("An error occured while reading %s", path);
  }

  closedir(dir);
}

void
guppi_plug_in_load_all(void)
{
  gchar* env;

  guppi_plug_in_load_dir("../../plug-ins", TRUE);

  env = getenv("GUPPI_PLUGIN_PATH");
  if (env)
    guppi_plug_in_load_dir(env, TRUE);

#ifdef GUPPI_PLUGIN_DIR
  guppi_plug_in_load_dir(GUPPI_PLUGIN_DIR, TRUE);
#endif
}

struct foreach_info {
  void (*func)(GuppiPlugIn*, gpointer);
  gpointer data;
};

static void
hfunc_inner(gpointer key, gpointer val, gpointer user_data)
{
  struct foreach_info* fi = (struct foreach_info*)user_data;
  g_return_if_fail(GUPPI_IS_PLUG_IN(val));
  (fi->func)(GUPPI_PLUG_IN(val), fi->data);
}

static void
hfunc_outer(gpointer key, gpointer val, gpointer user_data)
{
  g_hash_table_foreach((GHashTable*)val, hfunc_inner, user_data);
}

void
guppi_plug_in_foreach(void (*func)(GuppiPlugIn*, gpointer), gpointer data)
{
  struct foreach_info fi;
  fi.func = func;
  fi.data = data;
  g_hash_table_foreach(plug_in_table, hfunc_outer, &fi);
}

void
guppi_plug_in_foreach_of_type(const gchar* type,
			      void (*func)(GuppiPlugIn*, gpointer),
			      gpointer data)
{
  GHashTable* subtable;
  struct foreach_info fi;
  fi.func = func;
  fi.data = data;

  subtable = (GHashTable*)g_hash_table_lookup(plug_in_table, type);
  if (subtable != NULL)
    g_hash_table_foreach(subtable, hfunc_inner, &fi);
}


/* $Id: guppi-plug-in.c,v 1.7 2000/04/14 14:56:00 trow Exp $ */
