/* $Id: guppi-canvas-item.c,v 1.15 2000/04/13 19:45:20 trow Exp $ */

/*
 * guppi-canvas-item.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and 
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-rgb.h>
#include "guppi-canvas-item.h"

enum {
  RESCALED_X,
  RESCALED_Y,
  REBOUNDED_X,
  REBOUNDED_Y,
  RENEGOTIATE_X_START,
  RENEGOTIATE_X_DECLARE,
  RENEGOTIATE_Y_START,
  RENEGOTIATE_Y_DECLARE,
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_X1, ARG_Y1, ARG_X2, ARG_Y2
};

static GtkObjectClass* parent_class = NULL;
static guint gci_signals[LAST_SIGNAL] = { 0 };

typedef void (*GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE)(GtkObject*, double arg1, double arg2, double arg3, double arg4, gpointer user_data);
typedef void (*GuppiSignal_NONE__DOUBLE_DOUBLE)(GtkObject*, double arg1, double arg2, gpointer user_data);

/*
static void
guppi_marshal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE(GtkObject* object
,						GtkSignalFunc func,
						gpointer func_data,
						GtkArg* args)
{
  GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE rfunc;
  rfunc = (GuppiSignal_NONE__DOUBLE_DOUBLE_DOUBLE_DOUBLE)func;
  (*rfunc)(object, 
	   GTK_VALUE_DOUBLE(args[0]),
	   GTK_VALUE_DOUBLE(args[1]),
	   GTK_VALUE_DOUBLE(args[2]),
	   GTK_VALUE_DOUBLE(args[3]),
	   func_data);
}
*/

static void
guppi_marshal_NONE__DOUBLE_DOUBLE(GtkObject* object,
				  GtkSignalFunc func,
				  gpointer func_data,
				  GtkArg* args)
{
  GuppiSignal_NONE__DOUBLE_DOUBLE rfunc;
  rfunc = (GuppiSignal_NONE__DOUBLE_DOUBLE)func;
  (*rfunc)(object, 
	   GTK_VALUE_DOUBLE(args[0]),
	   GTK_VALUE_DOUBLE(args[1]),
	   func_data);
}

static void
guppi_canvas_item_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);

  switch (arg_id) {
  case ARG_X1:
    GTK_VALUE_DOUBLE(*arg) = gci->wx0;
    break;
  case ARG_Y1:
    GTK_VALUE_DOUBLE(*arg) = gci->wy0;
    break;
  case ARG_X2:
    GTK_VALUE_DOUBLE(*arg) = gci->wx1;
    break;
  case ARG_Y2:
    GTK_VALUE_DOUBLE(*arg) = gci->wy1;
    break;
  default:
    break;
  }
}

static void
guppi_canvas_item_destroy(GtkObject* obj)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);
  static void guppi_canvas_item_remove_state(GuppiCanvasItem*);
  
  guppi_canvas_item_remove_state(gci);
}

static void
guppi_canvas_item_finalize(GtkObject* obj)
{

}


static void
guppi_canvas_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(obj);
  gboolean did_something = FALSE;

  switch (arg_id) {
  case ARG_X1:
    gci->wx0 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_Y1:
    gci->wy0 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_X2:
    gci->wx1 = GTK_VALUE_DOUBLE(*arg);
    break;
  case ARG_Y2:
    gci->wy1 = GTK_VALUE_DOUBLE(*arg);
    break;
  default:
    did_something = TRUE;
    break;
  }

  if (did_something)
    guppi_canvas_item_request_update(gci);
    
}

static void
guppi_canvas_item_update(GnomeCanvasItem* item, double affine[6],
			 ArtSVP* clip_path, gint flags)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  // Convert from world to canvas coordinates, and remember.
  gnome_canvas_w2c(item->canvas, gci->wx0, gci->wy0, &gci->cx0, &gci->cy0);
  gnome_canvas_w2c(item->canvas, gci->wx1, gci->wy1, &gci->cx1, &gci->cy1);
  
  transform_set_pixel_bounds(gci->x_transform, gci->cx0, gci->cx1);
  /* By default, always reverse the y-coordinates to correspond to
     mathematical, rather than computer graphics, conventions. */
  transform_set_pixel_bounds(gci->y_transform, gci->cy1, gci->cy0);

  gnome_canvas_update_bbox(item, gci->wx0, gci->wy0, gci->wx1, gci->wy1);
}

static void
guppi_canvas_item_render(GnomeCanvasItem* item,
			 GnomeCanvasBuf* buf)
{
  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  if (buf->is_bg) {
    gnome_canvas_buf_ensure_buf(buf);
    buf->is_bg = FALSE;
  }
}

static double
guppi_canvas_item_point(GnomeCanvasItem* item,
			double x, double y, gint cx, gint cy,
			GnomeCanvasItem** actual_item)
{
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);

  double dist;

  if (gci->wx0 <= x && x <= gci->wx1 && gci->wy0 <= y && y <= gci->wy1) 
    dist = 0;
  else
    dist = MIN(MIN(fabs(x-gci->wx0), fabs(x-gci->wx1)),
	       MIN(fabs(y-gci->wy0), fabs(y-gci->wy1)));

  *actual_item = item;
  return dist;
}


static void
guppi_canvas_item_class_init(GuppiCanvasItemClass* klass)
{
  GtkObjectClass* object_class;
  GnomeCanvasItemClass* item_class;

  object_class = (GtkObjectClass*)klass;
  item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  parent_class = gtk_type_class(GNOME_TYPE_CANVAS_ITEM);

  gci_signals[RESCALED_X] =
    gtk_signal_new("rescaled_x",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rescaled_x),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RESCALED_Y] =
    gtk_signal_new("rescaled_y",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rescaled_y),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[REBOUNDED_X] =
    gtk_signal_new("rebounded_x",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rebounded_x),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[REBOUNDED_Y] =
    gtk_signal_new("rebounded_y",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, rebounded_y),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RENEGOTIATE_X_START] = 
    gtk_signal_new("renegotiate_x_start",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, renegotiate_x_start),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gci_signals[RENEGOTIATE_X_DECLARE] = 
    gtk_signal_new("renegotiate_x_declare",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, renegotiate_x_declare),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gci_signals[RENEGOTIATE_Y_START] = 
    gtk_signal_new("renegotiate_y_start",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, renegotiate_y_start),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gci_signals[RENEGOTIATE_Y_DECLARE] = 
    gtk_signal_new("renegotiate_y_declare",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiCanvasItemClass, renegotiate_y_declare),
		   guppi_marshal_NONE__DOUBLE_DOUBLE,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_DOUBLE, GTK_TYPE_DOUBLE);

  gtk_object_class_add_signals(object_class, gci_signals, LAST_SIGNAL);
    

  klass->rescaled_x = NULL;
  klass->rescaled_y = NULL;

  item_class->update = guppi_canvas_item_update;
  item_class->render = guppi_canvas_item_render;
  item_class->point = guppi_canvas_item_point;

  object_class->get_arg = guppi_canvas_item_get_arg;
  object_class->set_arg = guppi_canvas_item_set_arg;
  object_class->destroy = guppi_canvas_item_destroy;
  object_class->finalize = guppi_canvas_item_finalize;


  gtk_object_add_arg_type("GuppiCanvasItem::x1",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_X1);
  gtk_object_add_arg_type("GuppiCanvasItem::y1",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_Y1);
  gtk_object_add_arg_type("GuppiCanvasItem::x2",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_X2);
  gtk_object_add_arg_type("GuppiCanvasItem::y2",
			  GTK_TYPE_DOUBLE,
			  GTK_ARG_READWRITE,
			  ARG_Y2);
}

static void
guppi_canvas_item_init(GuppiCanvasItem* gci)
{
  gci->x_transform  = transform_new();
  gci->y_transform  = transform_new();

  gci->x0 = gci->y0 = 0;
  gci->x1 = gci->y1 = 1;

  gci->bb_x0 = gci->bb_y0 = 0;
  gci->bb_x1 = gci->bb_y1 = 1;

  gci->full_x0 = gci->full_y0 = 0;
  gci->full_x1 = gci->full_y1 = 1;

  transform_set_plot_bounds(gci->x_transform, gci->x0, gci->x1);
  transform_set_plot_bounds(gci->y_transform, gci->y0, gci->y1);

  gci->mag_factor = 1.0;

  /* Hard-wired position for testing */
  gci->wx0 = 0;
  gci->wx1 = 100;
  gci->wy0 = 0;
  gci->wy1 = 100;
}

GtkType
guppi_canvas_item_get_type(void)
{
  static GtkType gci_type = 0;

  if (!gci_type) {
    static const GtkTypeInfo gci_info = {
      "GuppiCanvasItem",
      sizeof(GuppiCanvasItem),
      sizeof(GuppiCanvasItemClass),
      (GtkClassInitFunc)guppi_canvas_item_class_init,
      (GtkObjectInitFunc)guppi_canvas_item_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    gci_type = gtk_type_unique(GNOME_TYPE_CANVAS_ITEM, &gci_info);
  }

  return gci_type;
}

/*****************************************************************************/

GuppiItemState*
guppi_canvas_item_state(GuppiCanvasItem* gci)
{
  g_return_val_if_fail(gci != NULL, NULL);

  return gci->state;
}

static void
guppi_canvas_item_remove_state(GuppiCanvasItem* gci)
{
  GuppiCanvasItemClass* klass;

  g_return_if_fail(gci != NULL);

  if (gci->state == NULL)
    return;

  klass = GUPPI_CANVAS_ITEM_CLASS(GTK_OBJECT(gci)->klass);
  if (klass->state_unset_hook)
    (klass->state_unset_hook)(gci);

  gtk_signal_disconnect_by_data(GTK_OBJECT(gci->state), gci);
  gtk_object_unref(GTK_OBJECT(gci->state));
  gci->state = NULL;
}

static void 
state_update_cb(GuppiItemState* gis, gpointer data)
{
  g_return_if_fail(data != NULL);
  guppi_canvas_item_request_update(GUPPI_CANVAS_ITEM(data));
}


void
guppi_canvas_item_set_state(GuppiCanvasItem* gci, GuppiItemState* state)
{
  GuppiCanvasItemClass* klass;
  g_return_if_fail(gci != NULL);

  if (state == gci->state)
    return;

  if (gci->state)
    guppi_canvas_item_remove_state(gci);

  gci->state = state;
  gtk_object_ref(GTK_OBJECT(state));

  if (gci->state) {

    klass = GUPPI_CANVAS_ITEM_CLASS(GTK_OBJECT(gci)->klass);
    if (klass->state_set_hook)
      (klass->state_set_hook)(gci, state);

    gtk_signal_connect(GTK_OBJECT(gci->state),
		       "changed",
		       GTK_SIGNAL_FUNC(state_update_cb),
		       gci);
  }

  guppi_canvas_item_request_update(gci);
}

/*****************************************************************************/

void
guppi_canvas_item_set_x_scale(GuppiCanvasItem* gci, double x0, double x1)
{
  g_return_if_fail(gci);

  if (gci->x0 == x0 && gci->x1 == x1)
    return;

  gci->x0 = x0;
  gci->x1 = x1;
  transform_set_plot_bounds(gci->x_transform, x0, x1);

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RESCALED_X], x0, x1);
  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_y_scale(GuppiCanvasItem* gci, double y0, double y1)
{
  g_return_if_fail(gci);

  if (gci->y0 == y0 && gci->y1 == y1)
    return;

  gci->y0 = y0;
  gci->y1 = y1;
  transform_set_plot_bounds(gci->y_transform, y0, y1);

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RESCALED_Y], y0, y1);
  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_scales(GuppiCanvasItem* gci,
			     double x0, double y0,
			     double x1, double y1)
{
  g_return_if_fail(gci);

  guppi_canvas_item_set_x_scale(gci, x0, x1);
  guppi_canvas_item_set_y_scale(gci, y0, y1);
}

/*****************************************************************************/

void
guppi_canvas_item_rescale_around_point(GuppiCanvasItem* gci,
				       double x, double y,
				       double factor)
{
  g_return_if_fail(gci != NULL);
  g_return_if_fail(factor > 0);

  guppi_canvas_item_set_scales(gci,
			       factor * (gci->x0 - x) + x,
			       factor * (gci->y0 - y) + y,
			       factor * (gci->x1 - x) + x,
			       factor * (gci->y1 - y) + y);
}

void
guppi_canvas_item_recenter_around_point(GuppiCanvasItem* gci,
					double x, double y)
{
  double xc, yc;
  g_return_if_fail(gci != NULL);

  xc = (gci->x0 + gci->x1) / 2;
  yc = (gci->y0 + gci->y1) / 2;

  guppi_canvas_item_set_scales(gci,
			       gci->x0 + x - xc, gci->y0 + y - yc,
			       gci->x1 + x - xc, gci->y1 + y - yc);
  
}

void
guppi_canvas_item_translate(GuppiCanvasItem* gci, double dx, double dy)
{
  g_return_if_fail(gci != NULL);
  guppi_canvas_item_set_scales(gci,
			       gci->x0 + dx, gci->y0 + dy,
			       gci->x1 + dx, gci->y1 + dy);
}

void
guppi_canvas_item_home_view(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);

  guppi_canvas_item_set_scales(gci,
			       gci->full_x0, gci->full_y0,
			       gci->full_x1, gci->full_y1);
}

/*****************************************************************************/

/* Plot coordinate negotiations */

static void
renegotiate_x_start(GuppiCanvasItem* gci)
{
  if (gci->reneg_x_started)
    return;
  
  gci->full_x0 = gci->bb_x0;
  gci->full_x1 = gci->bb_x1;
  gci->reneg_x_started = TRUE;
  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_X_START]);
  gci->reneg_x_started = FALSE;
  if (!gci->bb_passive)
    gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_X_DECLARE],
		    gci->full_x0, gci->full_x1);
}

static void
renegotiate_x_accept(GuppiCanvasItem* src, double x0, double x1,
		     GuppiCanvasItem* gci)
{
  double m0, m1;

  if (gci->bb_passive && gci->full_x0 == gci->bb_x0 && gci->full_x1 == gci->bb_x1) {
    m0 = x0;
    m1 = x1;
  } else {
    m0 = MIN(gci->full_x0, x0);
    m1 = MAX(gci->full_x1, x1);
  }

  if (gci->full_x0 == m0 && gci->full_x1 == m1)
    return;
  
  gci->full_x0 = m0;
  gci->full_x1 = m1;

  /* If anything changed, we re-declare */
  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_X_DECLARE],
		  gci->full_x0, gci->full_x1);

  guppi_canvas_item_request_update(gci);
}

static void
renegotiate_y_start(GuppiCanvasItem* gci)
{
  if (gci->reneg_y_started)
    return;
  gci->full_y0 = gci->bb_y0;
  gci->full_y1 = gci->bb_y1;
  gci->reneg_y_started = TRUE;
  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_Y_START]);
  gci->reneg_y_started = FALSE;
  if (!gci->bb_passive)
    gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_Y_DECLARE],
		    gci->full_y0, gci->full_y1);
}

static void
renegotiate_y_accept(GuppiCanvasItem* src, double y0, double y1,
		     GuppiCanvasItem* gci)
{
  double m0, m1;

  if (gci->bb_passive && gci->full_y0 == gci->bb_y0 && gci->full_y1 == gci->bb_y1) {
    m0 = y0;
    m1 = y1;
  } else {
    m0 = MIN(gci->full_y0, y0);
    m1 = MAX(gci->full_y1, y1);
  }

  if (gci->full_y0 == m0 && gci->full_y1 == m1)
    return;

  gci->full_y0 = m0;
  gci->full_y1 = m1;
  
  /* If anything changed, we re-declare */
  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[RENEGOTIATE_Y_DECLARE],
		  gci->full_y0, gci->full_y1);

  guppi_canvas_item_request_update(gci);
}

/*****************************************************************************/

void
guppi_canvas_item_set_x_world(GuppiCanvasItem* gci, double x0, double x1)
{
  g_return_if_fail(gci != NULL);

  if (gci->wx0 == x0 && gci->wx1 == x1)
    return;

  gci->wx0 = x0;
  gci->wx1 = x1;

  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_y_world(GuppiCanvasItem* gci, double y0, double y1)
{
  g_return_if_fail(gci != NULL);

  if (gci->wy0 == y0 && gci->wy1 == y1)
    return;

  gci->wy0 = y0;
  gci->wy1 = y1;

  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_world(GuppiCanvasItem* gci,
			    double x0, double y0, double x1, double y1)
{
  g_return_if_fail(gci != NULL);
  guppi_canvas_item_set_x_world(gci, x0, x1);
  guppi_canvas_item_set_y_world(gci, y0, y1);
}

/*****************************************************************************/

void
guppi_canvas_item_set_x_bounds(GuppiCanvasItem* gci, double x0, double x1)
{
  g_return_if_fail(gci != NULL);

  if (gci->bb_x0 == x0 && gci->bb_x1 == x1)
    return;

  gci->bb_x0 = x0;
  gci->bb_x1 = x1;

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[REBOUNDED_X], x0, x1);  
  renegotiate_x_start(gci);
  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_y_bounds(GuppiCanvasItem* gci, double y0, double y1)
{
  g_return_if_fail(gci != NULL);

  if (gci->bb_y0 == y0 && gci->bb_y1 == y1)
    return;

  gci->bb_y0 = y0;
  gci->bb_y1 = y1;

  gtk_signal_emit(GTK_OBJECT(gci), gci_signals[REBOUNDED_Y], y0, y1);  
  renegotiate_y_start(gci);
  guppi_canvas_item_request_update(gci);
}

void
guppi_canvas_item_set_bounds(GuppiCanvasItem* gci,
			     double x0, double y0,
			     double x1, double y1)
{
  g_return_if_fail(gci != NULL);

  guppi_canvas_item_set_x_bounds(gci, x0, x1);
  guppi_canvas_item_set_y_bounds(gci, y0, y1);
}

/*****************************************************************************/

double
guppi_canvas_item_x_magnification(const GuppiCanvasItem* gci)
{
  double mag;

  g_return_val_if_fail(gci != NULL, 0);

  mag = gci->mag_factor * fabs( (gci->full_x1-gci->full_x0)/(gci->x1-gci->x0) );

  return mag;
}

double
guppi_canvas_item_y_magnification(const GuppiCanvasItem* gci)
{
  double mag;

  g_return_val_if_fail(gci != NULL, 0);

  mag = gci->mag_factor * fabs( (gci->full_y1-gci->full_y0)/(gci->y1-gci->y0) );

  return mag;

}

double
guppi_canvas_item_magnification(const GuppiCanvasItem* gci)
{
  double mx, my;

  g_return_val_if_fail(gci != NULL, 0);

  mx = guppi_canvas_item_x_magnification(gci);
  my = guppi_canvas_item_y_magnification(gci);

  return (mx+my)/2;
}

/*****************************************************************************/

void
guppi_canvas_item_plot2pixel(const GuppiCanvasItem* gci,
			     double x, double y,
			     gint* cx, gint* cy)
{
  g_return_if_fail(gci != NULL);

  if (cx)
    *cx = transform_plot2pixel(gci->x_transform, x);
  if (cy)
    *cy = transform_plot2pixel(gci->y_transform, y);
}

void
guppi_canvas_item_pixel2plot(const GuppiCanvasItem* gci,
			     gint cx, gint cy,
			     double* x, double* y)
{
  g_return_if_fail(gci != NULL);
  
  if (x)
    *x = transform_pixel2plot(gci->x_transform, cx);
  if (y)
    *y = transform_pixel2plot(gci->y_transform, cy);
}

void
guppi_canvas_item_world2plot(const GuppiCanvasItem* gci,
			     double wx, double wy,
			     double* x, double* y)
{
  gint cx, cy;

  g_return_if_fail(gci != NULL);

  gnome_canvas_w2c(GNOME_CANVAS_ITEM(gci)->canvas, wx, wy, &cx, &cy);
  guppi_canvas_item_pixel2plot(gci, cx, cy, x, y);
}

void
guppi_canvas_item_plot2world(const GuppiCanvasItem* gci,
			     double x, double y,
			     double* wx, double* wy)
{
  gint cx, cy;

  g_return_if_fail(gci != NULL);

  guppi_canvas_item_plot2pixel(gci, x, y, &cx, &cy);
  gnome_canvas_c2w(GNOME_CANVAS_ITEM(gci)->canvas, cx, cy, wx, wy);
}

void
guppi_canvas_item_request_redraw(GuppiCanvasItem* gci,
				 double x0, double y0,
				 double x1, double y1)
{
  gint cx0,cy0,cx1,cy1,t;

  g_return_if_fail(gci != NULL);

  guppi_canvas_item_plot2pixel(gci, x0, y0, &cx0, &cy0);
  guppi_canvas_item_plot2pixel(gci, x1, y1, &cx1, &cy1);

  if (cx0 > cx1) { t = cx0; cx0 = cx1; cx1 = t; }
  if (cy0 > cy1) { t = cy0; cy0 = cy1; cy1 = t; }

  gnome_canvas_request_redraw(GNOME_CANVAS_ITEM(gci)->canvas,
			      cx0-1, cy0-1,
			      cx1+1, cy1+1);
}

void
guppi_canvas_item_request_total_redraw(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);

  gnome_canvas_request_redraw(GNOME_CANVAS_ITEM(gci)->canvas,
			      gci->cx0, gci->cy0,
			      gci->cx1, gci->cy1);
}

void
guppi_canvas_item_request_update(GuppiCanvasItem* gci)
{
  g_return_if_fail(gci != NULL);
  gnome_canvas_item_request_update(GNOME_CANVAS_ITEM(gci));
}

/***************************************************************************/

static void
rescale_x_cb(GuppiCanvasItem* a, double x0, double x1, GuppiCanvasItem* b)
{
  guppi_canvas_item_set_x_scale(b, x0, x1);
}

static void
rescale_y_cb(GuppiCanvasItem* a, double y0, double y1, GuppiCanvasItem* b)
{
  guppi_canvas_item_set_y_scale(b, y0, y1);
}

void
guppi_canvas_item_connect_x_scales(GuppiCanvasItem* a, GuppiCanvasItem* b)
{
  g_return_if_fail(a != NULL);
  g_return_if_fail(b != NULL);

  gtk_signal_connect(GTK_OBJECT(a),
		     "rescaled_x",
		     GTK_SIGNAL_FUNC(rescale_x_cb),
		     b);

  gtk_signal_connect_object(GTK_OBJECT(a),
			    "renegotiate_x_start",
			    GTK_SIGNAL_FUNC(renegotiate_x_start),
			    GTK_OBJECT(b));

  gtk_signal_connect(GTK_OBJECT(a),
		     "renegotiate_x_declare",
		     GTK_SIGNAL_FUNC(renegotiate_x_accept),
		     b);

  gtk_signal_connect(GTK_OBJECT(b),
		     "rescaled_x",
		     GTK_SIGNAL_FUNC(rescale_x_cb),
		     a);

  gtk_signal_connect_object(GTK_OBJECT(b),
			    "renegotiate_x_start",
			    GTK_SIGNAL_FUNC(renegotiate_x_start),
			    GTK_OBJECT(a));

  gtk_signal_connect(GTK_OBJECT(b),
		     "renegotiate_x_declare",
		     GTK_SIGNAL_FUNC(renegotiate_x_accept),
		     a);

}

void
guppi_canvas_item_connect_y_scales(GuppiCanvasItem* a, GuppiCanvasItem* b)
{
  g_return_if_fail(a != NULL);
  g_return_if_fail(b != NULL);

  gtk_signal_connect(GTK_OBJECT(a),
		     "rescaled_y",
		     GTK_SIGNAL_FUNC(rescale_y_cb),
		     b);

  gtk_signal_connect_object(GTK_OBJECT(a),
			    "renegotiate_y_start",
			    GTK_SIGNAL_FUNC(renegotiate_y_start),
			    GTK_OBJECT(b));

  gtk_signal_connect(GTK_OBJECT(a),
		     "renegotiate_y_declare",
		     GTK_SIGNAL_FUNC(renegotiate_y_accept),
		     b);

  gtk_signal_connect(GTK_OBJECT(b),
		     "rescaled_y",
		     GTK_SIGNAL_FUNC(rescale_y_cb),
		     a);

  gtk_signal_connect_object(GTK_OBJECT(b),
			    "renegotiate_y_start",
			    GTK_SIGNAL_FUNC(renegotiate_y_start),
			    GTK_OBJECT(a));

  gtk_signal_connect(GTK_OBJECT(b),
		     "renegotiate_y_declare",
		     GTK_SIGNAL_FUNC(renegotiate_y_accept),
		     a);

}

void
guppi_canvas_item_connect_scales(GuppiCanvasItem* a, GuppiCanvasItem* b)
{
  guppi_canvas_item_connect_x_scales(a, b);
  guppi_canvas_item_connect_y_scales(a, b);
}

/****************************************************************************/

void
guppi_canvas_item_disconnect(GuppiCanvasItem* a, GuppiCanvasItem* b)
{
  gtk_signal_disconnect_by_func(GTK_OBJECT(a),
				GTK_SIGNAL_FUNC(rescale_x_cb),
				b);

  gtk_signal_disconnect_by_func(GTK_OBJECT(b),
				GTK_SIGNAL_FUNC(rescale_x_cb),
				a);

  gtk_signal_disconnect_by_func(GTK_OBJECT(a),
				GTK_SIGNAL_FUNC(rescale_y_cb),
				b);

  gtk_signal_disconnect_by_func(GTK_OBJECT(b),
				GTK_SIGNAL_FUNC(rescale_y_cb),
				a);

}

/* $Id: guppi-canvas-item.c,v 1.15 2000/04/13 19:45:20 trow Exp $ */


