/* This is -*- C -*- */
/* $Id: guppi-data-tree-view.c,v 1.3 2000/04/13 19:45:19 trow Exp $ */

/*
 * guppi-data-tree-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-data-tree-view.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_data_tree_view_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_tree_view_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_tree_view_destroy(GtkObject* obj)
{
  GuppiDataTreeView* tv = GUPPI_DATA_TREE_VIEW(obj);

  if (parent_class->destroy)
    parent_class->destroy(obj);

  gtk_signal_disconnect_by_data(GTK_OBJECT(tv->tree), tv);
  gtk_object_unref(GTK_OBJECT(tv->tree));
  tv->tree = NULL;
}

static void
guppi_data_tree_view_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_data_tree_view_class_init(GuppiDataTreeViewClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_CTREE);

  object_class->get_arg = guppi_data_tree_view_get_arg;
  object_class->set_arg = guppi_data_tree_view_set_arg;
  object_class->destroy = guppi_data_tree_view_destroy;
  object_class->finalize = guppi_data_tree_view_finalize;

}

static void
guppi_data_tree_view_init(GuppiDataTreeView* obj)
{

}

GtkType
guppi_data_tree_view_get_type(void)
{
  static GtkType guppi_data_tree_view_type = 0;
  if (!guppi_data_tree_view_type) {
    static const GtkTypeInfo guppi_data_tree_view_info = {
      "GuppiDataTreeView",
      sizeof(GuppiDataTreeView),
      sizeof(GuppiDataTreeViewClass),
      (GtkClassInitFunc)guppi_data_tree_view_class_init,
      (GtkObjectInitFunc)guppi_data_tree_view_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_data_tree_view_type = gtk_type_unique(GTK_TYPE_CTREE, &guppi_data_tree_view_info);
  }
  return guppi_data_tree_view_type;
}

/****************************************************************************/

static GtkCTreeNode*
guppi_data_tree_view_build_node(GuppiDataTreeView* tv,
				GtkCTreeNode* parent,
				GuppiDataTreeNode* node)
{
  gchar* text[4];
  GtkCTreeNode* sib = NULL;
  GtkCTreeNode* us;
  gint mem;
  float mem_val;
  gchar buf[32];

  if (node->sibling_next) 
    sib = guppi_data_tree_view_build_node(tv, parent, node->sibling_next);

  text[0] = (gchar*)guppi_data_label(node->data);
  text[1] = (gchar*)guppi_data_get_type_name(node->data);

  guppi_data_get_size_info(node->data, buf, 32);
  text[2] = g_strdup(buf);

  mem = guppi_data_get_size_in_bytes(node->data);
  if (mem > 0) {
    if (mem < 1024)
      g_snprintf(buf, 32, "%db", mem);
    else {
      mem_val = mem / 1024.0;
      if (mem_val < 1024)
	g_snprintf(buf, 32, "%0.2fk", mem_val);
      else
	g_snprintf(buf, 32, "%0.2fM", mem_val/1024);
    }
  }

  text[3] = mem > 0 ? g_strdup(buf) : "?";
  

  us = gtk_ctree_insert_node(GTK_CTREE(tv), parent, sib,
			     text, 5,
			     NULL,
			     NULL,
			     NULL,
			     NULL,
			     node->child == NULL,
			     TRUE);

  if (node->child)
    guppi_data_tree_view_build_node(tv, us, node->child);

  return us;
}

static void
guppi_data_tree_view_build(GuppiDataTreeView* tv)
{
  GuppiDataTreeNode* root;

  g_return_if_fail(tv != NULL);

  gtk_clist_freeze(GTK_CLIST(tv));
  gtk_clist_clear(GTK_CLIST(tv));

  root = tv->tree->root;
  if (root)
    guppi_data_tree_view_build_node(tv, NULL, root);

  gtk_clist_thaw(GTK_CLIST(tv));
}

void
guppi_data_tree_view_construct(GuppiDataTreeView* tv,
			       GuppiDataTree* tree)
{
  gchar* titles[] = { N_("Data"),
		      N_("Type"),
		      N_("Size"),
		      N_("Mem") };
  gint column_count = 4;
  gint i;


  g_return_if_fail(tv != NULL);
  g_return_if_fail(tree != NULL);
  g_return_if_fail(tv->tree == NULL);

  tv->tree = tree;
  gtk_object_ref(GTK_OBJECT(tv->tree));

  gtk_signal_connect_object(GTK_OBJECT(tv->tree),
			    "changed",
			    GTK_SIGNAL_FUNC(guppi_data_tree_view_build),
			    GTK_OBJECT(tv));
		     

  gtk_ctree_construct(GTK_CTREE(tv), column_count, 0, titles);
  for (i=0; i<column_count; ++i)
    gtk_clist_set_column_auto_resize(GTK_CLIST(tv), i, TRUE);
  gtk_ctree_set_line_style(GTK_CTREE(tv),
			   GTK_CTREE_LINES_DOTTED);
  gtk_ctree_set_expander_style(GTK_CTREE(tv),
			       GTK_CTREE_EXPANDER_TRIANGLE);

  guppi_data_tree_view_build(tv);
}

GtkWidget*
guppi_data_tree_view_new(GuppiDataTree* tree)
{
  GuppiDataTreeView* tv;

  g_return_val_if_fail(tree != NULL, NULL);
  tv = GUPPI_DATA_TREE_VIEW(gtk_type_new(guppi_data_tree_view_get_type()));

  guppi_data_tree_view_construct(tv, tree);

  return GTK_WIDGET(tv);
}

/* For debugging */
void
guppi_data_tree_view_popup(GuppiDataTree* tree)
{
  GtkWidget* w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  GtkWidget* tv = guppi_data_tree_view_new(tree);
  gtk_container_add(GTK_CONTAINER(w), tv);
  gtk_widget_show_all(w);
}



/* $Id: guppi-data-tree-view.c,v 1.3 2000/04/13 19:45:19 trow Exp $ */
