/* This is -*- C -*- */
/* $Id: guppi-seq-boolean-core-impl.c,v 1.4 2000/04/13 19:45:19 trow Exp $ */

/*
 * guppi-seq-boolean-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <guppi-string.h>
#include "guppi-seq-boolean-core-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_boolean_core_impl_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_core_impl_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_core_impl_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_seq_boolean_core_impl_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

/*****************************************************************************/

#define ASIZE(core) (1+((core)->size >> 5))
#define BITVEC_GET(data, i) ((data)[(i)>>5] & (1 << ((i)&31)))
#define BITVEC_SET(data, i) ((data)[(i)>>5] |= (1 << ((i)&31)))
#define BITVEC_UNSET(data, i) ((data)[(i)>>5] &= ~(1 << ((i)&31)))
#define LAST_MASK(size) ((~0) >> (32-((size)&31)))

static gboolean
v_seq_boolean_get(const GuppiSeqBooleanImpl* impl, gint i)
{
  const GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  const guint32* data = (const guint32*)guppi_garray_data(core->garray);
  
  i -= core->index_basis;
  return BITVEC_GET(data, i);
}

static void
v_seq_boolean_set(GuppiSeqBooleanImpl* impl, gint i, gboolean x)
{
  const GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data = (guint32*)guppi_garray_data(core->garray);
  
  i -= core->index_basis;
  if (x)
    BITVEC_SET(data, i);
  else
    BITVEC_UNSET(data, i);
}

static void
v_seq_boolean_set_all(GuppiSeqBooleanImpl* impl, gboolean x)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data = (guint32*)guppi_garray_data(core->garray);
  gint i, N;

  N = ASIZE(core);
  if (x) {
    for (i=0; i<N-1; ++i)
      data[i] = ~0;
  } else {
    for (i=0; i<N-1; ++i)
      data[i] = 0;
  }

  data[N-1] = x ? LAST_MASK(core->size) : 0;
}

static void
v_seq_boolean_insert(GuppiSeqBooleanImpl* impl, gint i, gboolean x)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data;
  const guint32 top_bit = 1<<31;
  gint j,k,old_size;
  gboolean carry;
  guint32 r;

  if (guppi_garray_size(core->garray) <= ASIZE(core)) {
    old_size = guppi_garray_size(core->garray);
    guppi_garray_set_size(core->garray, MAX(32, 2*ASIZE(core)));
    data = (guint32*)guppi_garray_data(core->garray);
    for (j=old_size; j<guppi_garray_size(core->garray); ++j)
      data[j]=0;
  }

  data = (guint32*)guppi_garray_data(core->garray);

  i -= core->index_basis;

  j = i >> 5;
  k = i & 31;
  r = data[j];
  carry = r & top_bit;

  data[j++] = ((r & ~((1<<k)-1)) << 1) | (r & ((1<<k)-1));
  while (j < ASIZE(core)) {
    r = data[j] << 1;
    if (carry) r |= 1;
    carry = data[j] & top_bit;
    data[j] = r;
    ++j;
  }
  ++core->size;

  if (x)
    BITVEC_SET(data, i);
  else
    BITVEC_UNSET(data, i);
}

static void
v_seq_boolean_insert_many(GuppiSeqBooleanImpl* impl, gint i,
			  gboolean x, gsize N)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data;
  gint old_size, j, k, sz;
  gint step_big, step_small;
  guint32 mask, carry, r;

  /* grow the array, if necessary */
  sz = 1 + ((core->size + N) >> 5);
  if (guppi_garray_size(core->garray) <= sz) {
    old_size = guppi_garray_size(core->garray);
    guppi_garray_set_size(core->garray, MAX(32, 2*sz));
    data = (guint32*)guppi_garray_data(core->garray);
    for (j=old_size; j<guppi_garray_size(core->garray); ++j)
      data[j] = 0;
  }

  data = (guint32*)guppi_garray_data(core->garray);

  i -= core->index_basis;

  j  = i >> 5;
  k = i & 31;

  step_big = N >> 5;
  step_small = N & 31;

  r = data[j];
  carry = step_small ? r >> (32 - step_small) : 0;
  mask = k ? ( ((guint32)(~0)) >> (32 - k)) : 0;
  data[j++] = (r & mask) | ((r & ~mask) << step_small);
  while (j < ASIZE(core)) {
    r = (data[j] << step_small) | carry;
    carry = step_small ? data[j] >> (32 - step_small) : 0;
    data[j] = r;
    ++j;
  }
  core->size += N;
  

  /* initialize the inserted slots */
  /* this could be optimized to operate an guint32 at a time */
  if (x) {
    for (j=0; j<N; ++j)
      BITVEC_SET(data, i+j);
  } else {
    for (j=0; j<N; ++j)
      BITVEC_UNSET(data, i+j);
  }
}

static void
v_seq_boolean_bitwise_and(GuppiSeqBooleanImpl* impl,
			  const GuppiSeqBooleanImpl* other)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  GuppiSeqBooleanCoreImpl* other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(other);
  gint i;
  guint32* data;
  const guint32* other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32*)guppi_garray_data(core->garray);
    other_data = (const guint32*)guppi_garray_data(other_core->garray);
    for (i=0; i<ASIZE(core); ++i)
      data[i] &= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached();
}

static void
v_seq_boolean_bitwise_or(GuppiSeqBooleanImpl* impl,
			 const GuppiSeqBooleanImpl* other)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  GuppiSeqBooleanCoreImpl* other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(other);
  gint i;
  guint32* data;
  const guint32* other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32*)guppi_garray_data(core->garray);
    other_data = (const guint32*)guppi_garray_data(other_core->garray);
    for (i=0; i<ASIZE(core); ++i)
      data[i] |= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached();
}

static void
v_seq_boolean_bitwise_xor(GuppiSeqBooleanImpl* impl,
			 const GuppiSeqBooleanImpl* other)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  GuppiSeqBooleanCoreImpl* other_core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(other);
  gint i;
  guint32* data;
  const guint32* other_data;

  if (core->index_basis == other_core->index_basis &&
      core->size == other_core->size) {
    data = (guint32*)guppi_garray_data(core->garray);
    other_data = (const guint32*)guppi_garray_data(other_core->garray);
    for (i=0; i<ASIZE(core); ++i)
      data[i] ^= other_data[i];
    return;
  }

  /* We'll implement the hard case later. */
  g_assert_not_reached();
}

static void
v_seq_boolean_bitwise_not(GuppiSeqBooleanImpl* impl)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  gint i, N;
  guint32* data;

  data = (guint32*)guppi_garray_data(core->garray);
  N = ASIZE(core);
  for (i=0; i<N; ++i)
    data[i] = ~data[i];

  /* Make sure surplus bits are zero */
  data[N-1] &= LAST_MASK(core->size);
}

static gint
v_seq_boolean_next_true(const GuppiSeqBooleanImpl* impl, gint i)
{
  const GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  const guint32* data;
  gint j, k;
  guint32 q;

  if (core->size == 0)
    return 1;

  data = (const guint32*)guppi_garray_data(core->garray);
  i -= core->index_basis;

  if (i < 0) {
    if (data[0] & 1)
      return 0;
    i = 0;
  }

  j = i >> 5;
  k = i & 31;
  ++i;
  /* Check of our first guint32 */
  if (k != 31 && (q = (data[j] >> (k+1)))) {
    while (!(q&1)) {
      q = q>>1;
      ++i;
    }
    return i + core->index_basis;
  }
  ++j;
  
  while (j < ASIZE(core) && data[j] == 0)
    ++j;
  if (j >= ASIZE(core)) 
    return core->index_basis + core->size;
  q = data[j];
  i = j<<5;
  while (!(q&1)) {
    q = q>>1;
    ++i;
  }
  return i + core->index_basis;
}

static gsize
v_seq_boolean_true_count(const GuppiSeqBooleanImpl* impl)
{
  const GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data;
  static guint8* bitcount = NULL;
  gint q,i,count,N;
  
  /* Initialize our bit-count dictionary, which pre-tallies the number
     of on-bits in each integer 0..255 */
  if (bitcount == NULL) {
    bitcount = g_new(guint8, 256);
    for(i=0; i<256; ++i) {
      count=0;
      q=i;
      while (q) {
	if (q&1) ++count;
	q = q>>1;
      }
      bitcount[i] = count;
    }
  }

  data = (guint32*)guppi_garray_data(core->garray);
  N = ASIZE(core);

  /* Be paranoid and insure that the surplus bits are zero. */
  data[N-1] &= LAST_MASK(core->size);

  count=0;
  for(i=0; i<ASIZE(core); ++i) {
    q = data[i]; 
    count += bitcount[q & 0xff] + bitcount[(q>>8) & 0xff] +
      bitcount[(q>>16) & 0xff] + bitcount[(q>>24) & 0xff];
  }
  return (gsize)count;
}

/* This could be optimized a lot more. */
static gint
v_seq_boolean_range_query(const GuppiSeqScalar* seq,
			  GuppiSeqBooleanImpl* impl,
			  double min, double max,
			  gboolean do_and)
{
  gconstpointer ptr;
  gint stride;
  GuppiSeqBooleanCoreImpl* core;
  guint32* data;
  gint i, i0, i1;
  double x;
  gint hits=0;

  ptr = guppi_seq_scalar_raw(seq, &stride);
  if (ptr == NULL)
    return -1;

  core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guppi_seq_indices(GUPPI_SEQ(seq), &i0, &i1);

  data = (const guint32*)guppi_garray_data(core->garray);
  i0 = MAX(i0, core->index_basis);
  i1 = MIN(i1, core->index_basis - 1 + core->size);

  /* Adjust to point at first raw data element */
  ptr = (gconstpointer)(((guint8*)ptr) + i0*stride);
  
  if (do_and) {
    /* AND behavior */
    for (i=i0; i<=i1; ++i) {
      if (BITVEC_GET(data, i-i0)) {
	x = *(const double*)ptr;
	if (x < min || max < x) {
	  BITVEC_UNSET(data, i-i0);
	} else {
	  ++hits;
	}
      }
      ptr = (gconstpointer)(((guint8*)ptr) + stride);
    }
  } else {
    /* overwrite behavior */
    v_seq_boolean_set_all(impl, FALSE);
    for (i=i0; i<=i1; ++i) {
      x = *(const double*)ptr;
      if (min <= x && x <= max) {
	BITVEC_SET(data, i-i0);
	++hits;
      } 
      ptr = (gconstpointer)(((guint8*)ptr) + stride);
    }
  }
  return hits;
}

static void
v_seq_size_hint(GuppiSeqImpl* impl, gsize n)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  if (guppi_garray_size(core->garray) < 1 + (n >> 5))
    guppi_garray_set_size(core->garray, 1 + (n >> 5));
}

static void
v_seq_get_bounds(const GuppiSeqImpl* impl, gint* min, gint* max)
{
  const GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);

  if (min)
    *min = core->index_basis;

  if (max)
    *max = core->index_basis - 1 + core->size;
}

static void
v_seq_shift_indices(GuppiSeqImpl* impl, gint delta)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  core->index_basis += delta;
}

static gboolean
v_seq_validate(const GuppiSeqImpl* impl, const gchar* str,
	       gchar* error_msg, gsize errlen)
{
  return TRUE;
}

static void
v_seq_get(const GuppiSeqImpl* impl, gint i, gchar* buf, gsize len)
{
  gboolean x = v_seq_boolean_get(GUPPI_SEQ_BOOLEAN_IMPL(impl), i);

  guppi_boolean2string(x, buf, len);
}

static void
v_seq_set(GuppiSeqImpl* impl, gint i, const gchar* buf)
{
  gboolean x;
  g_return_if_fail(guppi_string_is_boolean(buf));
  x = guppi_string2boolean(buf);
  v_seq_boolean_set(GUPPI_SEQ_BOOLEAN_IMPL(impl), i, x);
}

static void
v_seq_insert(GuppiSeqImpl* impl, gint i, const gchar* buf)
{
  gboolean x;
  g_return_if_fail(guppi_string_is_boolean(buf));
  x = guppi_string2boolean(buf);
  v_seq_boolean_insert(GUPPI_SEQ_BOOLEAN_IMPL(impl), i, x);
}


static void
v_seq_delete_many(GuppiSeqImpl* impl, gint i, gsize N)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  guint32* data = (guint32*)guppi_garray_data(core->garray);
  guint32 mask, r, s;
  gint j,k, step_small, step_big;
  
  N = MIN(N, core->size-i);
  if (N == 0)
    return;

  j = i >> 5;
  k = i & 31;
  step_big = N >> 5;
  step_small = N & 31;

  mask = k ? (~0u) >> (32 - k) : 0;

  r = data[j] & mask;
  s = 0;
  if (j+step_big < ASIZE(core))
    s = (data[j+step_big] >> step_small) & ~mask;
  data[j] = r | s;
  if (step_small && j+step_big+1 < ASIZE(core))
    data[j] |= data[j+step_big+1] << (32 - step_small);
  ++j;

  while (j+step_big < ASIZE(core)) {
    if (step_small) {
      data[j] = data[j+step_big] >> step_small;
      if (j+step_big+1 < ASIZE(core))
	data[j] |= data[j+step_big+1] << (32 - step_small);
    } else {
      data[j] = data[j+step_big];
    }
    ++j;
  }
  core->size -= N;
}

static GuppiDataImpl*
v_data_copy(GuppiDataImpl* impl)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  GuppiSeqBooleanCoreImpl* copy;
  gint i;
  guint32* data;
  guint32* copy_data;

  copy = GUPPI_SEQ_BOOLEAN_CORE_IMPL(gtk_type_new(GUPPI_TYPE_SEQ_BOOLEAN_CORE_IMPL));

  copy->index_basis = core->index_basis;
  copy->size = core->size;
  copy->garray = guppi_garray_new(sizeof(double));
  guppi_garray_set_size(copy->garray, ASIZE(copy));

  data = (guint32*)guppi_garray_data(core->garray);
  copy_data = (guint32*)guppi_garray_data(copy->garray);
  for (i=0; i<ASIZE(core); ++i)
    copy_data[i] = data[i];

  return GUPPI_DATA_IMPL(copy);

}

static gint
v_data_size_in_bytes(GuppiDataImpl* impl)
{
  GuppiSeqBooleanCoreImpl* core = GUPPI_SEQ_BOOLEAN_CORE_IMPL(impl);
  return guppi_garray_size(core->garray) * sizeof(guint32) + 
    sizeof(GuppiSeqBooleanCoreImpl);
}

/*****************************************************************************/

static void
guppi_seq_boolean_core_impl_class_init(GuppiSeqBooleanCoreImplClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GuppiDataImplClass* data_class = GUPPI_DATA_IMPL_CLASS(klass);
  GuppiSeqImplClass* seq_class = GUPPI_SEQ_IMPL_CLASS(klass);
  GuppiSeqBooleanImplClass* seq_boolean_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_SEQ_BOOLEAN_IMPL);

  object_class->get_arg = guppi_seq_boolean_core_impl_get_arg;
  object_class->set_arg = guppi_seq_boolean_core_impl_set_arg;
  object_class->destroy = guppi_seq_boolean_core_impl_destroy;
  object_class->finalize = guppi_seq_boolean_core_impl_finalize;

  data_class->impl_name = _("Core Boolean Sequence");

  seq_boolean_class->get = v_seq_boolean_get;
  seq_boolean_class->set = v_seq_boolean_set;
  seq_boolean_class->set_all = v_seq_boolean_set_all;
  seq_boolean_class->insert = v_seq_boolean_insert;
  seq_boolean_class->insert_many = v_seq_boolean_insert_many;
  seq_boolean_class->bitwise_and = v_seq_boolean_bitwise_and;
  seq_boolean_class->bitwise_or = v_seq_boolean_bitwise_or;
  seq_boolean_class->bitwise_xor = v_seq_boolean_bitwise_xor;
  seq_boolean_class->bitwise_not = v_seq_boolean_bitwise_not;
  seq_boolean_class->next_true = v_seq_boolean_next_true;
  seq_boolean_class->true_count = v_seq_boolean_true_count;
  seq_boolean_class->range_query = v_seq_boolean_range_query;

  seq_class->size_hint = v_seq_size_hint;
  seq_class->get_bounds = v_seq_get_bounds;
  seq_class->shift_indices = v_seq_shift_indices;
  seq_class->validate = v_seq_validate;
  seq_class->get = v_seq_get;
  seq_class->set = v_seq_set;
  seq_class->insert = v_seq_insert;
  seq_class->delete_many = v_seq_delete_many;

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_size_in_bytes;
}

static void
guppi_seq_boolean_core_impl_init(GuppiSeqBooleanCoreImpl* obj)
{
  obj->index_basis = 0;
  obj->size = 0;
  obj->garray = guppi_garray_new(sizeof(guint32));
}

GtkType
guppi_seq_boolean_core_impl_get_type(void)
{
  static GtkType guppi_seq_boolean_core_impl_type = 0;
  if (!guppi_seq_boolean_core_impl_type) {
    static const GtkTypeInfo guppi_seq_boolean_core_impl_info = {
      "GuppiSeqBooleanCoreImpl",
      sizeof(GuppiSeqBooleanCoreImpl),
      sizeof(GuppiSeqBooleanCoreImplClass),
      (GtkClassInitFunc)guppi_seq_boolean_core_impl_class_init,
      (GtkObjectInitFunc)guppi_seq_boolean_core_impl_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_boolean_core_impl_type = gtk_type_unique(GUPPI_TYPE_SEQ_BOOLEAN_IMPL, &guppi_seq_boolean_core_impl_info);
  }
  return guppi_seq_boolean_core_impl_type;
}

GtkObject*
guppi_seq_boolean_core_impl_new(void)
{
  return GTK_OBJECT(gtk_type_new(guppi_seq_boolean_core_impl_get_type()));
}



/* $Id: guppi-seq-boolean-core-impl.c,v 1.4 2000/04/13 19:45:19 trow Exp $ */
