/* This is -*- C -*- */
/* $Id: guppi-array.c,v 1.2 2000/04/13 19:45:19 trow Exp $ */

/*
 * guppi-array.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-array.h"
#include "guppi-array-impl.h"
#include "guppi-array-core-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_array_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_array_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_array_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_array_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_array_class_init(GuppiArrayClass* klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
  GuppiDataClass* data_class = GUPPI_DATA_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  data_class->default_impl = GUPPI_TYPE_ARRAY_CORE_IMPL;
  data_class->type_name = _("Array");

  object_class->get_arg = guppi_array_get_arg;
  object_class->set_arg = guppi_array_set_arg;
  object_class->destroy = guppi_array_destroy;
  object_class->finalize = guppi_array_finalize;

}

static void
guppi_array_init(GuppiArray* obj)
{

}

GtkType
guppi_array_get_type(void)
{
  static GtkType guppi_array_type = 0;
  if (!guppi_array_type) {
    static const GtkTypeInfo guppi_array_info = {
      "GuppiArray",
      sizeof(GuppiArray),
      sizeof(GuppiArrayClass),
      (GtkClassInitFunc)guppi_array_class_init,
      (GtkObjectInitFunc)guppi_array_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_array_type = gtk_type_unique(GUPPI_TYPE_DATA, &guppi_array_info);
  }
  return guppi_array_type;
}

/*****************************************************************************/

void
guppi_array_construct(GuppiArray* array, gint dims,
			   const gint* sizes, const gint* mins)
{
  GuppiArrayImpl* impl;
  GuppiArrayImplClass* impl_class;

  g_return_if_fail(array != NULL);
  g_return_if_fail(dims > 0);
  g_return_if_fail(sizes != NULL);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));
  impl_class = GUPPI_ARRAY_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->construct)
    (impl_class->construct)(impl, dims, sizes, mins);
}

GuppiData*
guppi_array_new(gint dims, const gint* sizes, const gint* mins)
{
  return guppi_array_with_impl_new(0, dims, sizes, mins);
}

GuppiData*
guppi_array_with_impl_new(GtkType type, gint dims,
			       const gint* sizes, const gint* mins)
{
  GuppiData* gd;

  g_return_val_if_fail(dims > 0, NULL);
  g_return_val_if_fail(sizes != NULL, NULL);

  gd = GUPPI_DATA(gtk_type_new(GUPPI_TYPE_ARRAY));
  
  if (type)
    gd->requested_impl = type;

  guppi_array_construct(GUPPI_ARRAY(gd), dims, sizes, mins);

  return gd;
}

/*****************************************************************************/

gint
guppi_array_dimensions(const GuppiArray* array)
{
  GuppiArrayImpl* impl;

  g_return_val_if_fail(array != NULL, -1);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));
  g_assert(impl->dims);

  return *(impl->dims);
}

gint
guppi_array_min_index(const GuppiArray* array, gint i)
{
  gint min = 0;
  guppi_array_index_bounds(array, i, &min, NULL);
  return min;
}

gint
guppi_array_max_index(const GuppiArray* array, gint i)
{
  gint max = 0;
  guppi_array_index_bounds(array, i, NULL, &max);
  return max;
}

void
guppi_array_index_bounds(const GuppiArray* array, gint i,
			      gint* a, gint* b)
{
  GuppiArrayImpl* impl;
  gint min;

  if (a) *a = 0;
  if (b) *b = -1;

  g_return_if_fail(array != NULL);
  g_return_if_fail(i>=0);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));

  g_assert(impl->dims != NULL);
  g_assert(impl->index_sizes != NULL);
  g_assert(*impl->index_sizes != NULL);

  g_return_if_fail(i < *(impl->dims));

  min = impl->index_min && *impl->index_min ? (*impl->index_min)[i] : 0; 
  if (a) *a = min;
  if (b) *b = min + (*impl->index_sizes)[i] - 1;
}

gboolean
guppi_array_valid_index(const GuppiArray* array, gint i, gint j)
{
  gint min, max;
  guppi_array_index_bounds(array, i, &min, &max);
  return min <= j && j <= max;
}

gboolean
guppi_array_validv(const GuppiArray* array, const gint* v)
{
  GuppiArrayImpl* impl;
  gint i, min;

  g_return_val_if_fail(array != NULL, FALSE);
  g_return_val_if_fail(v != NULL, FALSE);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));
  g_assert(impl->dims != NULL);
  g_assert(impl->index_sizes != NULL);
  g_assert(*impl->index_sizes != NULL);

  for (i=0; i < *impl->dims; ++i) {
    min = impl->index_min && *impl->index_min ? (*impl->index_min)[i] : 0; 
    if (v[i] < min || v[i] >= min + (*impl->index_sizes)[i])
      return FALSE;
  }
  
  return TRUE;
}

/*****************************************************************************/

static gint*
va2ivec(int size, va_list* va)
{
  gint* v;
  gint i;

  g_return_val_if_fail(size > 0, NULL);
  g_return_val_if_fail(va != NULL, NULL);

  v = g_new(int,size);
  for(i=0; i<size; ++i)
    v[i] = va_arg(*va, gint);

 return v;
}

double
guppi_array_get(const GuppiArray* array, ...)
{
  va_list va;
  gint* v;
  double x;

  g_return_val_if_fail(GUPPI_ARRAY(array) != NULL, 0);

  va_start(va, array);
  v = va2ivec(guppi_array_dimensions(array), &va);
  g_return_val_if_fail(v != NULL, 0);
  va_end(va);

  x = guppi_array_getv(array, v);
  g_free(v);

  return x;
}

double
guppi_array_getv(const GuppiArray* array, const gint* v)
{
  GuppiArrayImpl* impl;
  GuppiArrayImplClass* impl_class;
  gint i, j;

  g_return_val_if_fail(GUPPI_ARRAY(array), 0);
  g_return_val_if_fail(v != NULL, 0);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));

  impl_class = GUPPI_ARRAY_IMPL_CLASS(GTK_OBJECT(impl)->klass);
  if (impl_class->get_cb)
    (impl_class->get_cb)(impl, v);

  if (! guppi_array_validv(array, v)) {
    g_warning("guppi_array_getv(): index out of bounds");
    return 0;
  }

   j = 0;
  for (i=0; i<(*impl->dims); ++i)
    j += (*impl->strides)[i] *
      (v[i] - (impl->index_min && *impl->index_min ? (*impl->index_min)[i]:0));

  g_assert(impl->data && *impl->data);

  return (*impl->data)[j];
}

void
guppi_array_set(GuppiArray* array, double x, ...)
{
  va_list va;
  gint* v;

  g_return_if_fail(GUPPI_ARRAY(array) != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(array)));

  va_start(va, x);
  v = va2ivec(guppi_array_dimensions(array), &va);
  g_return_if_fail(v != NULL);
  va_end(va);

  guppi_array_setv(array, x, v);
  g_free(v);
}

void
guppi_array_setv(GuppiArray* array, double x, const gint* v)
{
  GuppiArrayImpl* impl;
  GuppiArrayImplClass* impl_class;
  gint i, j;

  g_return_if_fail(GUPPI_ARRAY(array));
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(array)));
  g_return_if_fail(v != NULL);

  impl = GUPPI_ARRAY_IMPL(guppi_data_impl(GUPPI_DATA(array)));

  if (! guppi_array_validv(array, v)) {
    g_warning("guppi_array_setv(): index out of bounds");
    return;
  }

   j = 0;
  for (i=0; i<(*impl->dims); ++i)
    j += (*impl->strides)[i] *
      (v[i] - (impl->index_min && *impl->index_min ? (*impl->index_min)[i]:0));

  g_assert(impl->data && *impl->data);

  (*impl->data)[j] = x;

  impl_class = GUPPI_ARRAY_IMPL_CLASS(GTK_OBJECT(impl)->klass);
  if (impl_class->set_cb)
    (impl_class->set_cb)(impl, x, v);

}

/* $Id: guppi-array.c,v 1.2 2000/04/13 19:45:19 trow Exp $ */

