/* This is -*- C -*- */
/* $Id: guppi-seq-scalar-core-impl.c,v 1.3 2000/03/04 18:50:14 trow Exp $ */

/*
 * guppi-seq-scalar-core-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-seq-scalar-core-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_scalar_core_impl_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_core_impl_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_core_impl_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_seq_scalar_core_impl_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

/****************************************************************************/

static void
v_seq_scalar_range(const GuppiSeqScalarImpl* impl, double* min, double* max)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  double* data;
  double x, m0, m1;
  gint i;

  if (core->cached_minmax) {
    if (min) *min = core->min;
    if (max) *max = core->max;
  }

  data = (double*)guppi_garray_data(core->garray);

  m0 = m1 = *data;
  for (i=1; i<core->size; ++i) {
    x = data[i];
    if (x < m0) m0 = x;
    if (x > m1) m1 = x;
  }

  core->min = m0;
  core->max = m1;
  core->cached_minmax = TRUE;

  if (min) *min = m0;
  if (max) *max = m1;
}

static double
v_seq_scalar_get(const GuppiSeqScalarImpl* impl, gint i)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  double* data;

  data = (double*)guppi_garray_data(core->garray);
  i -= core->index_basis;
  return data[i];
}

static void
v_seq_scalar_set(GuppiSeqScalarImpl* impl, gint i, double x)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  double* data;

  data = (double*)guppi_garray_data(core->garray);
  i -= core->index_basis;

  if (core->cached_minmax) {
    if (data[i] == core->min) {
      if (x < core->min)
	core->min = x;
      else
	core->cached_minmax = FALSE;
    }
    if (data[i] == core->max) {
      if (x > core->max)
	core->max = x;
      else
	core->cached_minmax = FALSE;
    }
  }
  data[i] = x;
}

static void
v_seq_scalar_insert(GuppiSeqScalarImpl* impl, gint i, double x)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  double* data;
  gint j;

  i -= core->index_basis;

  if (guppi_garray_size(core->garray) <= core->size)
    guppi_garray_set_size(core->garray, MAX(20,2*core->size));

  if (core->cached_minmax) {
    if (x < core->min) core->min = x;
    if (x > core->max) core->max = x;
  }

  data = (double*)guppi_garray_data(core->garray);  
  for(j=core->size-1; i<=j; --j)
    data[j+1] = data[j];
  data[i] = x;
  ++core->size;
}

static gconstpointer
v_seq_scalar_raw(const GuppiSeqScalarImpl* impl, gint* stride)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  gint8* p;

  p = (gint8*)guppi_garray_data(core->garray);
  p -= sizeof(double) * core->index_basis;
  *stride = sizeof(double);

  return (gconstpointer)p;
}


static void
v_seq_size_hint(GuppiSeqImpl* impl, gsize n)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);

  if (guppi_garray_size(core->garray) < n)
    guppi_garray_set_size(core->garray, n);
}

static void
v_seq_get_bounds(const GuppiSeqImpl* impl, gint* min, gint* max)
{
  const GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);

  if (min)
    *min = core->index_basis;

  if (max) 
    *max = core->index_basis - 1 + core->size;

}

static void
v_seq_shift_indices(GuppiSeqImpl* impl, gint delta)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  core->index_basis += delta;
}

static gboolean
v_seq_validate(const GuppiSeqImpl* impl, const gchar* str, 
	       gchar* error_msg, gsize errlen)
{
  return TRUE;
}

static void
v_seq_get(const GuppiSeqImpl* impl, gint i, gchar* buf, gsize len)
{
  double x;

  x = v_seq_scalar_get(GUPPI_SEQ_SCALAR_IMPL(impl), i);
  g_snprintf(buf, len, "%f", x);
}

static void
v_seq_set(GuppiSeqImpl* impl, gint i, const gchar* buf)
{
  v_seq_scalar_set(GUPPI_SEQ_SCALAR_IMPL(impl), i, atof(buf));
}

static void
v_seq_insert(GuppiSeqImpl* impl, gint i, const gchar* buf)
{
  v_seq_scalar_insert(GUPPI_SEQ_SCALAR_IMPL(impl), i, atof(buf));
}

static void
v_seq_delete_many(GuppiSeqImpl* impl, gint i, gsize N)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  double* data;
  gint j;

  /* This isn't strictly necessary --- we can preserve the cached values
     in many cases.  I'll put that in later... */
  core->cached_minmax = FALSE;

  data = (double*)guppi_garray_data(core->garray);
  i -= core->index_basis;

  for (j=i; j+N<core->size; ++j)
    data[j] = data[j+N];

  core->size -= N;
}

static GuppiDataImpl*
v_data_copy(GuppiDataImpl* impl)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  GuppiSeqScalarCoreImpl* copy;
  gint i;
  double* data;
  double* copy_data;

  copy = GUPPI_SEQ_SCALAR_CORE_IMPL(gtk_type_new(GUPPI_TYPE_SEQ_SCALAR_CORE_IMPL));

  copy->index_basis = core->index_basis;
  copy->size = core->size;
  copy->garray = guppi_garray_new(sizeof(double));
  guppi_garray_set_size(copy->garray, copy->size);

  data = (double*)guppi_garray_data(core->garray);
  copy_data = (double*)guppi_garray_data(copy->garray);
  for (i=0; i<core->size; ++i)
    copy_data[i] = data[i];

    return GUPPI_DATA_IMPL(copy);
}

static gint
v_data_size_in_bytes(GuppiDataImpl* impl)
{
  GuppiSeqScalarCoreImpl* core = GUPPI_SEQ_SCALAR_CORE_IMPL(impl);
  return guppi_garray_size(core->garray) * sizeof(double) +
    sizeof(GuppiSeqScalarCoreImpl);
}
		  

/****************************************************************************/

static void
guppi_seq_scalar_core_impl_class_init(GuppiSeqScalarCoreImplClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GuppiDataImplClass* data_class = GUPPI_DATA_IMPL_CLASS(klass);
  GuppiSeqImplClass* seq_class = GUPPI_SEQ_IMPL_CLASS(klass);
  GuppiSeqScalarImplClass* seq_scalar_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_SEQ_SCALAR_IMPL);

  object_class->get_arg = guppi_seq_scalar_core_impl_get_arg;
  object_class->set_arg = guppi_seq_scalar_core_impl_set_arg;
  object_class->destroy = guppi_seq_scalar_core_impl_destroy;
  object_class->finalize = guppi_seq_scalar_core_impl_finalize;

  data_class->impl_name = _("Core Scalar Sequence");

  seq_scalar_class->range = v_seq_scalar_range;
  seq_scalar_class->get = v_seq_scalar_get;
  seq_scalar_class->set = v_seq_scalar_set;
  seq_scalar_class->insert = v_seq_scalar_insert;
  seq_scalar_class->raw_access = v_seq_scalar_raw;

  seq_class->size_hint = v_seq_size_hint;
  seq_class->get_bounds = v_seq_get_bounds;
  seq_class->shift_indices = v_seq_shift_indices;

  seq_class->validate = v_seq_validate;
  seq_class->get = v_seq_get; 
  seq_class->set = v_seq_set;
  seq_class->insert = v_seq_insert;
  seq_class->delete_many = v_seq_delete_many;

  data_class->copy = v_data_copy;
  data_class->get_size_in_bytes = v_data_size_in_bytes;

}

static void
guppi_seq_scalar_core_impl_init(GuppiSeqScalarCoreImpl* obj)
{
  obj->index_basis = 0;
  obj->size = 0;
  obj->garray = guppi_garray_new(sizeof(double));
}

GtkType
guppi_seq_scalar_core_impl_get_type(void)
{
  static GtkType guppi_seq_scalar_core_impl_type = 0;
  if (!guppi_seq_scalar_core_impl_type) {
    static const GtkTypeInfo guppi_seq_scalar_core_impl_info = {
      "GuppiSeqScalarCoreImpl",
      sizeof(GuppiSeqScalarCoreImpl),
      sizeof(GuppiSeqScalarCoreImplClass),
      (GtkClassInitFunc)guppi_seq_scalar_core_impl_class_init,
      (GtkObjectInitFunc)guppi_seq_scalar_core_impl_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_scalar_core_impl_type = gtk_type_unique(GUPPI_TYPE_SEQ_SCALAR_IMPL, &guppi_seq_scalar_core_impl_info);
  }
  return guppi_seq_scalar_core_impl_type;
}




/* $Id: guppi-seq-scalar-core-impl.c,v 1.3 2000/03/04 18:50:14 trow Exp $ */
