/* $Id: guppi-scatter-state.c,v 1.10 2000/03/08 17:36:41 trow Exp $ */

/*
 * guppi-scatter-state.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include <gtk/gtkmarshal.h>
#include "guppi-scatter-state.h"

enum {
  CHANGED_STYLE,
  LAST_SIGNAL
};

static GtkObjectClass* parent_class = NULL;
static guint gss_signals[LAST_SIGNAL] = { 0 };

static void
guppi_scatter_state_destroy(GtkObject* obj)
{
  GuppiScatterState* gss = GUPPI_SCATTER_STATE(obj);
  gint i;

  if (gss->x_data)
    gtk_object_unref(GTK_OBJECT(gss->x_data));
  if (gss->y_data)
    gtk_object_unref(GTK_OBJECT(gss->y_data));

  for(i=0; i<GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {
    if (gss->style[i] != NULL)
      gtk_object_unref(GTK_OBJECT(gss->style[i]));
    if (gss->mask[i] != NULL)
      gtk_object_unref(GTK_OBJECT(gss->mask[i]));
  }

 if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_scatter_state_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

/****************************************************************************/

/* Some virtual functions */

static gboolean
x_data_set(GuppiItemState* state, GuppiData* data)
{
  GuppiScatterState* ss;
  GuppiSeqScalar* sd;

  ss = GUPPI_SCATTER_STATE(state);
  sd = GUPPI_SEQ_SCALAR0(data);

  g_return_val_if_fail(ss != NULL, FALSE);

  if (ss->x_data == sd)
    return FALSE;

  if (ss->x_data)
    gtk_object_unref(GTK_OBJECT(ss->x_data));

  ss->x_data = sd;

  if (sd)
    gtk_object_ref(GTK_OBJECT(sd));

  return TRUE;
}

static GuppiData*
x_data_get(GuppiItemState* state)
{
  GuppiScatterState* ss;

  ss = GUPPI_SCATTER_STATE(state);
  g_return_val_if_fail(ss != NULL, FALSE);
  return GUPPI_DATA0(ss->x_data);
}

static gboolean
y_data_set(GuppiItemState* state, GuppiData* data)
{
  GuppiScatterState* ss;
  GuppiSeqScalar* sd;

  ss = GUPPI_SCATTER_STATE(state);
  sd = GUPPI_SEQ_SCALAR0(data);

  g_return_val_if_fail(ss != NULL, FALSE);

  if (ss->y_data == sd)
    return FALSE;

  if (ss->y_data)
    gtk_object_unref(GTK_OBJECT(ss->y_data));

  ss->y_data = sd;

  if (sd)
    gtk_object_ref(GTK_OBJECT(sd));

  return TRUE;
}

static GuppiData*
y_data_get(GuppiItemState* state)
{
  GuppiScatterState* ss;

  ss = GUPPI_SCATTER_STATE(state);
  g_return_val_if_fail(ss != NULL, FALSE);
  return GUPPI_DATA0(ss->y_data);
}


static void
guppi_scatter_state_class_init(GuppiScatterStateClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GuppiItemStateClass* is_class = GUPPI_ITEM_STATE_CLASS(klass);

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  gss_signals[CHANGED_STYLE] =
    gtk_signal_new("changed_style",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStateClass, changed_style),
		   gtk_marshal_NONE__INT,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_INT);

  gtk_object_class_add_signals(object_class, gss_signals, LAST_SIGNAL);

  object_class->destroy = guppi_scatter_state_destroy;
  object_class->finalize = guppi_scatter_state_finalize;

  is_class->set_x_data = x_data_set;
  is_class->set_y_data = y_data_set;

  is_class->get_x_data = x_data_get;
  is_class->get_y_data = y_data_get;

				     
}

static void
guppi_scatter_state_init(GuppiScatterState* ss)
{
  ss->x_data = NULL;
  ss->y_data = NULL;
}

GtkType
guppi_scatter_state_get_type(void)
{
  static GtkType gss_type = 0;

  if (!gss_type) {
    static const GtkTypeInfo gss_info = {
      "GuppiScatterState",
      sizeof(GuppiScatterState),
      sizeof(GuppiScatterStateClass),
      (GtkClassInitFunc)guppi_scatter_state_class_init,
      (GtkObjectInitFunc)guppi_scatter_state_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    gss_type = gtk_type_unique(GUPPI_TYPE_ITEM_STATE, &gss_info);
  }

  return gss_type;
}

GuppiScatterState*
guppi_scatter_state_new(void)
{
  return GUPPI_SCATTER_STATE(gtk_type_new(guppi_scatter_state_get_type()));
}

/**************************************************************************/

/* Convenience functions */

void
guppi_scatter_state_set_x_data(GuppiScatterState* ss, GuppiSeqScalar* sd)
{
  guppi_item_state_set_x_data(GUPPI_ITEM_STATE(ss), GUPPI_DATA0(sd));
}

void
guppi_scatter_state_set_y_data(GuppiScatterState* ss, GuppiSeqScalar* sd)
{
  guppi_item_state_set_y_data(GUPPI_ITEM_STATE(ss), GUPPI_DATA0(sd));
}


static void
style_change_cb(GuppiScatterStyle* style, GuppiScatterState* state)
{
  /* a bad hack */
  gtk_signal_emit(GTK_OBJECT(state), gss_signals[CHANGED_STYLE], 0);

  guppi_item_state_changed(GUPPI_ITEM_STATE(state));
}

GuppiScatterStyle*
guppi_scatter_state_get_style(GuppiScatterState* ss, gint i)
{
  g_return_val_if_fail(ss != NULL, NULL);
  g_return_val_if_fail(0 <= i && i < GUPPI_SCATTER_STATE_STYLE_COUNT, NULL);

  if (ss->style[i] == NULL) {
    ss->style[i] = GUPPI_SCATTER_STYLE(guppi_scatter_style_new_stock(i));
    gtk_signal_connect(GTK_OBJECT(ss->style[i]),
		       "changed",
		       GTK_SIGNAL_FUNC(style_change_cb),
		       ss);
  }

  return ss->style[i];
}

static void
guppi_scatter_state_brush_from_boolean_data(GuppiScatterState* ss,
					    gint brush_number,
					    GuppiSeqBoolean* bd)
{
  gint i, imax;
  gint j;

  g_return_if_fail(ss);
  g_return_if_fail(bd);

  g_return_if_fail(brush_number >= 0);
  g_return_if_fail(brush_number < GUPPI_SCATTER_STATE_STYLE_COUNT);

  if (ss->mask[0] == NULL) {
    ss->mask[0] = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(bd)));
    /* Everything starts out in style 0 */
    guppi_seq_boolean_set_all(ss->mask[0], TRUE);
  }
  
  if (ss->mask[brush_number] == NULL) {
    ss->mask[brush_number] = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(bd)));
  }

  imax = guppi_seq_max_index(GUPPI_SEQ(bd));
  for(i = guppi_seq_boolean_first_true(bd); i <= imax;
      i = guppi_seq_boolean_next_true(bd, i)) {

    /* Sweep through our masks, setting the given bit to zero in all
       of the masks except for the one corresponding to the current
       brushing.  Obviously, we might have performance problems if we
       made GUPPI_SCATTER_STATE_STYLE_COUNT too big.  (We'd have to
       start keeping an array of which element was in which brush
       category.  Not the end of the word, but not anything I want to
       get into right now.  */
    for(j=0; j<GUPPI_SCATTER_STATE_STYLE_COUNT; ++j) {
      if (ss->mask[j] != NULL) {
	guppi_seq_boolean_set(ss->mask[j], i, j == brush_number);
      }
    }
  }
}

gboolean
guppi_scatter_state_closest_point(GuppiScatterState* ss,
				  double x, double y,
				  double r,
				  double x_scale, double y_scale,
				  gint* index)
{
  GuppiSeqBoolean* bd = NULL;
  gint hits;
  gint i, imax, min_i=0;
  double xd, yd, dist, min_dist=1e+12;

  g_return_val_if_fail(index != NULL, FALSE);
  g_return_val_if_fail(ss != NULL, FALSE);
  g_return_val_if_fail(ss->x_data != NULL && ss->y_data != NULL, FALSE);

  bd = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(ss->x_data)));
  hits = guppi_seq_scalar_in_place_range_query(ss->x_data, bd, 
					       x-r*x_scale, x+r*x_scale);

  if (hits == 0) {
    gtk_object_unref(GTK_OBJECT(bd));
    return FALSE;
  }

  hits = guppi_seq_scalar_bitwise_and_range_query(ss->y_data, bd,
						  y-r*y_scale, y+r*y_scale);

  if (hits == 0) {
    gtk_object_unref(GTK_OBJECT(bd));
    return FALSE;
  }

  imax = guppi_seq_max_index(GUPPI_SEQ(bd));
  for(i = guppi_seq_boolean_first_true(bd); i <= imax;
      i = guppi_seq_boolean_next_true(bd, i)) {
    xd = (guppi_seq_scalar_get(ss->x_data, i) - x)/x_scale;
    yd = (guppi_seq_scalar_get(ss->y_data, i) - y)/y_scale;
    dist = xd*xd + yd*yd;
    if (dist < min_dist) {
      min_dist = dist;
      min_i = i;
    }
  }

  *index = min_i;

  gtk_object_unref(GTK_OBJECT(bd));
  return TRUE;
  
}

void
guppi_scatter_state_brush_rectangle(GuppiScatterState* ss,
				    gint brush_number,
				    double x0, double y0,
				    double x1, double y1)
{
  GuppiSeqBoolean* bd = NULL;
  gint hits;

  g_return_if_fail(ss != NULL);
  g_return_if_fail(ss->x_data != NULL && ss->y_data);

  bd = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(ss->x_data)));
  hits = guppi_seq_scalar_in_place_range_query(ss->x_data, bd, x0, x1);

  if (hits) {
    hits = guppi_seq_scalar_bitwise_and_range_query(ss->y_data, bd, y0, y1);
  }

  if (hits) {
    guppi_scatter_state_brush_from_boolean_data(ss,brush_number,bd);
  }
  
  gtk_object_unref(GTK_OBJECT(bd));

  guppi_item_state_changed(GUPPI_ITEM_STATE(ss));
}

/* Mostly code dup from above */
void
guppi_scatter_state_brush_circle(GuppiScatterState* ss,
				 gint brush_number,
				 double x, double y, double r,
				 double x_scale,
				 double y_scale)
{
  GuppiSeqBoolean* bd = NULL;
  gint hits;
  gint i, imax;
  double xd,yd,rr;

  g_return_if_fail(ss != NULL);
  
  r = abs(r);
  rr = r*r;

  bd = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(ss->x_data)));
  hits = guppi_seq_scalar_in_place_range_query(ss->x_data,
					       bd,
					       x-r*x_scale,
					       x+r*x_scale);

  if (hits) {
    hits = guppi_seq_scalar_bitwise_and_range_query(ss->y_data,
						    bd,
						    y-r*y_scale,
						    y+r*y_scale);
  }

  if (hits) {

    /* blow away anything outside of the radius */
    imax = guppi_seq_max_index(GUPPI_SEQ(bd));
    for(i = guppi_seq_boolean_first_true(bd); i <= imax;
	i = guppi_seq_boolean_next_true(bd, i)) {
      xd = (guppi_seq_scalar_get(ss->x_data, i) - x)/x_scale;
      yd = (guppi_seq_scalar_get(ss->y_data, i) - y)/y_scale;
      if (xd*xd + yd*yd > rr)
	guppi_seq_boolean_set(bd, i, FALSE);
    }

    guppi_scatter_state_brush_from_boolean_data(ss,brush_number,bd);
  }
  
  gtk_object_unref(GTK_OBJECT(bd));

  guppi_item_state_changed(GUPPI_ITEM_STATE(ss));
}


				    

/* $Id: guppi-scatter-state.c,v 1.10 2000/03/08 17:36:41 trow Exp $ */
