/* This is -*- C -*- */
/* $Id: guppi-seq-string.c,v 1.1 2000/02/09 21:04:56 trow Exp $ */

/*
 * guppi-seq-string.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-seq-string.h"
#include "guppi-seq-string-impl.h"
#include "guppi-seq-string-core-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_string_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_string_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_string_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_seq_string_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_seq_string_class_init(GuppiSeqStringClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GuppiDataClass* data_class = GUPPI_DATA_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_SEQ);

  data_class->default_impl = GUPPI_TYPE_SEQ_STRING_CORE_IMPL;
  data_class->type_name = _("String Sequence");

  object_class->get_arg = guppi_seq_string_get_arg;
  object_class->set_arg = guppi_seq_string_set_arg;
  object_class->destroy = guppi_seq_string_destroy;
  object_class->finalize = guppi_seq_string_finalize;

}

static void
guppi_seq_string_init(GuppiSeqString* obj)
{

}

GtkType
guppi_seq_string_get_type(void)
{
  static GtkType guppi_seq_string_type = 0;
  if (!guppi_seq_string_type) {
    static const GtkTypeInfo guppi_seq_string_info = {
      "GuppiSeqString",
      sizeof(GuppiSeqString),
      sizeof(GuppiSeqStringClass),
      (GtkClassInitFunc)guppi_seq_string_class_init,
      (GtkObjectInitFunc)guppi_seq_string_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_string_type = gtk_type_unique(GUPPI_TYPE_SEQ, &guppi_seq_string_info);
  }
  return guppi_seq_string_type;
}

GuppiData*
guppi_seq_string_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_seq_string_get_type()));
}

/****************************************************************************/

const gchar*
guppi_seq_string_get(const GuppiSeqString* seq, gint i)
{
  GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, NULL);
  g_return_val_if_fail(guppi_seq_in_bounds(GUPPI_SEQ(seq), i), NULL);
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->get);
  return (impl_class->get)(impl, i);
}

void
guppi_seq_string_set_nc(GuppiSeqString* seq, gint i, gchar* str)
{
  GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(guppi_seq_in_bounds(GUPPI_SEQ(seq), i));
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->set);
  (impl_class->set)(impl, i, str);
}

void
guppi_seq_string_set(GuppiSeqString* seq, gint i, const gchar* str)
{
  /* We do the basic checks here to try to avoid leaking memory in
     the event of a basic error. */
  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(guppi_seq_in_bounds(GUPPI_SEQ(seq), i));

  guppi_seq_string_set_nc(seq, i, g_strdup(str));
}

void
guppi_seq_string_prepend_nc(GuppiSeqString* seq, gchar* str)
{
  gint first;
  first = guppi_seq_min_index(GUPPI_SEQ(seq));
  guppi_seq_string_insert_nc(seq, first, str);
}

void
guppi_seq_string_prepend(GuppiSeqString* seq, const gchar* str)
{
  gint first;
  first = guppi_seq_min_index(GUPPI_SEQ(seq));
  guppi_seq_string_insert(seq, first, str);
}

void
guppi_seq_string_append_nc(GuppiSeqString* seq, gchar* str)
{
  gint last;
  last = guppi_seq_max_index(GUPPI_SEQ(seq));
  guppi_seq_string_insert_nc(seq, last+1, str);
}

void
guppi_seq_string_append(GuppiSeqString* seq, const gchar* str)
{
  gint last;
  last = guppi_seq_max_index(GUPPI_SEQ(seq));
  guppi_seq_string_insert(seq, last+1, str);
}

void
guppi_seq_string_insert_nc(GuppiSeqString* seq, gint i, gchar* str)
{
  GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;
  
  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(str != NULL);
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->insert);
  (impl_class->insert)(impl, i, str);
}

void
guppi_seq_string_insert(GuppiSeqString* seq, gint i, const gchar* str)
{
  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(str != NULL);

  guppi_seq_string_insert_nc(seq, i, g_strdup(str));
}

gint
guppi_seq_string_lookup(const GuppiSeqString* seq, const gchar* str)
{
  const GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, 0);
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->lookup) {
    
    return (impl_class->lookup)((GuppiSeqStringImpl*)impl, str);

  } else {
    gint i0 = guppi_seq_min_index(GUPPI_SEQ(seq));
    gint i1 = guppi_seq_max_index(GUPPI_SEQ(seq));
    gint i;

    g_assert(impl_class->get);

    for (i=i0; i<=i1; ++i) {
      if (!strcmp(str, (impl_class->get)(impl, i)))
	return i;
    }
    return i1+1;
  }
}

gboolean
guppi_seq_string_contains(const GuppiSeqString* seq, const gchar* str)
{
  const GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, 0);
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->contains) {
    
    return (impl_class->contains)((GuppiSeqStringImpl*)impl, str);

  } else {

    return guppi_seq_in_bounds(GUPPI_SEQ(seq), 
			       guppi_seq_string_lookup(seq, str));
  }
}

gsize
guppi_seq_string_distinct_values(const GuppiSeqString* seq)
{
  const GuppiSeqStringImpl* impl;
  GuppiSeqStringImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, 0);
  
  impl = GUPPI_SEQ_STRING_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->distinct) {
    
    return (impl_class->distinct)((GuppiSeqStringImpl*)impl);

  } else {
    gsize count = 0;
    GHashTable* hash = g_hash_table_new(g_str_hash, g_str_equal);
    const gchar* str;
    gpointer data;
    gint i0 = guppi_seq_min_index(GUPPI_SEQ(seq));
    gint i1 = guppi_seq_max_index(GUPPI_SEQ(seq));
    gint i;

    g_assert(impl_class->get);

    for (i=i0; i<=i1; ++i) {
      str = (impl_class->get)(impl, i);
      data = g_hash_table_lookup(hash, str);
      if (data == NULL) {
	++count;
	g_hash_table_insert(hash, (gchar*)str, (gpointer)str);
      }
    }

    g_hash_table_destroy(hash);
    return count;
  }
}



/* $Id: guppi-seq-string.c,v 1.1 2000/02/09 21:04:56 trow Exp $ */
