/* $Id: guppi-scatter-item.c,v 1.8 2000/02/28 05:06:17 trow Exp $ */

/*
 * guppi-scatter-item.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "guppi-scatter-item.h"

enum {
  ARG_0,
  ARG_STATE
};

static GtkObjectClass* parent_class = NULL;

static void
guppi_scatter_item_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  GuppiScatterItem* si = GUPPI_SCATTER_ITEM(obj);

  switch (arg_id) {
  case ARG_STATE:
    guppi_scatter_item_set_state(si,
				 GUPPI_SCATTER_STATE(GTK_VALUE_POINTER(*arg)));
    break;
  default:
    return;
    break;
  }

  guppi_canvas_item_request_update(GUPPI_CANVAS_ITEM(si));
}

static void
guppi_scatter_item_reset_x_bounds(GuppiScatterItem* si)
{
  g_return_if_fail(si != NULL);
  if (si->state && si->state->x_data)
    guppi_canvas_item_set_x_bounds(GUPPI_CANVAS_ITEM(si),
				   guppi_seq_scalar_min(si->state->x_data),
				   guppi_seq_scalar_max(si->state->x_data));
}

static void
guppi_scatter_item_reset_y_bounds(GuppiScatterItem* si)
{
  g_return_if_fail(si != NULL);
  if (si->state && si->state->y_data)
    guppi_canvas_item_set_y_bounds(GUPPI_CANVAS_ITEM(si),
				   guppi_seq_scalar_min(si->state->y_data),
				   guppi_seq_scalar_max(si->state->y_data));
}


static void
guppi_scatter_item_update(GnomeCanvasItem* item, double affine[6],
			  ArtSVP* clip_path, gint flags)
{
  GuppiScatterItem* si = GUPPI_SCATTER_ITEM(item);

  /* Chain our calls to update */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->update)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->update(item,affine,clip_path,flags);

  if (si->state == NULL)
    return;

  if (si->state->x_data == NULL || si->state->y_data == NULL)
    return;

  if (si->vis_mask)
    gtk_object_unref(GTK_OBJECT(si->vis_mask));
  si->vis_mask = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(si->state->x_data)));

  guppi_canvas_item_request_total_redraw(GUPPI_CANVAS_ITEM(item));
}

static void
guppi_scatter_item_render(GnomeCanvasItem* item,
			  GnomeCanvasBuf* buf)
{
  double rx0, rx1, ry0, ry1;
  GuppiScatterItem* si = GUPPI_SCATTER_ITEM(item);
  GuppiCanvasItem* gci = GUPPI_CANVAS_ITEM(item);
  GuppiScatterStyle* style;
  gint fuzz=5;
  gboolean got_hits = TRUE;
  gint i;
  double xmag, ymag, scale_factor;
  GuppiSeqBoolean* vis_mask_copy;

  /* Chain our calls to render */
  if (parent_class && GNOME_CANVAS_ITEM_CLASS(parent_class)->render)
    GNOME_CANVAS_ITEM_CLASS(parent_class)->render(item, buf);

  if (si->state == NULL)
    return;
  if (si->state->x_data == NULL || si->state->y_data == NULL)
    return;

  xmag = guppi_canvas_item_x_magnification(gci);
  ymag = guppi_canvas_item_y_magnification(gci);
  scale_factor = (xmag + ymag)/2;

  for (i=0; i<GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) 
    if (i == 0 || si->state->mask[i] != NULL) {

      style = guppi_scatter_state_get_style(si->state, i);

      if (style->marker == SCATTER_MARKER_ALPHA_TEMPLATE &&
	  style->alpha_template != NULL) {
	fuzz = MAX(fuzz, style->alpha_template->width/2 + 1);
	fuzz = MAX(fuzz, style->alpha_template->height/2 + 1);
      } else {

	fuzz = MAX(fuzz, 1.5 * style->size1 * scale_factor + 1);

	if (style->size1_data != NULL)
	  fuzz = MAX(fuzz, 1.5 * style->size1_high * scale_factor + 1);

      }
    }

  /* Find the plot coordinates of our render area */
  guppi_canvas_item_pixel2plot(gci,
			       buf->rect.x0-fuzz, buf->rect.y0-fuzz,
			       &rx0, &ry0);
  guppi_canvas_item_pixel2plot(gci,
			       buf->rect.x1+fuzz, buf->rect.y1+fuzz,
			       &rx1, &ry1);

  /* Perform our window query and build up a bit mask indicating which
     points we need to draw. */
  guppi_seq_boolean_clear(si->vis_mask);
  got_hits = guppi_seq_scalar_in_place_range_query(si->state->x_data,
						   si->vis_mask,
						   rx0, rx1);
  if (got_hits) {
    got_hits = guppi_seq_scalar_bitwise_and_range_query(si->state->y_data,
							si->vis_mask,
							ry0, ry1);
  }
    
  if (got_hits) {
    
    for (i=0; i<GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {

      /* We let the base mask 0 through, and don't interfere with si->vis_mask
	 if mask[i] if NULL.  When no brushing is active, the extra
	 overhead of looping and testing all of the mask[i]'s (which will
	 all be NULL) should be totally negligable. */

      if (i == 0 || si->state->mask[i] != NULL) {
	
	style = guppi_scatter_state_get_style(si->state, i);
	if (style == NULL)
	  g_error("Evil stalks the land");

	if (si->state->mask[i]) {
	  vis_mask_copy = GUPPI_SEQ_BOOLEAN(guppi_data_copy(GUPPI_DATA(si->vis_mask)));
	  guppi_seq_boolean_bitwise_and(vis_mask_copy, si->state->mask[i]);
	} else {
	  vis_mask_copy = si->vis_mask;
	  gtk_object_ref(GTK_OBJECT(si->vis_mask));
	}

	guppi_scatter_style_paint(style,
				  buf->buf, buf->buf_rowstride,
				  buf->rect.x1 - buf->rect.x0,
				  buf->rect.y1 - buf->rect.y0,
				  buf->rect.x0, buf->rect.y0,
				  gci->x_transform, gci->y_transform,
				  si->state->x_data, si->state->y_data,
				  vis_mask_copy,
				  scale_factor);
	gtk_object_unref(GTK_OBJECT(vis_mask_copy));

      }
    }
  }
}

static void
bound_hack(GnomeCanvasItem* c, double* x1, double* y1, double* x2, double* y2)
{
  g_assert_not_reached();
}

static void
guppi_scatter_item_class_init(GuppiScatterItemClass* klass)
{
  GtkObjectClass* object_class;
  GuppiCanvasItemClass* gci_class;
  GnomeCanvasItemClass* item_class;

  object_class = (GtkObjectClass*)klass;
  gci_class = GUPPI_CANVAS_ITEM_CLASS(klass);
  item_class = GNOME_CANVAS_ITEM_CLASS(klass);
  parent_class = gtk_type_class(GUPPI_TYPE_CANVAS_ITEM);

  gtk_object_add_arg_type("GuppiScatterItem::state",
			  GTK_TYPE_POINTER,
			  GTK_ARG_WRITABLE,
			  ARG_STATE);

  object_class->set_arg = guppi_scatter_item_set_arg;

  item_class->update = guppi_scatter_item_update;
  item_class->render = guppi_scatter_item_render;
  item_class->bounds = bound_hack;
}

static void
guppi_scatter_item_init(GuppiScatterItem* si)
{
  si->state = NULL;
  si->vis_mask = NULL;

}

GtkType
guppi_scatter_item_get_type(void)
{
  static GtkType scatter_item_type = 0;

  if (!scatter_item_type) {
    static const GtkTypeInfo scatter_item_info = {
      "GuppiScatterItem",
      sizeof(GuppiScatterItem),
      sizeof(GuppiScatterItemClass),
      (GtkClassInitFunc)guppi_scatter_item_class_init,
      (GtkObjectInitFunc)guppi_scatter_item_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    scatter_item_type = gtk_type_unique(GUPPI_TYPE_CANVAS_ITEM,
					&scatter_item_info);
  }

  return scatter_item_type;
}

void
guppi_scatter_item_set_state(GuppiScatterItem* si,
			     GuppiScatterState* ss)
{
  g_return_if_fail(si != NULL);

  /* We should do all sorts of things here: connect signals, 
     add references, clean things up if si->state was non-null prior
     to this assignment, etc. etc. These details can come later. */
  si->state = ss;

  guppi_canvas_item_set_state(GUPPI_CANVAS_ITEM(si),
			      ss ? GUPPI_ITEM_STATE(ss) : NULL);

  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_x_data",
			    GTK_SIGNAL_FUNC(guppi_scatter_item_reset_x_bounds),
			    GTK_OBJECT(si));
  gtk_signal_connect_object(GTK_OBJECT(ss),
			    "changed_y_data",
			    GTK_SIGNAL_FUNC(guppi_scatter_item_reset_y_bounds),
			    GTK_OBJECT(si));

  guppi_scatter_item_reset_x_bounds(si);
  guppi_scatter_item_reset_y_bounds(si);
  guppi_canvas_item_request_update(GUPPI_CANVAS_ITEM(si));


}



/* $Id: guppi-scatter-item.c,v 1.8 2000/02/28 05:06:17 trow Exp $ */
