/* $Id: guppi-scatter-state.c,v 1.3 2000/01/21 06:11:36 trow Exp $ */

/*
 * guppi-scatter-state.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include <gtk/gtkmarshal.h>
#include "guppi-scatter-state.h"

enum {
  CHANGED_X_DATA,
  CHANGED_Y_DATA,
  CHANGED_STYLE,
  CHANGED,
  LAST_SIGNAL
};

static GtkObjectClass* parent_class = NULL;
static guint gss_signals[LAST_SIGNAL] = { 0 };

static void
guppi_scatter_state_destroy(GtkObject* obj)
{
  GuppiScatterState* gss = GUPPI_SCATTER_STATE(obj);
  gint i;

  if (gss->x_data)
    gtk_object_unref(GTK_OBJECT(gss->x_data));
  if (gss->y_data)
    gtk_object_unref(GTK_OBJECT(gss->y_data));

  for(i=0; i<GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {
    if (gss->style[i] != NULL)
      gtk_object_unref(GTK_OBJECT(gss->style[i]));
    if (gss->mask[i] != NULL)
      gtk_object_unref(GTK_OBJECT(gss->mask[i]));
  }

 if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_scatter_state_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_scatter_state_class_init(GuppiScatterStateClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  gss_signals[CHANGED_X_DATA] =
    gtk_signal_new("changed_x_data",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStateClass, changed_x_data),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gss_signals[CHANGED_Y_DATA] =
    gtk_signal_new("changed_y_data",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStateClass, changed_y_data),
		   gtk_marshal_NONE__POINTER,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);

  gss_signals[CHANGED_STYLE] =
    gtk_signal_new("changed_style",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStateClass, changed_style),
		   gtk_marshal_NONE__INT,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_INT);

  gss_signals[CHANGED] =
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStateClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gss_signals, LAST_SIGNAL);

  object_class->destroy = guppi_scatter_state_destroy;
  object_class->finalize = guppi_scatter_state_finalize;
				     
}

static void
guppi_scatter_state_init(GuppiScatterState* ss)
{
  ss->x_data = NULL;
  ss->y_data = NULL;
}

GtkType
guppi_scatter_state_get_type(void)
{
  static GtkType gss_type = 0;

  if (!gss_type) {
    static const GtkTypeInfo gss_info = {
      "GuppiScatterState",
      sizeof(GuppiScatterState),
      sizeof(GuppiScatterStateClass),
      (GtkClassInitFunc)guppi_scatter_state_class_init,
      (GtkObjectInitFunc)guppi_scatter_state_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    gss_type = gtk_type_unique(GUPPI_TYPE_ITEM_STATE, &gss_info);
  }

  return gss_type;
}

GuppiScatterState*
guppi_scatter_state_new(void)
{
  return GUPPI_SCATTER_STATE(gtk_type_new(guppi_scatter_state_get_type()));
}

/****************************************************************************/

static void
guppi_scatter_state_request_update(GuppiScatterState* ss)
{
  guppi_item_state_request_update(GUPPI_ITEM_STATE(ss));
}

static void
guppi_scatter_state_unset_data(GuppiScatterState* ss, gboolean is_x)
{
  gpointer ptr;
  guint handler;

  g_return_if_fail(ss != NULL);

  ptr = is_x ? ss->x_data : ss->y_data;
  if (ptr == NULL)
    return;

  handler = is_x ? ss->x_signal_hander : ss->y_signal_handler;

  gtk_signal_disconnect(GTK_OBJECT(ptr), handler);
  gtk_object_unref(GTK_OBJECT(ptr));
}

/* Callbacks to redirect any data-change signals into scatter-state signals */

static void
changed_x_cb(GuppiData* d, gpointer us)
{
  gtk_signal_emit(GTK_OBJECT(us), gss_signals[CHANGED_X_DATA], d);
  gtk_signal_emit(GTK_OBJECT(us), gss_signals[CHANGED]);
}

static void
changed_y_cb(GuppiData* d, gpointer us)
{
  gtk_signal_emit(GTK_OBJECT(us), gss_signals[CHANGED_Y_DATA], d);
  gtk_signal_emit(GTK_OBJECT(us), gss_signals[CHANGED]);
}

void
guppi_scatter_state_set_x_data(GuppiScatterState* ss, GuppiScalarData* sd)
{
  g_return_if_fail(ss);

  if (sd == ss->x_data)
    return;

  guppi_scatter_state_unset_data(ss, TRUE);
  if (sd != NULL)
    gtk_object_ref(GTK_OBJECT(sd));
  ss->x_data = sd;

  ss->x_signal_hander = 0;
  if (sd != NULL)
    ss->x_signal_hander = gtk_signal_connect(GTK_OBJECT(sd),
					     "changed",
					     changed_x_cb,
					     ss);

  gtk_signal_emit(GTK_OBJECT(ss), gss_signals[CHANGED_X_DATA], sd);
  gtk_signal_emit(GTK_OBJECT(ss), gss_signals[CHANGED]);

  guppi_scatter_state_request_update(ss);
}

void
guppi_scatter_state_set_y_data(GuppiScatterState* ss, GuppiScalarData* sd)
{
  g_return_if_fail(ss);

  if (sd == ss->y_data)
    return;

  guppi_scatter_state_unset_data(ss, FALSE);
  if (sd != NULL)
    gtk_object_ref(GTK_OBJECT(sd));
  ss->y_data = sd;

  ss->y_signal_handler = 0;
  if (sd != NULL)
    ss->y_signal_handler = gtk_signal_connect(GTK_OBJECT(sd),
					      "changed",
					      changed_y_cb,
					      ss);

  gtk_signal_emit(GTK_OBJECT(ss), gss_signals[CHANGED_Y_DATA], sd);
  gtk_signal_emit(GTK_OBJECT(ss), gss_signals[CHANGED]);

  guppi_scatter_state_request_update(ss);
}

static void
style_change_cb(GuppiScatterStyle* style, GuppiScatterState* state)
{
  /* a bad hack */
  gtk_signal_emit(GTK_OBJECT(state), gss_signals[CHANGED_STYLE], 0);
  gtk_signal_emit(GTK_OBJECT(state), gss_signals[CHANGED]);

  guppi_scatter_state_request_update(state);
}

GuppiScatterStyle*
guppi_scatter_state_get_style(GuppiScatterState* ss, gint i)
{
  g_return_val_if_fail(ss != NULL, NULL);
  g_return_val_if_fail(0 <= i && i < GUPPI_SCATTER_STATE_STYLE_COUNT, NULL);

  if (ss->style[i] == NULL) {
    ss->style[i] = GUPPI_SCATTER_STYLE(guppi_scatter_style_new_stock(i));
    gtk_signal_connect(GTK_OBJECT(ss->style[i]),
		       "changed",
		       GTK_SIGNAL_FUNC(style_change_cb),
		       ss);
  }

  return ss->style[i];
}

static void
guppi_scatter_state_brush_from_boolean_data(GuppiScatterState* ss,
					    gint brush_number,
					    GuppiBooleanData* bd)
{
  dindex_t i, imax;
  gint j;

  g_return_if_fail(ss);
  g_return_if_fail(bd);

  g_return_if_fail(brush_number >= 0);
  g_return_if_fail(brush_number < GUPPI_SCATTER_STATE_STYLE_COUNT);

  if (ss->mask[0] == NULL) {
    ss->mask[0] = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(bd)));
    /* Everything starts out in style 0 */
    guppi_boolean_data_logical_not(ss->mask[0]);
  }
  
  if (ss->mask[brush_number] == NULL) {
    ss->mask[brush_number] = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(bd)));
  }

  imax = guppi_data_max_index(GUPPI_DATA(bd));
  for(i = guppi_boolean_data_first_true(bd); i <= imax;
      i = guppi_boolean_data_next_true(bd, i)) {

    /* Sweep through our masks, setting the given bit to zero in all
       of the masks except for the one corresponding to the current
       brushing.  Obviously, we might have performance problems if we
       made GUPPI_SCATTER_STATE_STYLE_COUNT too big.  (We'd have to
       start keeping an array of which element was in which brush
       category.  Not the end of the word, but not anything I want to
       get into right now.  */
    for(j=0; j<GUPPI_SCATTER_STATE_STYLE_COUNT; ++j) {
      if (ss->mask[j] != NULL) {
	if (j == brush_number) {
	  guppi_boolean_data_set_true_unsafe(ss->mask[j], i);
	} else {
	  guppi_boolean_data_set_false_unsafe(ss->mask[j], i);
	}
      }
    }
  }
}

void
guppi_scatter_state_brush_rectangle(GuppiScatterState* ss,
				    gint brush_number,
				    double x0, double y0,
				    double x1, double y1)
{
  GuppiBooleanData* bd1 = NULL;
  GuppiBooleanData* bd2 = NULL;
  gint hits;

  g_return_if_fail(ss != NULL);

  bd1 = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(ss->x_data)));
  hits = guppi_scalar_data_range_query(ss->x_data,x0,x1,bd1);

  if (hits) {
    bd2 = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(ss->x_data)));
    hits = guppi_scalar_data_range_query(ss->y_data,y0,y1,bd2);
  }

  if (hits) {
    guppi_boolean_data_logical_and(bd1, bd2);
    guppi_scatter_state_brush_from_boolean_data(ss,brush_number,bd1);
  }
  
  gtk_object_unref(GTK_OBJECT(bd1));
  gtk_object_unref(GTK_OBJECT(bd2));

  gtk_signal_emit(GTK_OBJECT(ss), gss_signals[CHANGED]);

  guppi_scatter_state_request_update(ss);
}

/* Mostly code dup from above */
void
guppi_scatter_state_brush_circle(GuppiScatterState* ss,
				 gint brush_number,
				 double x, double y, gint r,
				 double x_scale,
				 double y_scale)
{
  GuppiBooleanData* bd1 = NULL;
  GuppiBooleanData* bd2 = NULL;
  gint hits;
  dindex_t i, imax;
  double xd,yd,rr;

  g_return_if_fail(ss != NULL);
  
  r = abs(r);
  rr = r*r;

  bd1 = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(ss->x_data)));
  hits = guppi_scalar_data_range_query(ss->x_data,
				       x-r*x_scale,
				       x+r*x_scale,
				       bd1);

  if (hits) {
    bd2 = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(GUPPI_DATA(ss->x_data)));
    hits = guppi_scalar_data_range_query(ss->y_data,
					 y-r*y_scale,
					 y+r*y_scale,
					 bd2);
  }

  if (hits) {
    guppi_boolean_data_logical_and(bd1, bd2);

    /* blow away anything outside of the radius */
    imax = guppi_data_max_index(GUPPI_DATA(bd1));
    for(i = guppi_boolean_data_first_true(bd1); i <= imax;
	i = guppi_boolean_data_next_true(bd1, i)) {
      xd = (guppi_scalar_data_get_unsafe(ss->x_data, i) - x)/x_scale;
      yd = (guppi_scalar_data_get_unsafe(ss->y_data, i) - y)/y_scale;
      if (xd*xd + yd*yd > rr)
	guppi_boolean_data_set_false_unsafe(bd1, i);
    }

    guppi_scatter_state_brush_from_boolean_data(ss,brush_number,bd1);
  }
  
  gtk_object_unref(GTK_OBJECT(bd1));
  gtk_object_unref(GTK_OBJECT(bd2));

  guppi_scatter_state_request_update(ss);
}


				    

/* $Id: guppi-scatter-state.c,v 1.3 2000/01/21 06:11:36 trow Exp $ */
