/* $Id: guppi-scm-categorical-data.c,v 1.1 2000/01/17 05:44:54 trow Exp $ */

/*
 * guppi-scm-categorical-data.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "guppi-scm-data.h"
#include "guppi-scm-categorical-data.h"


#define SCM_TO_CATEGORICAL_DATA(x) (GUPPI_CATEGORICAL_DATA(SCM_CDR(x)))
#define SCM_CATEGORICAL_DATA_P(x) \
     (SCM_NIMP(x) && scm_data_p(x) && GUPPI_IS_CATEGORICAL_DATA(SCM_CDR(x)))

gboolean
scm_categorical_data_p(SCM x)
{
  return SCM_CATEGORICAL_DATA_P(x);
}

GuppiCategoricalData*
scm2categorical_data(SCM x)
{
  return SCM_CATEGORICAL_DATA_P(x) ? SCM_TO_CATEGORICAL_DATA(x) : NULL;
}

/*****************************************************************************/

GUPPI_PROC(catd_new, "make-categorical-data",
	   0,0,0, ())
{
  return data2scm(GUPPI_DATA(guppi_categorical_data_new()));
}

GUPPI_PROC(catdp, "categorical-data?",
	   1,0,0, (SCM x))
{
  return gh_bool2scm(SCM_CATEGORICAL_DATA_P(x));
}

GUPPI_PROC(catd_frozen_p, "categorical-data-frozen?",
	   1,0,0, (SCM scd))
{
  GuppiCategoricalData* cd;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_frozen_p);
  cd = SCM_TO_CATEGORICAL_DATA(scd);

  return gh_bool2scm(guppi_categorical_data_frozen(cd));
}

GUPPI_PROC(catd_freeze, "categorical-data-freeze!",
	   1,0,0, (SCM scd))
{
  GuppiCategoricalData* cd;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_freeze);
  cd = SCM_TO_CATEGORICAL_DATA(scd);
  guppi_categorical_data_freeze(cd);

  return scd;
}

GUPPI_PROC(catd_unfreeze, "categorical-data-unfreeze!",
	   1,0,0, (SCM scd))
{
  GuppiCategoricalData* cd;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_unfreeze);
  cd = SCM_TO_CATEGORICAL_DATA(scd);
  guppi_categorical_data_unfreeze(cd);

  return scd;
}

GUPPI_PROC(catd_encode, "categorical-data-encode",
	   2,0,0, (SCM scd, SCM sstr))
{
  GuppiCategoricalData* cd;
  gchar* str;
  guint c;
  gint len;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_encode);
  SCM_ASSERT(gh_string_p(sstr), sstr, SCM_ARG2, str_catd_encode);

  cd = SCM_TO_CATEGORICAL_DATA(scd);

  str = gh_scm2newstr(sstr, &len);
  c = guppi_categorical_data_encode(cd, str);
  g_free(str);

  if (c == GUPPI_CATEGORICAL_DATA_BAD_CODE)
    return SCM_BOOL_F;

  return gh_int2scm(c);
}

GUPPI_PROC(catd_decode, "categorical-data-decode",
	   2,0,0, (SCM scd, SCM sc))
{
  GuppiCategoricalData* cd;
  const gchar* str;
  gint c;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_decode);
  SCM_ASSERT(gh_exact_p(sc), sc, SCM_ARG2, str_catd_decode);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  c = gh_scm2int(sc);
  if (c < 0)
    return SCM_BOOL_F;
  str = guppi_categorical_data_decode(cd, c);
  if (str == NULL)
    return SCM_BOOL_F;
  return gh_str02scm((gchar*)str);
}

GUPPI_PROC(catd_get, "categorical-data-get",
	   2,0,0, (SCM scd, SCM si))
{
  GuppiCategoricalData* cd;
  gint i;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_get);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_get);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_get);

  return gh_str02scm((gchar*)guppi_categorical_data_get(cd, i));
}

GUPPI_PROC(catd_get_code, "categorical-data-get-code",
	   2,0,0, (SCM scd, SCM si))
{
  GuppiCategoricalData* cd;
  gint i;
  guint c;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_get_code);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_get_code);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_get_code);
  
  c = guppi_categorical_data_get_code(cd, i);
  return gh_int2scm(c);

}

GUPPI_PROC(catd_set, "categorical-data-set!",
	   3,0,0, (SCM scd, SCM si, SCM sx))
{
  GuppiCategoricalData* cd;
  gint i;
  gchar* x;
  gint len;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_set);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_set);
  SCM_ASSERT(gh_string_p(sx), sx, SCM_ARG3, str_catd_set);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_set);
  x = gh_scm2newstr(sx, &len);
  guppi_categorical_data_set(cd, i, x);

  return scd;
}

GUPPI_PROC(catd_set_code, "categorical-data-set-code!",
	   3,0,0, (SCM scd, SCM si, SCM sx))
{
  GuppiCategoricalData* cd;
  gint i;
  gint c;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_set_code);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_set_code);
  SCM_ASSERT(gh_exact_p(sx), sx, SCM_ARG3, str_catd_set_code);

  cd = SCM_TO_CATEGORICAL_DATA(scd);

  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si, 
	     SCM_OUTOFRANGE, str_catd_set_code);

  c = gh_scm2int(sx);
  SCM_ASSERT(c >= 0, sx, SCM_ARG3, str_catd_set_code);
  guppi_categorical_data_set_code(cd, i, c);

  return scd;
}

GUPPI_PROC(catd_add, "categorical-data-add!",
	   2,0,0, (SCM cd, SCM x))
{
  gchar* s;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(cd), cd, SCM_ARG1, str_catd_add);
  SCM_ASSERT(gh_string_p(x), x, SCM_ARG2, str_catd_add);

  s = gh_scm2newstr(x, NULL);
  guppi_categorical_data_add(SCM_TO_CATEGORICAL_DATA(cd), s);
  g_free(s);

  return cd;
}

GUPPI_PROC(catd_add_code, "categorical-data-add-code!",
	   2,0,0, (SCM cd, SCM x))
{
  gint c;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(cd), cd, SCM_ARG1, str_catd_add_code);
  SCM_ASSERT(gh_exact_p(x), x, SCM_ARG2, str_catd_add_code);

  c = gh_scm2int(x);
  SCM_ASSERT(c >= 0, x, SCM_OUTOFRANGE, str_catd_add_code);

  guppi_categorical_data_add_code(SCM_TO_CATEGORICAL_DATA(cd), (guint)c);

  return cd;
}

GUPPI_PROC(catd_ins, "categorical-data-insert!",
	   3,0,0, (SCM scd, SCM si, SCM sx))
{
  GuppiCategoricalData* cd;
  gint i;
  gchar* x;
  gint len;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_ins);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_ins);
  SCM_ASSERT(gh_string_p(sx), sx, SCM_ARG3, str_catd_ins);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_ins);

  x = gh_scm2newstr(sx, &len);
  guppi_categorical_data_insert(cd, i, x);

  return scd;
}

GUPPI_PROC(catd_ins_code, "categorical-data-insert-code!",
	   3,0,0, (SCM scd, SCM si, SCM sx))
{
  GuppiCategoricalData* cd;
  gint i;
  gint c;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_ins_code);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_ins_code);
  SCM_ASSERT(gh_exact_p(sx), sx, SCM_ARG3, str_catd_ins_code);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_ins);

  c = gh_scm2int(sx);
  SCM_ASSERT(c >= 0, sx, SCM_OUTOFRANGE, str_catd_ins_code);
  guppi_categorical_data_insert_code(cd, i, c);

  return scd;
}

GUPPI_PROC(catd_del, "categorical-data-delete!",
	   2,0,0, (SCM scd, SCM si))
{
  GuppiCategoricalData* cd;
  gint i;

  SCM_ASSERT(SCM_CATEGORICAL_DATA_P(scd), scd, SCM_ARG1, str_catd_del);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_catd_del);

  cd = SCM_TO_CATEGORICAL_DATA(scd);
  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(GUPPI_DATA(cd), i), si,
	     SCM_OUTOFRANGE, str_catd_del);

  guppi_categorical_data_delete(cd, i);

  return scd;
}


/*****************************************************************************/

void
guppi_scm_categorical_data_init(void)
{
  static gboolean initialized = FALSE;
  g_return_if_fail(!initialized);
  initialized = TRUE;

#include "guppi-scm-categorical-data.x"
}


/* $Id: guppi-scm-categorical-data.c,v 1.1 2000/01/17 05:44:54 trow Exp $ */
