/* $Id: guppi-scm-scatter-style.c,v 1.4 2000/01/18 21:06:00 trow Exp $ */

/*
 * guppi-scm-scatter-style.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <guppi-scm-scalar-data.h>
#include "guppi-scm-scatter-style.h"
#include "guppi-scatter-style-control.h"
#include <rgb.h>

static long scatstyle_type_tag;

#define SCM_TO_SCATTER_STYLE(x) (GUPPI_SCATTER_STYLE(SCM_CDR(x)))
#define SCM_SCATTER_STYLE_P(x) (SCM_NIMP(x) && SCM_CAR(x) == scatstyle_type_tag)

gboolean
scm_scatter_style_p(SCM x)
{
  return SCM_SCATTER_STYLE_P(x);
}

SCM
scatter_style2scm(GuppiScatterStyle* x)
{
  SCM smob;

  SCM_DEFER_INTS;
  SCM_NEWCELL(smob);
  SCM_SETCAR(smob, scatstyle_type_tag);
  SCM_SETCDR(smob, x);
  gtk_object_ref(GTK_OBJECT(x));
  SCM_ALLOW_INTS;

  return smob;
}

GuppiScatterStyle*
scm2scatter_style(SCM x)
{
  return SCM_SCATTER_STYLE_P(x) ? SCM_TO_SCATTER_STYLE(x) : NULL;
}

/*****************************************************************************/

GUPPI_PROC(ssty, "scatter-style?",
	   1,0,0, (SCM x))
{
  return gh_bool2scm(SCM_SCATTER_STYLE_P(x));
}

GUPPI_PROC(ssty_new, "make-scatter-style",
	   0,0,0, ())
{
  return scatter_style2scm(guppi_scatter_style_new());
}

GUPPI_PROC(ssty_stock_new, "make-stock-scatter-style",
	   1,0,0, (SCM si))
{
  gint i;
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG1, str_ssty_stock_new);

  i = gh_scm2int(si);
  SCM_ASSERT(i>=0, si, SCM_OUTOFRANGE, str_ssty_stock_new);

  return scatter_style2scm(guppi_scatter_style_new_stock(i));
}

GUPPI_PROC(ssty_freeze, "scatter-style-freeze",
	   1,0,0, (SCM ss))
{
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ss), ss, SCM_ARG1, str_ssty_freeze);
  guppi_scatter_style_freeze(SCM_TO_SCATTER_STYLE(ss));
  return ss;
}

GUPPI_PROC(ssty_thaw, "scatter-style-thaw",
	   1,0,0, (SCM ss))
{
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ss), ss, SCM_ARG1, str_ssty_thaw);
  guppi_scatter_style_thaw(SCM_TO_SCATTER_STYLE(ss));
  return ss;
}

GUPPI_PROC(ssty_marker, "scatter-style-marker",
	   1,0,0, (SCM ssty))
{
  GuppiScatterStyle* sty;
  GuppiScatterMarker marker;
  const gchar* name;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_marker);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  marker = guppi_scatter_style_marker(sty);
  name = guppi_scatter_marker_info[marker].name;

  return gh_symbol2scm((gchar*)name);
}

GUPPI_PROC(ssty_set_marker, "scatter-style-set-marker!",
	   2,0,0, (SCM ssty, SCM x))
{
  GuppiScatterStyle* sty;
  GuppiScatterMarker marker;
  gchar* symb;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_marker);
  SCM_ASSERT(gh_symbol_p(x), x, SCM_ARG2, str_ssty_set_marker);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  symb = gh_symbol2newstr(x, NULL);
  marker = string2scatter_marker(symb);
  SCM_ASSERT(marker != SCATTER_MARKER_NONE, x, SCM_OUTOFRANGE, str_ssty_set_marker);

  guppi_scatter_style_set_marker(sty, marker);

  g_free(symb);
  return ssty;
}

GUPPI_PROC(ssty_set_sizes, "scatter-style-set-marker-sizes!",
	   3,0,0, (SCM ssty, SCM ss1, SCM ss2))
{ 
  GuppiScatterStyle* sty;
  double s1, s2;
  GuppiScatterMarker marker;
  const GuppiScatterMarkerInfo* info;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_sizes);
  SCM_ASSERT(gh_number_p(ss1), ss1, SCM_ARG2, str_ssty_set_sizes);
  SCM_ASSERT(gh_number_p(ss2), ss2, SCM_ARG3, str_ssty_set_sizes);

  sty = SCM_TO_SCATTER_STYLE(ssty);

  marker = sty->marker;
  info = &(guppi_scatter_marker_info[(gint)marker]);
	   
  s1 = gh_scm2double(ss1);
  s2 = gh_scm2double(ss2);

  if (info->size1_desc != NULL) {
    SCM_ASSERT(s1 >= info->size1_min, ss1, SCM_OUTOFRANGE, str_ssty_set_sizes);
    SCM_ASSERT(s1 <= info->size1_max, ss1, SCM_OUTOFRANGE, str_ssty_set_sizes);
  }

  if (info->size2_desc != NULL) {
    SCM_ASSERT(s2 >= info->size2_min, ss2, SCM_OUTOFRANGE, str_ssty_set_sizes);
    SCM_ASSERT(s2 <= info->size2_max, ss2, SCM_OUTOFRANGE, str_ssty_set_sizes);
  }

  guppi_scatter_style_set_marker_sizes(sty, s1, s2);

  return ssty;
}

GUPPI_PROC(ssty_set_defsz, "scatter-style-set-default-marker-sizes!",
	   1,0,0, (SCM ssty))
{
  GuppiScatterStyle* sty;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_defsz);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  guppi_scatter_style_set_marker_sizes_default(sty);

  return ssty;
}

/* missing set_color */

GUPPI_PROC(ssty_set_cdata, "scatter-style-set-color-data!",
	   2,0,0, (SCM ssty, SCM ssd))
{
  GuppiScatterStyle* sty;
  GuppiScalarData* sd;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_cdata);
  SCM_ASSERT(scm_scalar_data_p(ssd), ssd, SCM_ARG2, str_ssty_set_cdata);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  sd = scm2scalar_data(ssd);

  guppi_scatter_style_set_gradient_data(sty, sd);

  return ssty;
}

/* missing set_color_low */

GUPPI_PROC(ssty_rev_cgradp, "scatter-style-reverse-color-gradient?",
	   1,0,0, (SCM ssty))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_cgradp);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  return gh_bool2scm(guppi_scatter_style_reverse_color_gradient(sty));
}

GUPPI_PROC(ssty_rev_cgrad, "scatter-style-set-reverse-color-gradient!",
	   1,0,0, (SCM ssty, SCM sx))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_cgrad);
  SCM_ASSERT(gh_boolean_p(sx), sx, SCM_ARG2, str_ssty_rev_cgrad);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  guppi_scatter_style_set_reverse_color_gradient(sty, gh_scm2bool(sx));
  return ssty;
}

GUPPI_PROC(ssty_set_s1data, "scatter-style-set-size1-data!",
	   2,0,0, (SCM ssty, SCM ssd))
{
  GuppiScatterStyle* sty;
  GuppiScalarData* sd;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_s1data);
  SCM_ASSERT(scm_scalar_data_p(ssd), ssd, SCM_ARG2, str_ssty_set_s1data);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  sd = scm2scalar_data(ssd);

  guppi_scatter_style_set_size1_data(sty, sd);

  return ssty;
}

/* missing set_size1_high */

GUPPI_PROC(ssty_rev_s1gradp, "scatter-style-reverse-size1-gradient?",
	   1,0,0, (SCM ssty))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_s1gradp);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  return gh_bool2scm(guppi_scatter_style_reverse_size1_gradient(sty));
}

GUPPI_PROC(ssty_rev_s1grad, "scatter-style-set-reverse-size1-gradient!",
	   1,0,0, (SCM ssty, SCM sx))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_s1grad);
  SCM_ASSERT(gh_boolean_p(sx), sx, SCM_ARG2, str_ssty_rev_s1grad);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  guppi_scatter_style_set_reverse_size1_gradient(sty, gh_scm2bool(sx));
  return ssty;
}


GUPPI_PROC(ssty_set_s2data, "scatter-style-set-size2-data!",
	   2,0,0, (SCM ssty, SCM ssd))
{
  GuppiScatterStyle* sty;
  GuppiScalarData* sd;

  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_set_s2data);
  SCM_ASSERT(scm_scalar_data_p(ssd), ssd, SCM_ARG2, str_ssty_set_s2data);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  sd = scm2scalar_data(ssd);

  guppi_scatter_style_set_size2_data(sty, sd);

  return ssty;
}


/* missing set_size2_high */

GUPPI_PROC(ssty_rev_s2gradp, "scatter-style-reverse-size2-gradient?",
	   1,0,0, (SCM ssty))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_s2gradp);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  return gh_bool2scm(guppi_scatter_style_reverse_size2_gradient(sty));
}

GUPPI_PROC(ssty_rev_s2grad, "scatter-style-set-reverse-size2-gradient!",
	   1,0,0, (SCM ssty, SCM sx))
{
  GuppiScatterStyle* sty;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(ssty), ssty, SCM_ARG1, str_ssty_rev_s2grad);
  SCM_ASSERT(gh_boolean_p(sx), sx, SCM_ARG2, str_ssty_rev_s2grad);

  sty = SCM_TO_SCATTER_STYLE(ssty);
  guppi_scatter_style_set_reverse_size2_gradient(sty, gh_scm2bool(sx));
  return ssty;
}




/*****************************************************************************/

GUPPI_PROC(hack, "edit-scatter-style", 1,0,0, (SCM x))
{
  GtkWidget* win;
  GtkWidget* w;
  SCM_ASSERT(SCM_SCATTER_STYLE_P(x), x, SCM_ARG1, str_hack);

  w = guppi_scatter_style_control_new(SCM_TO_SCATTER_STYLE(x));
  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_add(GTK_CONTAINER(win), w);
  gtk_widget_show_all(win);
  return SCM_UNSPECIFIED;
}

/*****************************************************************************/

static SCM
mark_data(SCM x)
{
  return SCM_BOOL_F;
}

static scm_sizet
free_data(SCM x)
{
  GuppiScatterStyle* d = SCM_TO_SCATTER_STYLE(x);

  SCM_DEFER_INTS;
  gtk_object_unref(GTK_OBJECT(d));
  SCM_ALLOW_INTS;

  return 0;
}

static int
print_data(SCM x, SCM port, scm_print_state* state)
{
  gchar buffer[128];
  guchar r,g,b,a;
  GuppiScatterStyle* ss = SCM_TO_SCATTER_STYLE(x);

  UINT_TO_RGBA(ss->color, &r, &g, &b, &a);

  g_snprintf(buffer, 128,
	     "<ScatterStyle r:%d g:%d b:%d a:%d>",r,g,b,a);
	     
  scm_puts(buffer, port);

  return 1;
}

/*****************************************************************************/

void
guppi_scm_scatter_style_init(void)
{
  static gboolean initialized = FALSE;
  static struct scm_smobfuns scatstyle_fns = {
    mark_data, free_data, print_data, NULL
  };

  g_return_if_fail(!initialized);
  initialized = TRUE;

  scatstyle_type_tag = scm_newsmob(&scatstyle_fns);

#include "guppi-scm-scatter-style.x"
  
}




/* $Id: guppi-scm-scatter-style.c,v 1.4 2000/01/18 21:06:00 trow Exp $ */
