/* This is -*- C -*- */
/* $Id: guppi-scatter-style.c,v 1.9 2000/01/21 17:07:47 trow Exp $ */

/*
 * guppi-scatter-style.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <ctype.h>
#include <math.h>
#include "guppi-scatter-style.h"
#include "rgb.h"
#ifdef BENCH
#include <sys/time.h>
#include <unistd.h>
#endif

const GuppiScatterMarkerInfo
guppi_scatter_marker_info[SCATTER_MARKER_LAST_MARKER] = {
  { SCATTER_MARKER_NONE, N_("None"),
    NULL, 0,0,0,
    NULL, 0,0,0 },
  { SCATTER_MARKER_CIRCLE, N_("Circle"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_DIAMOND, N_("Diamond"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_CROSS, N_("Cross"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_X, N_("X"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_SQUARE, N_("Square"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_AST, N_("Asterisk"),
    N_("Radius"), 1, 15, 4,
    N_("Thickness"), 0.5, 12, 1 },
  { SCATTER_MARKER_FILLED_CIRCLE, N_("Filled Circle"),
    N_("Radius"), 1, 15, 4,
    NULL, 0, 0, 0 },
  { SCATTER_MARKER_FILLED_SQUARE, N_("Filled Square"),
    N_("Radius"), 1, 15, 4,
    NULL, 0, 0, 0 },
  { SCATTER_MARKER_FILLED_DIAMOND, N_("Filled Diamond"),
    N_("Radius"), 1, 15, 4,
    NULL, 0.5, 12, 1 },
  { SCATTER_MARKER_ALPHA_TEMPLATE, NULL,
    NULL, -1, -1, -1,
    NULL, -1, -1, -1 }
};

static gboolean 
loose_match(const gchar* a, const gchar* b)
{
  gchar aa, bb;
  while (*a != '\0' && *b != '\0') {
    aa = *a;
    bb = *b;

    if (isupper(aa)) aa = tolower(aa);
    if (isupper(bb)) bb = tolower(bb);

    if (isspace(aa))
      ++a;
    else if (isspace(bb))
      ++b;
    else if (aa != bb)
      return FALSE;
    else {
      if (*a) ++a;
      if (*b) ++b;
    }
  }
  return TRUE;
}

GuppiScatterMarker
string2scatter_marker(const gchar* s)
{
  gint i=0;
  while (guppi_scatter_marker_info[i].name != NULL) {
    if (loose_match(guppi_scatter_marker_info[i].name, s)) {
      return guppi_scatter_marker_info[i].marker;
    }
    ++i;
  }
  return SCATTER_MARKER_NONE;
}

AlphaTemplate*
guppi_scatter_marker_make_alpha_template(GuppiScatterMarker m,
					 double sz1, double sz2,
					 double scale_factor)
{
  g_return_val_if_fail(sz1 >= 0, NULL);
  g_return_val_if_fail(sz2 >= 0, NULL);
  g_return_val_if_fail(scale_factor > 0, NULL);

  sz1 *= scale_factor;
  sz2 *= scale_factor;

  switch (m) {
  case SCATTER_MARKER_NONE:
  case SCATTER_MARKER_ALPHA_TEMPLATE:
    return NULL;
  case SCATTER_MARKER_CIRCLE:
    return alpha_template_new_ring(sz1, sz2);
  case SCATTER_MARKER_SQUARE:
    return alpha_template_new_rectangle(sz1, sz2, 0);
  case SCATTER_MARKER_DIAMOND:
    return alpha_template_new_rectangle(sz1, sz2, M_PI/4);
  case SCATTER_MARKER_CROSS:
    return alpha_template_new_cross(sz1, sz2, 0);
  case SCATTER_MARKER_X:
    return alpha_template_new_cross(sz1, sz2, M_PI/4);
  case SCATTER_MARKER_AST:
    return alpha_template_new_ast(sz1, sz2, 0);
  case SCATTER_MARKER_FILLED_CIRCLE:
    return alpha_template_new_circle(sz1);
  case SCATTER_MARKER_FILLED_SQUARE:
    return alpha_template_new_box(sz1, 0);
  case SCATTER_MARKER_FILLED_DIAMOND:
    return alpha_template_new_box(sz1, M_PI/4);
  default:
    g_assert_not_reached();
    return NULL;
  }
}

///////////////////////////////////////////////////////////////////////////////


static GtkObjectClass* parent_class = NULL;

enum {
  CHANGED,
  LAST_SIGNAL
};

static guint guppi_scatter_style_signals[LAST_SIGNAL] = { 0 };

enum {
  ARG_0
};

static void
guppi_scatter_style_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_scatter_style_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_scatter_style_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_scatter_style_finalize(GtkObject* obj)
{
  GuppiScatterStyle* sty = GUPPI_SCATTER_STYLE(obj);

  alpha_template_free(sty->alpha_template);
  
  if (sty->gradient_data)
    gtk_object_unref(GTK_OBJECT(sty->gradient_data));

  if (sty->size1_data)
    gtk_object_unref(GTK_OBJECT(sty->size1_data));

  if (sty->size2_data)
    gtk_object_unref(GTK_OBJECT(sty->size2_data));

  alpha_template_free(sty->last_alpha_template);

  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_scatter_style_class_init(GuppiScatterStyleClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  guppi_scatter_style_signals[CHANGED] =
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiScatterStyleClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class,
			       guppi_scatter_style_signals,
			       LAST_SIGNAL);

  object_class->get_arg = guppi_scatter_style_get_arg;
  object_class->set_arg = guppi_scatter_style_set_arg;
  object_class->destroy = guppi_scatter_style_destroy;
  object_class->finalize = guppi_scatter_style_finalize;

}

static void
guppi_scatter_style_init(GuppiScatterStyle* obj)
{
  /* Some not-unreasonable set of default values */

  obj->marker = SCATTER_MARKER_CIRCLE;
  obj->size1 = guppi_scatter_marker_info[obj->marker].size1_default;
  obj->size2 = guppi_scatter_marker_info[obj->marker].size2_default;

  obj->alpha_template = NULL;
  
  obj->color = 0;
  
  obj->gradient_data = NULL;
  obj->reverse_color_gradient = FALSE;
  obj->color_low = 0;

  obj->reverse_size1_gradient = FALSE;
  obj->reverse_size2_gradient = FALSE;

  obj->freeze_count = 0;
  obj->pending_change = FALSE;

  obj->last_alpha_template = NULL;
  obj->last_scale = -1;
}

GtkType
guppi_scatter_style_get_type(void)
{
  static GtkType guppi_scatter_style_type = 0;
  if (!guppi_scatter_style_type) {
    static const GtkTypeInfo guppi_scatter_style_info = {
      "GuppiScatterStyle",
      sizeof(GuppiScatterStyle),
      sizeof(GuppiScatterStyleClass),
      (GtkClassInitFunc)guppi_scatter_style_class_init,
      (GtkObjectInitFunc)guppi_scatter_style_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_scatter_style_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_scatter_style_info);
  }
  return guppi_scatter_style_type;
}

GuppiScatterStyle*
guppi_scatter_style_new(void)
{
  return GUPPI_SCATTER_STYLE(gtk_type_new(guppi_scatter_style_get_type()));
}

GuppiScatterStyle*
guppi_scatter_style_new_stock(gint i)
{
  /* These are RGB values - assume an alpha of 0xff, which we tack on later */
  /* 0x000001 is the code for the final entry */
  static const guint32 color_sequence[] = {
    0x000000, 0xff0000, 0x40b040, 0x0000ff, 0xffff00, 0xff00ff, 0x00ffff,
    0xff8080, 0x80ff80, 0x8080ff, 0x808080,
    0x000001 };
  static gint color_sequence_length = 0;

  GuppiScatterStyle* sty;

  g_return_val_if_fail(i>=0, NULL);

  if (color_sequence_length == 0) {

    while (color_sequence[color_sequence_length] != 0x000001)
      ++color_sequence_length;

    if (color_sequence_length == 0)
      g_error("Something really strange has happened!");
  }

  sty = guppi_scatter_style_new();
  
  sty->marker = (i % (gint)(SCATTER_MARKER_LAST_MARKER-1))+1;

  /* Our stock styles always have the default sizes (at least for now) */
  sty->size1 = guppi_scatter_marker_info[sty->marker].size1_default;
  sty->size2 = guppi_scatter_marker_info[sty->marker].size2_default;

  sty->color = (color_sequence[i % color_sequence_length] << 8) | 0xff;
  if (i == 0)
    sty->color_low = 0x2020c0ff;
  
  return sty;
}

static void
emit_changed_signal(GuppiScatterStyle* sty)
{
  if (sty->freeze_count > 0) {
    sty->pending_change = TRUE;
  } else {
    gtk_signal_emit(GTK_OBJECT(sty), guppi_scatter_style_signals[CHANGED]);
  }
}

void
guppi_scatter_style_freeze(GuppiScatterStyle* sty)
{
  g_return_if_fail(sty != NULL);
  ++sty->freeze_count;
}

void
guppi_scatter_style_thaw(GuppiScatterStyle* sty)
{
  g_return_if_fail(sty != NULL);
  g_return_if_fail(sty->freeze_count > 0);

  --sty->freeze_count;
  if (sty->freeze_count == 0) {
    if (sty->pending_change)
      emit_changed_signal(sty);
    sty->pending_change = FALSE;
  }
}

GuppiScatterMarker
guppi_scatter_style_marker(const GuppiScatterStyle* sty)
{
  g_return_val_if_fail(sty != NULL, SCATTER_MARKER_NONE);
  return sty->marker;
}

void
guppi_scatter_style_set_marker(GuppiScatterStyle* sty, GuppiScatterMarker m)
{
  g_return_if_fail(sty != NULL);

  if (sty->marker != m) {

    sty->marker = m;
    sty->size1 = CLAMP(sty->size1,
		       guppi_scatter_marker_info[m].size1_min,
		       guppi_scatter_marker_info[m].size1_max);
    sty->size2 = CLAMP(sty->size2,
		       guppi_scatter_marker_info[m].size2_min,
		       guppi_scatter_marker_info[m].size2_max);

    /* Remove our cached alpha template */
    if (sty->last_alpha_template != NULL) {
      alpha_template_free(sty->last_alpha_template);
      sty->last_alpha_template = NULL;
    }

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_marker_size1(GuppiScatterStyle* sty, double x)
{
  g_return_if_fail(sty != NULL);
  
  if (guppi_scatter_marker_info[sty->marker].size1_desc == NULL)
    return;
  
  x = CLAMP(x, 
	    guppi_scatter_marker_info[sty->marker].size1_min,
	    guppi_scatter_marker_info[sty->marker].size1_max);
  if (x != sty->size1) {
    sty->size1 = x;

    /* Remove our cached alpha template */
    if (sty->last_alpha_template != NULL) {
      alpha_template_free(sty->last_alpha_template);
      sty->last_alpha_template = NULL;
    }

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_marker_size2(GuppiScatterStyle* sty, double x)
{
  g_return_if_fail(sty != NULL);
  
  if (guppi_scatter_marker_info[sty->marker].size2_desc == NULL)
    return;
  
  x = CLAMP(x, 
	    guppi_scatter_marker_info[sty->marker].size2_min,
	    guppi_scatter_marker_info[sty->marker].size2_max);
  if (x != sty->size2) {
    sty->size2 = x;

    /* Remove our cached alpha template */
    if (sty->last_alpha_template != NULL) {
      alpha_template_free(sty->last_alpha_template);
      sty->last_alpha_template = NULL;
    }

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_marker_sizes(GuppiScatterStyle* sty,
				     double s1, double s2)
{
  g_return_if_fail(sty != NULL);

  s1 = CLAMP(s1,
	     guppi_scatter_marker_info[sty->marker].size1_min,
	     guppi_scatter_marker_info[sty->marker].size1_max);
  s2 = CLAMP(s2,
	     guppi_scatter_marker_info[sty->marker].size2_min,
	     guppi_scatter_marker_info[sty->marker].size2_max);

  if (sty->size1 != s1 || sty->size2 != s2) {

    sty->size1 = s1;
    sty->size2 = s2;

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_marker_sizes_default(GuppiScatterStyle* sty)
{
  gint s1, s2;

  g_return_if_fail(sty != NULL);

  s1 = guppi_scatter_marker_info[sty->marker].size1_default;
  s2 = guppi_scatter_marker_info[sty->marker].size2_default;

  if (sty->size1 != s1 || sty->size2 != s2) {

    sty->size1 = s1;
    sty->size2 = s2;

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_alpha_template(GuppiScatterStyle* sty, AlphaTemplate* atemp)
{
  g_return_if_fail(sty != NULL);
  g_return_if_fail(atemp != NULL);

  g_free(sty->alpha_template);
  sty->alpha_template = atemp;

  /* automatically change marker as well */
  sty->marker = SCATTER_MARKER_ALPHA_TEMPLATE;

  emit_changed_signal(sty);
}

void
guppi_scatter_style_set_color(GuppiScatterStyle* sty, guint32 col)
{
  g_return_if_fail(sty != NULL);

  if (sty->color != col) {
    sty->color = col;
    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_gradient_data(GuppiScatterStyle* sty,
				      GuppiScalarData* data)
{
  g_return_if_fail(sty != NULL);

  if (data != sty->gradient_data) {

    if (sty->gradient_data != NULL)
      gtk_object_unref(GTK_OBJECT(sty->gradient_data));

    sty->gradient_data = data;

    if (sty->gradient_data != NULL)
      gtk_object_ref(GTK_OBJECT(sty->gradient_data));

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_color_low(GuppiScatterStyle* sty, guint32 col)
{
  g_return_if_fail(sty != NULL);

  if (sty->color_low != col) {
    sty->color_low = col;
    emit_changed_signal(sty);
  }
}

gboolean
guppi_scatter_style_reverse_color_gradient(const GuppiScatterStyle* sty)
{
  g_return_val_if_fail(sty != NULL, FALSE);
  return sty->reverse_color_gradient;
}

void
guppi_scatter_style_set_reverse_color_gradient(GuppiScatterStyle* sty,
					       gboolean x)
{
  g_return_if_fail(sty != NULL);
  if (sty->reverse_color_gradient != x) {
    sty->reverse_color_gradient = x;
    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_size1_data(GuppiScatterStyle* sty, GuppiScalarData* d)
{
  g_return_if_fail(sty != NULL);

  if (sty->size1_data != d) {

    if (sty->size1_data != NULL)
      gtk_object_unref(GTK_OBJECT(sty->size1_data));

    sty->size1_data = d;
    
    if (sty->size1_data != NULL)
      gtk_object_ref(GTK_OBJECT(sty->size1_data));

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_size1_high(GuppiScatterStyle* sty, double x)
{
  g_return_if_fail(sty != NULL);
  g_return_if_fail(x > 0);

  if (sty->size1_high != x) {
    sty->size1_high = x;
    emit_changed_signal(sty);
  }
}

gboolean
guppi_scatter_style_reverse_size1_gradient(const GuppiScatterStyle* sty)
{
  g_return_val_if_fail(sty != NULL, FALSE);
  return sty->reverse_size1_gradient;
}

void
guppi_scatter_style_set_reverse_size1_gradient(GuppiScatterStyle* sty,
					       gboolean x)
{
  g_return_if_fail(sty != NULL);
  if (sty->reverse_size1_gradient != x) {
    sty->reverse_size1_gradient = x;
    emit_changed_signal(sty);
  }
}


void
guppi_scatter_style_set_size2_data(GuppiScatterStyle* sty, GuppiScalarData* d)
{
  g_return_if_fail(sty != NULL);

  if (sty->size2_data != d) {

    if (sty->size2_data != NULL)
      gtk_object_unref(GTK_OBJECT(sty->size2_data));

    sty->size2_data = d;
    
    if (sty->size2_data != NULL)
      gtk_object_ref(GTK_OBJECT(sty->size2_data));

    emit_changed_signal(sty);
  }
}

void
guppi_scatter_style_set_size2_high(GuppiScatterStyle* sty, double x)
{
  g_return_if_fail(sty != NULL);
  g_return_if_fail(x > 0);

  if (sty->size2_high != x) {
    sty->size2_high = x;
    emit_changed_signal(sty);
  }
}

gboolean
guppi_scatter_style_reverse_size2_gradient(const GuppiScatterStyle* sty)
{
  g_return_val_if_fail(sty != NULL, FALSE);
  return sty->reverse_size2_gradient;
}

void
guppi_scatter_style_set_reverse_size2_gradient(GuppiScatterStyle* sty,
					       gboolean x)
{
  g_return_if_fail(sty != NULL);
  if (sty->reverse_size2_gradient != x) {
    sty->reverse_size2_gradient = x;
    emit_changed_signal(sty);
  }
}



/*****************************************************************************
 *
 * Scatter Painting Code
 *
 *****************************************************************************/

void
guppi_scatter_style_paint(GuppiScatterStyle* sty,
			  guchar* rgb_buffer, gint row_stride,
			  gint w, gint h, gint adj_x, gint adj_y,
			  const Transform* x_transform,
			  const Transform* y_transform,
			  const GuppiScalarData* x_data,
			  const GuppiScalarData* y_data,
			  const GuppiBooleanData* mask,
			  double scale_factor)
{
#ifdef BENCH
  static double t=0;
  struct timeval time0, time1;
  static guint point_count=0;
#endif
  gint r,g,b,a;

  gboolean do_gradient_coloring;
  gint rlo=0, glo=0, blo=0, alo=0;
  gint rnow, gnow, bnow, anow;
  double grad_min=0, grad_max=1, grad;

  gboolean do_size1_gradient, do_size2_gradient;
  double size1_grad_min=0, size1_grad_max=1, size1_now;
  double size2_grad_min=0, size2_grad_max=1, size2_now;

  AlphaTemplate* atemp = NULL;
  gboolean temporary_atemp = FALSE;
  
  gint x, y;
  gint xtweak=0, ytweak=0;
  dindex_t k, k_max;

  g_return_if_fail(sty != NULL);
  g_return_if_fail(rgb_buffer != NULL);
  g_return_if_fail(x_transform != NULL);
  g_return_if_fail(y_transform != NULL);

  if (x_data == NULL || y_data == NULL)
    return;

  /* g_message("painting %dx%d area at (%d,%d)",w,h,adj_x,adj_y);*/

  do_gradient_coloring = (sty->gradient_data != NULL);
  do_size1_gradient = (sty->size1_data != NULL);
  do_size2_gradient = (sty->size2_data != NULL);

  UINT_TO_RGBA(sty->color, &r, &g, &b, &a);
  rnow = r, gnow = g, bnow = b, anow = a;

  size1_now = sty->size1;
  size2_now = sty->size2;

  if (do_gradient_coloring) {
    UINT_TO_RGBA(sty->color_low, &rlo, &glo, &blo, &alo);
    grad_min = guppi_scalar_data_min(sty->gradient_data);
    grad_max = guppi_scalar_data_max(sty->gradient_data);
    if (grad_min == grad_max)
      do_gradient_coloring = FALSE;
  }
  
  if (do_size1_gradient) {
    size1_grad_min = guppi_scalar_data_min(sty->size1_data);
    size1_grad_max = guppi_scalar_data_max(sty->size1_data);

    /* A hack, for now */
    sty->size1_high = 5*sty->size1;

    if (size1_grad_min == size1_grad_max)
      do_size1_gradient = FALSE;
  }

  if (do_size2_gradient) {
    size2_grad_min = guppi_scalar_data_min(sty->size2_data);
    size2_grad_max = guppi_scalar_data_max(sty->size2_data);

    /* A hack, for now */
    sty->size2_high = 4*sty->size2;
    
    if (size2_grad_min == size2_grad_max)
      do_size2_gradient = FALSE;
  }

  if (sty->marker == SCATTER_MARKER_ALPHA_TEMPLATE ||
      sty->alpha_template != NULL) {
    atemp = sty->alpha_template;
    temporary_atemp = FALSE;
  } else if (sty->last_scale == scale_factor &&
	     sty->last_alpha_template != NULL) {
    atemp = sty->last_alpha_template;
    temporary_atemp = FALSE;
  } else if (!do_size1_gradient && !do_size2_gradient) {
    atemp = guppi_scatter_marker_make_alpha_template(sty->marker,
						     sty->size1, sty->size2,
						     scale_factor);
    alpha_template_free(sty->last_alpha_template);
    sty->last_alpha_template = atemp;
    sty->last_scale = scale_factor;
    temporary_atemp = FALSE;
  }

  /*** Our actual drawing loop ***/

#ifdef BENCH
  gettimeofday(&time0, NULL);
#endif
    
  if (mask != NULL) 
    k = guppi_boolean_data_first_true(mask);
  else
    k = guppi_data_min_index(GUPPI_DATA(x_data));

  k_max = guppi_data_max_index(mask ? GUPPI_DATA(mask) : GUPPI_DATA(x_data));

  while (k <= k_max) {

    x = transform_plot2pixel_unsafe(x_transform,
				    guppi_scalar_data_get(x_data, k));

    y = transform_plot2pixel_unsafe(y_transform,
				    guppi_scalar_data_get(y_data, k));

    if (do_gradient_coloring) {
      grad = guppi_scalar_data_get_unsafe(sty->gradient_data, k);
      grad = (grad-grad_min)/(grad_max-grad_min);
      if (sty->reverse_color_gradient)
	grad = 1 - grad;

      rnow = (gint)(grad*rlo+(1-grad)*r);
      gnow = (gint)(grad*glo+(1-grad)*g);
      bnow = (gint)(grad*blo+(1-grad)*b);
      anow = (gint)(grad*alo+(1-grad)*a);
    }

    if (do_size1_gradient) {
      size1_now = guppi_scalar_data_get_unsafe(sty->size1_data, k);
      size1_now = (size1_now - size1_grad_min)/(size1_grad_max-size1_grad_min);
      if (sty->reverse_size1_gradient)
	size1_now = 1 - size1_now;
      size1_now = size1_now*sty->size1_high + (1-size1_now)*sty->size1;
    }

    if (do_size2_gradient) {
      size2_now = guppi_scalar_data_get_unsafe(sty->size2_data, k);
      size2_now = (size2_now - size2_grad_min)/(size2_grad_max-size2_grad_min);
      if (sty->reverse_size2_gradient)
	size2_now = 1 - size2_now;
      size2_now = size2_now*sty->size2_high + (1-size2_now)*sty->size2;
    }

    if (do_size1_gradient || do_size2_gradient) {
      /* Right now size-gradients don't scale well, since we need to
	 rebuild an alpha template for every point.

	 If we are just doing one size gradient, the right thing to do
	 is to buffer N templates and then just use the one that is
	 closest to size*_now.  N could be chosen so as to minimize the
	 maximum possible error between size*_now and the size* of the
	 buffered template.
	 
	 We could do something like this for two size gradients,
	 buffering NxM templates.  That just seems like it has the
	 potential to get annoying.  But in reality, it probably wouldn't
	 be any more annoying that the 1d case.
      */

      if (temporary_atemp)
	alpha_template_free(atemp);
      atemp = guppi_scatter_marker_make_alpha_template(sty->marker,
						       size1_now,
						       size2_now,
						       scale_factor);
      temporary_atemp = TRUE;
    }

    if (atemp) {
      xtweak = atemp->width/2;
      ytweak = atemp->height/2;
    }

    x -= xtweak;
    y -= ytweak;

    if (atemp) {

      /* This is cut&pasted from alpha_template_print(), so we don't have
	 any function-call overhead.  This should speed things up a little,
	 maybe a lot. */

      x -= adj_x;
      y -= adj_y;

      if (x + atemp->width >= 0 && x < w &&
	  y + atemp->height >= 0 && y < h) {

	gint atpr_x0, atpr_y0, atpr_x1, atpr_y1;
	gint atpr_i, atpr_j;
	guchar* atpr_ap;
	guchar* atpr_arun;
	guchar* atpr_rgbp;
	gchar* atpr_rgbrun;

	/* Clip to our viewing area */
	atpr_x0 = MAX(x, 0);
	atpr_y0 = MAX(y, 0);
	atpr_x1 = MIN(x + atemp->width - 1, w - 1);
	atpr_y1 = MIN(y + atemp->height - 1, h - 1);
	
	atpr_ap = atemp->data + atemp->width * (atpr_y0 - y) + (atpr_x0 - x);
	atpr_rgbp = rgb_buffer + row_stride * atpr_y0 + 3 * atpr_x0;
	for (atpr_j=atpr_y0; atpr_j<=atpr_y1; ++atpr_j) {
	  atpr_arun = atpr_ap;
	  atpr_rgbrun = atpr_rgbp;
	  for (atpr_i=atpr_x0; atpr_i<=atpr_x1; ++atpr_i) {
	    if (*atpr_arun > 0) {
	      PIXEL_RGBA(atpr_rgbrun, rnow, gnow, bnow,
			 (((1+anow) * (1+*atpr_arun)) >> 8));
	    }
	    ++atpr_arun;
	    atpr_rgbrun += 3;
	  }
	  atpr_ap += atemp->width;
	  atpr_rgbp += row_stride;
	}
      }
    } else {
      /* do nothing */
    }

    if (mask)
      k = guppi_boolean_data_next_true(mask, k);
    else
      ++k;

#ifdef BENCH
    ++point_count;
#endif

  }

#ifdef BENCH
  gettimeofday(&time1, NULL);
  t += time1.tv_sec - time0.tv_sec + (time1.tv_usec - time0.tv_usec)/1e+6;
  g_message("time=%f pts=%d pts/sec=%f", t, point_count, point_count/t);
#endif


  /*** Clean-up ***/

  if (temporary_atemp)
    alpha_template_free(atemp);
}
									      
				  

/* $Id: guppi-scatter-style.c,v 1.9 2000/01/21 17:07:47 trow Exp $ */
