/* $Id: guppi-data.c,v 1.3 2000/01/05 02:07:25 trow Exp $ */

/*
 * guppi-data.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and 
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-data.h"
#include <gtk/gtkmarshal.h>

enum {
  CHANGED_ONE,
  CHANGED_ADD,
  CHANGED_MANY,
  CHANGED,
  CHANGED_LABEL,
  CHANGED_DATASET,
  CHANGED_DATASET_CONTENT,
  LAST_SIGNAL
};

static GtkObjectClass* parent_class = NULL;
static guint data_signals[LAST_SIGNAL] = { 0 };

static void
guppi_data_finalize(GtkObject* obj)
{
  GuppiData* data = GUPPI_DATA(obj);
  if (data->dataset)
    gtk_object_unref(GTK_OBJECT(data->dataset));

  if (parent_class->finalize)
    (*parent_class->finalize)(obj);
}

static void
guppi_data_class_init(GuppiDataClass* klass)
{
  GtkObjectClass* object_class;

  object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  data_signals[CHANGED_ONE] = 
    gtk_signal_new("changed_one",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_one),
		   gtk_marshal_NONE__INT_POINTER,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_INT,
		   GTK_TYPE_POINTER);

  data_signals[CHANGED_ADD] = 
    gtk_signal_new("changed_add",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_add),
		   gtk_marshal_NONE__INT_POINTER,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_INT,
		   GTK_TYPE_POINTER);

  data_signals[CHANGED_MANY] = 
    gtk_signal_new("changed_many",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_many),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED] = 
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED_LABEL] = 
    gtk_signal_new("changed_label",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_label),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED_DATASET] = 
    gtk_signal_new("changed_dataset",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_dataset),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED_DATASET_CONTENT] = 
    gtk_signal_new("changed_dataset_content",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_dataset_content),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, data_signals, LAST_SIGNAL);

  klass->type_name = "<GuppiData>";

  klass->validate = NULL;
  klass->get = NULL;
  klass->set = NULL;
  klass->add = NULL;
  klass->del = NULL;
  klass->copy = NULL;

  klass->changed_one = NULL;
  klass->changed = NULL;
  klass->changed_label = NULL;
  klass->changed_dataset = NULL;
  klass->changed_dataset_content = NULL;

  object_class->finalize = guppi_data_finalize;
}

static void
guppi_data_init(GuppiData* data)
{
  static gint number = 1;

  data->min_index = 0;
  data->max_index = -1;

  data->label = g_strdup_printf("Unlabelled #%d", number);

  data->dataset = NULL;

  ++number;
}

GtkType
guppi_data_get_type(void)
{
  static GtkType data_type = 0;

  if (!data_type) {
    static const GtkTypeInfo data_info = {
      "GuppiData",
      sizeof(GuppiData),
      sizeof(GuppiDataClass),
      (GtkClassInitFunc)guppi_data_class_init,
      (GtkObjectInitFunc)guppi_data_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    data_type = gtk_type_unique(GTK_TYPE_OBJECT, &data_info);
  }
  
  return data_type;
}

dindex_t
guppi_data_min_index(const GuppiData* data)
{
  g_return_val_if_fail(data, 0);
  return data->min_index;
}

dindex_t
guppi_data_max_index(const GuppiData* data)
{
  g_return_val_if_fail(data, 0);
  return data->max_index;
}

gsize
guppi_data_size(const GuppiData* data)
{
  g_return_val_if_fail(data, 0);
  return data->max_index - data->min_index + 1;
}

gboolean
guppi_data_in_bounds(const GuppiData* data, dindex_t i)
{
  g_return_val_if_fail(data != NULL, FALSE);

  return data->min_index <= i && i <= data->max_index;
}

gboolean
guppi_data_contains_bounds(const GuppiData* data, const GuppiData* q)
{
  g_return_val_if_fail(data != NULL, FALSE);
  g_return_val_if_fail(q != NULL, FALSE);

  return data->min_index <= q->min_index && q->max_index <= data->max_index;
}

gboolean
guppi_data_equal_bounds(const GuppiData* a, const GuppiData* b)
{
  g_return_val_if_fail(a != NULL, FALSE);
  g_return_val_if_fail(b != NULL, FALSE);

  return a->min_index == b->min_index && a->max_index == b->max_index;
}

void
guppi_data_shift_indices(GuppiData* data, gint k)
{
  g_return_if_fail(data);

  data->min_index += k;
  data->max_index += k;

  if (k)
    guppi_data_touch(data);
}

/****************************************************************************/

const gchar*
guppi_data_label(const GuppiData* data)
{
  g_return_val_if_fail(data, NULL);

  return data->label;
}

void
guppi_data_set_label(GuppiData* data, const gchar* new_label)
{
  g_return_if_fail(data);

  g_free(data->label);
  data->label = new_label ? g_strdup(new_label) : NULL;
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_LABEL], NULL);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED], NULL);
}

GuppiDataset*
guppi_data_dataset(const GuppiData* data)
{
  g_return_val_if_fail(data, NULL);

  return data->dataset;
}

static void
converter_cb(GuppiDataset* ds, GuppiData* data)
{
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_DATASET_CONTENT]);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED]);
}

void
guppi_data_set_dataset(GuppiData* data, GuppiDataset* ds)
{
  g_return_if_fail(data != NULL);
  if (ds != data->dataset) {

    if (data->dataset != NULL) {
      gtk_object_unref(GTK_OBJECT(data->dataset));
      gtk_signal_disconnect_by_data(GTK_OBJECT(data->dataset), data);
    }

    data->dataset = ds;

    if (data->dataset != NULL) {
      gtk_object_ref(GTK_OBJECT(data->dataset));
      gtk_signal_connect(GTK_OBJECT(data->dataset),
			 "changed",
			 GTK_SIGNAL_FUNC(converter_cb),
			 data);
    }

    gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_DATASET], NULL);
    gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED], NULL);
  }
}

const gchar*
guppi_data_dataset_name(const GuppiData* data)
{
  GuppiDataset* ds;

  g_return_val_if_fail(data != NULL, NULL);

  ds = guppi_data_dataset(data);
  g_return_val_if_fail(ds != NULL, NULL);

  return guppi_dataset_name(ds);
}

/****************************************************************************/

const gchar*
guppi_data_type_name(const GuppiData* data)
{
  g_return_val_if_fail(data, NULL);

  return GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->type_name;
}

const gchar*
guppi_data_type_name_by_type(GtkType type)
{
  GuppiDataClass* klass;

  /* Conversion type must be derived from GuppiData, but must not
     actually be GuppiData. */
  g_return_val_if_fail(gtk_type_is_a(type, GUPPI_TYPE_DATA), NULL);
  g_return_val_if_fail(type != GUPPI_TYPE_DATA, NULL);

  klass = GUPPI_DATA_CLASS(gtk_type_class(type));
  g_return_val_if_fail(klass != NULL, NULL);

  return klass->type_name;
}

gboolean
guppi_data_validate(const GuppiData* data, const gchar* sbuf,
		    gchar* errbuf, gsize errbuf_len)
{
  gboolean (*fn)(const GuppiData*, const gchar*, gchar*, gsize);

  g_return_val_if_fail(data, FALSE);
  g_return_val_if_fail(sbuf, FALSE);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->validate;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_validate() is not defined for type %s", guppi_data_type_name(data));
    return FALSE;
  }

  return fn(data,sbuf,errbuf,errbuf_len);
}

gboolean
guppi_data_validate_by_type(GtkType type, const gchar* sbuf,
			    gchar* errbuf, gsize errbuf_len)
{
  GuppiDataClass* klass;

  /* Conversion type must be derived from GuppiData, but must not
     actually be GuppiData. */
  g_return_val_if_fail(gtk_type_is_a(type, GUPPI_TYPE_DATA), 0);
  g_return_val_if_fail(type != GUPPI_TYPE_DATA, 0);

  klass = GUPPI_DATA_CLASS(gtk_type_class(type));
  g_return_val_if_fail(klass != NULL, 0);
  g_return_val_if_fail(klass->validate != NULL, 0);

  return (klass->validate)(NULL, sbuf, errbuf, errbuf_len);
}

void
guppi_data_get(const GuppiData* data, dindex_t i,
	       gchar* sbuf, gsize max_sbuf_length)
{
  void (*fn)(const GuppiData*,dindex_t,gchar*,gsize);

  g_return_if_fail(data);
  g_return_if_fail(sbuf);
  g_return_if_fail(i >= data->min_index);
  g_return_if_fail(i <= data->max_index);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->get;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_get() is not defined for type %s", guppi_data_type_name(data));
  } else
    (*fn)(data,i,sbuf,max_sbuf_length);
}

void
guppi_data_set(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  void (*fn)(GuppiData*,dindex_t,const gchar*);

  g_return_if_fail(data);
  g_return_if_fail(sbuf);
  g_return_if_fail(i >= data->min_index);
  g_return_if_fail(i <= data->max_index);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->set;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_set() is not defined for type %s", guppi_data_type_name(data));
  } else
    (*fn)(data,i,sbuf);
}

void
guppi_data_add(GuppiData* data, const gchar* sbuf)
{
  void (*fn)(GuppiData*,const gchar*);

  g_return_if_fail(data);
  g_return_if_fail(sbuf);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->add;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_add() is not defined for type %s", guppi_data_type_name(data));
  } else
    (*fn)(data,sbuf);
}

void
guppi_data_insert(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  void (*fn)(GuppiData*,dindex_t i, const gchar*);

  g_return_if_fail(data);
  g_return_if_fail(sbuf);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->insert;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_insert() is not defined for type %s", guppi_data_type_name(data));
  } else
    (*fn)(data,i,sbuf);
}


void
guppi_data_delete(GuppiData* data, dindex_t i)
{
  void (*fn)(GuppiData*,dindex_t i);

  g_return_if_fail(data);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->del;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_delete() is not defined for type %s", guppi_data_type_name(data));
  } else
    (*fn)(data,i);
}

GuppiData*
guppi_data_copy(const GuppiData* data)
{
  GuppiData* (*fn)(const GuppiData*);
  GuppiData* cpy;
  gchar* lbl;

  g_return_val_if_fail(data, NULL);

  fn = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass)->copy;

  if (fn == NULL) {
    g_warning("The virtual function guppi_data_copy() is not defined for type %s", guppi_data_type_name(data));
    return NULL;
  } 
    
  cpy = (*fn)(data);

  lbl = g_strdup_printf("Copy of %s", guppi_data_label(data));
  guppi_data_set_label(cpy, lbl);
  g_free(lbl);

  guppi_data_set_dataset(cpy, guppi_data_dataset(data));
  
  return cpy;
}

void
guppi_data_dump(const GuppiData* data, FILE* out)
{
  dindex_t i;
  gchar buffer[256];

  for(i=data->min_index; i<=data->max_index; ++i) {
    guppi_data_get(data, i, buffer, 255);
    fprintf(out, "%5d: %s\n", i, buffer);
  }
} 

/****************************************************************************/

double
guppi_data_conversion_potential(const GuppiData* data, GtkType type)
{
  GuppiDataClass* klass;
  const gint buffer_len = 1024;
  gchar buffer[1024];
  dindex_t i, i0, i1;
  gint valid_count;

  g_return_val_if_fail(data != NULL, 0);
  g_return_val_if_fail(guppi_data_size(data) > 0, 0);

  /* Conversion type must be derived from GuppiData, but must not
     actually be GuppiData. */
  g_return_val_if_fail(gtk_type_is_a(type, GUPPI_TYPE_DATA), 0);
  g_return_val_if_fail(type != GUPPI_TYPE_DATA, 0);

  klass = GUPPI_DATA_CLASS(gtk_type_class(type));
  g_return_val_if_fail(klass != NULL, 0);
  g_return_val_if_fail(klass->validate != NULL, 0);

  i0 = guppi_data_min_index(data);
  i1 = guppi_data_max_index(data);
  valid_count = 0;

  for(i=i0; i<=i1; ++i) {
    guppi_data_get(data, i, buffer, buffer_len);
    if ( (klass->validate)(NULL, buffer, NULL, 0) ) 
      ++valid_count;
  }

  return valid_count / (double)guppi_data_size(data);
}

GuppiData*
guppi_data_convert(const GuppiData* data,
		   GtkType type,
		   const gchar* replace_invalid)
{
  const gint buffer_len = 1024;
  gchar buffer[1024];
  const gint errbuffer_len = 256;
  gchar errbuffer[256];

  GuppiData* conv;
  dindex_t i, i0, i1;
  gboolean valid;
  gboolean checked_replace_invalid = FALSE;

  g_return_val_if_fail(data != NULL, NULL);
  g_return_val_if_fail(gtk_type_is_a(type, GUPPI_TYPE_DATA), NULL);
  g_return_val_if_fail(type != GUPPI_TYPE_DATA, NULL);

  /* If the object is already of the desired type, just return a copy
     of the original. */
  if (GTK_CHECK_TYPE(GTK_OBJECT(data), type)) {
    return guppi_data_copy(data);
  }

  conv = GUPPI_DATA(gtk_type_new(type));
  g_assert(conv != NULL);

  i0 = guppi_data_min_index(data);
  i1 = guppi_data_max_index(data);

  for(i=i0; i<=i1; ++i) {

    guppi_data_get(data, i, buffer, buffer_len);

    valid = guppi_data_validate(conv, buffer, errbuffer, errbuffer_len);

    if (valid) {

      guppi_data_add(conv, buffer);
      
    } else if (replace_invalid != NULL) {

	valid = TRUE;
	if (!checked_replace_invalid) {
	  valid = guppi_data_validate(conv, replace_invalid, 
				      errbuffer, errbuffer_len);
	  if (valid)
	    checked_replace_invalid = TRUE;
	}

	if (valid) 
	  guppi_data_add(conv, replace_invalid);
	else
	  g_warning("Replacement item \"%s\" is invalid: %s",
		    replace_invalid, errbuffer);

    } else {

      g_warning("Item \"%s\" is invalid: %s", buffer, errbuffer);
      
    }
  }

  /* Align our indices. */
  guppi_data_shift_indices(conv, i0);

  /* Copy the label. */
  guppi_data_set_label(conv, guppi_data_label(data));

  /* Make the associated datasets match. */
  guppi_data_set_dataset(conv, guppi_data_dataset(data));

  return conv;
}

/****************************************************************************/

void
guppi_data_touch_one(GuppiData* data, dindex_t i, gpointer old_val)
{
  g_return_if_fail(data);

  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_ONE], i, old_val);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED]);
}

void
guppi_data_touch_add(GuppiData* data, dindex_t i, gpointer new_val)
{
  g_return_if_fail(data);

  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_ADD], i, new_val);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED]);
}

void
guppi_data_touch(GuppiData* data)
{
  g_return_if_fail(data);

  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_MANY]);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED]);
}

/****************************************************************************/

void
guppi_data_add_from_stream(GuppiData* data, FILE* in)
{
  gchar buffer[2048], err_msg[256];
  int line_no = 1;

  g_return_if_fail(data);
  g_return_if_fail(in);

  while (fgets(buffer,2048,in)) {
    g_strchomp(buffer);
    if (!guppi_data_validate(data, buffer, err_msg, 256)) {
      g_warning("line %d: %s", line_no, err_msg);
    } else {
      guppi_data_add(data, buffer);
    }
    ++line_no;
  }
}


void
guppi_data_add_from_file(GuppiData* data, const gchar* filename)
{
  FILE* in;

  g_return_if_fail(data);
  g_return_if_fail(filename);

  in = fopen(filename, "r");
  if (!in) {
    g_warning("Couldn't open file \"%s\"", filename);
    return;
  }

  guppi_data_add_from_stream(data, in);

  fclose(in);
}


/* $Id: guppi-data.c,v 1.3 2000/01/05 02:07:25 trow Exp $ */
