/* This is -*- C -*- */
/* $Id: guppi-categorical-data.c,v 1.2 1999/12/07 17:06:11 trow Exp $ */

/*
 * guppi-categorical-data.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include "guppi-categorical-data.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_categorical_data_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_categorical_data_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_categorical_data_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_categorical_data_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

/****************************************************************************/

static gboolean
guppi_categorical_data_string_validate(const GuppiData* d, const gchar* sbuf,
				       gchar* errstr, gsize errstr_len)
{
  const GuppiCategoricalData* cd;

  if (d == NULL)
    return TRUE;

  cd = GUPPI_CATEGORICAL_DATA(d);

  /* Anything goes if our category hasn't been frozen yet. */
  if (!cd->frozen)
    return TRUE;

  if (cd->category != NULL && guppi_string_data_contains(cd->category, sbuf))
    return TRUE;

  if (errstr)
    g_snprintf(errstr, errstr_len, "Unknown category \"%s\"", sbuf);

  return FALSE;
}

static void
guppi_categorical_data_string_get(const GuppiData* d, dindex_t i,
				  gchar* sbuf, gsize maxlen)
{
  const GuppiCategoricalData* cd = GUPPI_CATEGORICAL_DATA(d);

  g_snprintf(sbuf, maxlen, "%s", guppi_categorical_data_get(cd, i));
}

static void
guppi_categorical_data_string_set(GuppiData* d, dindex_t i, const gchar* sbuf)
{
  guppi_categorical_data_set(GUPPI_CATEGORICAL_DATA(d), i, sbuf);
}

static void
guppi_categorical_data_string_add(GuppiData* d, const gchar* sbuf)
{
  guppi_categorical_data_add(GUPPI_CATEGORICAL_DATA(d), sbuf);
}

static void
guppi_categorical_data_string_insert(GuppiData* d, dindex_t i,
				     const gchar* sbuf)
{
  guppi_categorical_data_insert(GUPPI_CATEGORICAL_DATA(d), i, sbuf);
}

static void
guppi_categorical_data_string_delete(GuppiData* d, dindex_t i)
{
  guppi_categorical_data_delete(GUPPI_CATEGORICAL_DATA(d), i);
}

static GuppiData*
guppi_categorical_data_copy(const GuppiData* d)
{
  g_error("Not implemented");
  return NULL;
}				 

/****************************************************************************/

static void
guppi_categorical_data_class_init(GuppiCategoricalDataClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  klass->parent_class.type_name = _("categorical");

  klass->parent_class.validate = guppi_categorical_data_string_validate;
  klass->parent_class.get = guppi_categorical_data_string_get;
  klass->parent_class.set = guppi_categorical_data_string_set;  
  klass->parent_class.add = guppi_categorical_data_string_add;
  klass->parent_class.insert = guppi_categorical_data_string_insert;
  klass->parent_class.del = guppi_categorical_data_string_delete;
  klass->parent_class.copy = guppi_categorical_data_copy;

  object_class->get_arg = guppi_categorical_data_get_arg;
  object_class->set_arg = guppi_categorical_data_set_arg;
  object_class->destroy = guppi_categorical_data_destroy;
  object_class->finalize = guppi_categorical_data_finalize;

}

static void
guppi_categorical_data_init(GuppiCategoricalData* obj)
{
  obj->frozen = FALSE;

  obj->category = GUPPI_STRING_DATA(guppi_string_data_new());
  
  obj->catsize = 0;
  obj->data = NULL;
  obj->intsize = 0;
  obj->size = 0;
  obj->poolsize = 0;
}

GtkType
guppi_categorical_data_get_type(void)
{
  static GtkType guppi_categorical_data_type = 0;
  if (!guppi_categorical_data_type) {
    static const GtkTypeInfo guppi_categorical_data_info = {
      "GuppiCategoricalData",
      sizeof(GuppiCategoricalData),
      sizeof(GuppiCategoricalDataClass),
      (GtkClassInitFunc)guppi_categorical_data_class_init,
      (GtkObjectInitFunc)guppi_categorical_data_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_categorical_data_type = gtk_type_unique(GUPPI_TYPE_DATA, &guppi_categorical_data_info);
  }
  return guppi_categorical_data_type;
}

GuppiData*
guppi_categorical_data_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_categorical_data_get_type()));
}

gboolean
guppi_categorical_data_frozen(const GuppiCategoricalData* cd)
{
  g_return_val_if_fail(cd != NULL, TRUE);
  return cd->frozen;
}

void
guppi_categorical_data_set_freeze(GuppiCategoricalData* cd, gboolean x)
{
  g_return_if_fail(cd != NULL);
  cd->frozen = x;
}

/*****************************************************************************/

static void
guppi_categorical_data_grow_catsize(GuppiCategoricalData* cd,
				    guint newcatsize)
{
  gint newintsize;
  gpointer newdata;
  gint i;
  guint cpy = 0;
  const guint catsize_nag_limit = 1000000;

  g_return_if_fail(cd != NULL);

  if (newcatsize <= cd->catsize)
    return;


  if (newcatsize >= catsize_nag_limit)
    g_warning("Expanding catsize of %s to %d",
	      guppi_data_label(GUPPI_DATA(cd)), newcatsize);

  if (newcatsize <= 256)
    newintsize = 1;
  else if (newcatsize <= 65536)
    newintsize = 2;
  else 
    newintsize = 4;

  if (newintsize != cd->intsize) {

    newdata = g_malloc(cd->poolsize * newintsize);

    for (i=0; i<cd->size; ++i) {

      if (cd->intsize == 1) 
	cpy = ((guint8*)cd->data)[i];
      else if (cd->intsize == 2)
	cpy = ((guint16*)cd->data)[i];
      else if (cd->intsize == 4)
	cpy = ((guint32*)cd->data)[i];
      else
	g_assert_not_reached();

      if (newintsize == 1)
	((guint8*)newdata)[i] = cpy;
      else if (newintsize == 2)
	((guint16*)newdata)[i] = cpy;
      else if (newintsize == 4)
      	((guint32*)newdata)[i] = cpy;
      else
	g_assert_not_reached();

    }

    g_free(cd->data);
    cd->data = newdata;
    cd->intsize = newintsize;
  }

  cd->catsize = newcatsize;
}

static void
guppi_categorical_data_grow(GuppiCategoricalData* cd, gsize newsize)
{
  gpointer newdata;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(cd->intsize != 0);

  if (newsize <= cd->poolsize)
    return;

  newdata = g_malloc(newsize * cd->intsize);
  memcpy(newdata, cd->data, cd->intsize * cd->size);
  g_free(cd->data);
  cd->data = newdata;
  cd->poolsize = newsize;
}

static guint
guppi_categorical_data_string2code(GuppiCategoricalData* cd,
				   const gchar* s)
{
  dindex_t i;

  g_return_val_if_fail(cd != NULL, GUPPI_CATEGORICAL_DATA_BAD_CODE);
  g_return_val_if_fail(s != NULL, GUPPI_CATEGORICAL_DATA_BAD_CODE);

  i = guppi_string_data_lookup(cd->category, s);
  if (guppi_data_in_bounds(GUPPI_DATA(cd->category), i))
    return (guint)i;

  g_assert(!cd->frozen);

  guppi_categorical_data_grow_catsize(cd, cd->catsize+1);
  guppi_string_data_add(cd->category, s);

  /* New item must be the top item. */
  return (guint)(cd->catsize-1);
}

guint
guppi_categorical_data_encode(const GuppiCategoricalData* cd,
			      const gchar* s)
{
  dindex_t i;

  g_return_val_if_fail(cd != NULL, GUPPI_CATEGORICAL_DATA_BAD_CODE);
  g_return_val_if_fail(s != NULL, GUPPI_CATEGORICAL_DATA_BAD_CODE);

  i = guppi_string_data_lookup(cd->category, s);

  return guppi_data_in_bounds(GUPPI_DATA(cd->category), i) ?
    (guint)i : GUPPI_CATEGORICAL_DATA_BAD_CODE;
}

const gchar*
guppi_categorical_data_decode(const GuppiCategoricalData* cd,
			      guint c)
{
  g_return_val_if_fail(cd != NULL, NULL);

  if (c >= cd->catsize)
    return NULL;

  return guppi_string_data_get(cd->category, (dindex_t)c);
}

gboolean
guppi_categorical_data_validate_code(const GuppiCategoricalData* cd,
				     guint c)
{
  g_return_val_if_fail(cd != NULL, FALSE);

  return c < cd->catsize;
}

const gchar*
guppi_categorical_data_get(const GuppiCategoricalData* cd, dindex_t i)
{
  guint c;
  
  g_return_val_if_fail(cd != NULL, NULL);
  g_return_val_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i), NULL);

  c = guppi_categorical_data_get_code(cd, i);
  return guppi_string_data_get(cd->category, (dindex_t)c);
}

guint
guppi_categorical_data_get_code(const GuppiCategoricalData* cd, dindex_t i)
{
  dindex_t i0;
  g_return_val_if_fail(cd != NULL,
		       GUPPI_CATEGORICAL_DATA_BAD_CODE);
  g_return_val_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i),
		       GUPPI_CATEGORICAL_DATA_BAD_CODE);

  i0 = guppi_data_min_index(GUPPI_DATA(cd));

  switch (cd->intsize) {

  case 0:
    return GUPPI_CATEGORICAL_DATA_BAD_CODE;

  case 1:
    return ((guint8*)cd->data)[i-i0];

  case 2:
    return ((guint16*)cd->data)[i-i0];

  case 4:
    return ((guint32*)cd->data)[i-i0];

  default:
    g_assert_not_reached();
    return GUPPI_CATEGORICAL_DATA_BAD_CODE;
  }
}

void
guppi_categorical_data_set(GuppiCategoricalData* cd, dindex_t i,
			   const gchar* sbuf)
{
  guint c;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i));
  g_return_if_fail(sbuf != NULL);

  c = guppi_categorical_data_string2code(cd, sbuf);
  guppi_categorical_data_set_code(cd, i, c);
}

void
guppi_categorical_data_set_code(GuppiCategoricalData* cd, dindex_t i, guint c)
{
  dindex_t i0;
  guint old_value;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i));
  g_return_if_fail(guppi_categorical_data_validate_code(cd, c));
  g_return_if_fail(cd->intsize != 0);

  i0 = guppi_data_min_index(GUPPI_DATA(cd));

  switch (cd->intsize) {

  case 1:
    old_value = ((guint8*)cd->data)[i - i0];
    ((guint8*)cd->data)[i - i0] = (guint8)c;
    break;

  case 2:
    old_value = ((guint16*)cd->data)[i - i0];
    ((guint16*)cd->data)[i - i0] = (guint16)c;
    break;

  case 4:
    old_value = ((guint32*)cd->data)[i - i0];
    ((guint32*)cd->data)[i - i0] = (guint32)c;
    break;

  default:
    g_assert_not_reached();
  }

  guppi_data_touch_one(GUPPI_DATA(cd), i, &old_value);
}

void
guppi_categorical_data_add(GuppiCategoricalData* cd, const gchar* sbuf)
{
  guint c;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(sbuf != NULL);

  c = guppi_categorical_data_string2code(cd, sbuf);
  guppi_categorical_data_add_code(cd, c);
}

void
guppi_categorical_data_add_code(GuppiCategoricalData* cd, guint c)
{
  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_categorical_data_validate_code(cd, c));
  g_return_if_fail(cd->intsize != 0);

  if (cd->size == cd->poolsize)
    guppi_categorical_data_grow(cd, cd->size ? 2*cd->size : 128);

  switch (cd->intsize) {

  case 1:
    ((guint8*)cd->data)[cd->size] = (guint8)c;
    break;

  case 2:
    ((guint16*)cd->data)[cd->size] = (guint16)c;
    break;

  case 4:
    ((guint32*)cd->data)[cd->size] = (guint32)c;
    break;

  default:
    g_assert_not_reached();
  }

  ++cd->size;
  ++cd->base.max_index;

  guppi_data_touch_add(GUPPI_DATA(cd),
		       guppi_data_max_index(GUPPI_DATA(cd)),
		       &c);
}

void
guppi_categorical_data_insert(GuppiCategoricalData* cd, dindex_t i,
			      const gchar* sbuf)
{
  guint c;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i));
  g_return_if_fail(sbuf != NULL);

  c = guppi_categorical_data_string2code(cd, sbuf);
  guppi_categorical_data_insert_code(cd, i, c);
}

void
guppi_categorical_data_insert_code(GuppiCategoricalData* cd,
				   dindex_t i, guint c)
{
  dindex_t i0;
  gsize j;
  guint8* p;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_categorical_data_validate_code(cd, c));
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i));

  if (cd->size == cd->poolsize)
    guppi_categorical_data_grow(cd, cd->size ? 2*cd->size : 128);

  p = (guint8*)cd->data + (cd->size-1)*cd->intsize;
  for(j=cd->size-1; j >= (i-cd->base.min_index); --j) {
    switch (cd->intsize) {
    case 1:
      p[1] = p[0];
      break;
    case 2:
      ((guint16*)p)[1] = ((guint16*)p)[0];
      break;
    case 4:
      ((guint32*)p)[1]= ((guint32*)p)[0];
      break;
    default:
      g_assert_not_reached();
    }
    p -= cd->intsize;
  }

  i0 = guppi_data_min_index(GUPPI_DATA(cd));

  switch (cd->intsize) {

  case 1:
    ((guint8*)cd->data)[i-i0] = (guint8)c;
    break;
    
  case 2:
    ((guint16*)cd->data)[i-i0] = (guint16)c;
    break;

  case 4:
    ((guint32*)cd->data)[i-i0] = (guint32)c;
    break;

  default:
    g_assert_not_reached();
  }

  ++cd->size;
  ++cd->base.max_index;

  guppi_data_touch_add(GUPPI_DATA(cd), i, &c);
}

void
guppi_categorical_data_delete(GuppiCategoricalData* cd, dindex_t i)
{
  dindex_t i0;
  gsize j;
  guint8* p;

  g_return_if_fail(cd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(cd), i));
  g_return_if_fail(cd->intsize != 0);

  i0 = guppi_data_min_index(GUPPI_DATA(cd));

  p = (guint8*)cd->data + cd->intsize * (i-i0);

  for (j=i+1-i0; j < cd->size; ++j) {
    switch (cd->intsize) {
    case 1:
      p[0] = p[1];
      break;
    case 2:
      ((guint16*)p)[0] = ((guint16*)p)[1];
      break;
    case 4:
      ((guint32*)p)[0] = ((guint32*)p)[1];
      break;
    default:
      g_assert_not_reached();
    }

    p += cd->intsize;
  }

  --cd->size;
  --cd->base.max_index;

  guppi_data_touch(GUPPI_DATA(cd));
}


/* $Id: guppi-categorical-data.c,v 1.2 1999/12/07 17:06:11 trow Exp $ */





