/* 
   Copyright (C) 2004 James Bowes <bowes@cs.dal.ca>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <gnome-keyring.h>

#include <string.h>

#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-attribute-display.h"

struct _GKMAttributeDisplayPrivate
{
  GtkWidget *attributes_book;
  GtkWidget *unknown_attributes_warning;

  GnomeKeyringItemType item_type;
  GnomeKeyringAttributeList *attributes;

  GtkWidget **attribute_labels;
};

enum
{
  USER_LABEL = 0,
  DOMAIN_LABEL,
  SERVER_LABEL,
  OBJECT_LABEL,
  AUTHTYPE_LABEL,
  PROTOCOL_LABEL,
  PORT_LABEL,
  NUM_LABELS
};

enum
{
  BLANK_PAGE = 0,
  SECRET_PAGE,
  NETWORK_PAGE,
  NOTE_PAGE,
  NUM_PAGES
};

static GtkWidget *generic_secret_contruct_page (GtkWidget **labels);
static gboolean   generic_secret_update_page (GKMAttributeDisplay *display);
static GtkWidget *network_password_construct_page (GtkWidget **labels);
static gboolean   network_password_update_page (GKMAttributeDisplay *display);
static GtkWidget *note_construct_page (GtkWidget **labels);
static gboolean   note_update_page (GKMAttributeDisplay *display);

void gkm_attribute_display_clear (GKMAttributeDisplay *display);

static void gkm_attribute_display_class_init (GKMAttributeDisplay *class);
static void gkm_attribute_display_init       (GKMAttributeDisplay *display);
static void gkm_attribute_display_finalize   (GObject *object);
static void gkm_attribute_display_destroy    (GtkObject *object);

static GtkVBoxClass *parent_class = NULL;

GType
gkm_attribute_display_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMAttributeDisplayClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_attribute_display_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 
	sizeof (GKMAttributeDisplay),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_attribute_display_init,
	0
      };

      type = g_type_register_static (GTK_TYPE_VBOX, "GKMAttributeDisplay", &info, 0);
    }

  return type;
}

static void
gkm_attribute_display_class_init (GKMAttributeDisplay *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_attribute_display_finalize;

  object_class->destroy = gkm_attribute_display_destroy;
}

static void
gkm_attribute_display_init (GKMAttributeDisplay *display)
{
  GtkWidget *page;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkWidget *image;
  GtkWidget *alignment;

  display->priv = g_new0 (GKMAttributeDisplayPrivate, 1);

  display->priv->attribute_labels = g_new0 (GtkWidget *, NUM_LABELS);
 
  display->priv->attributes_book = gtk_notebook_new ();
  gtk_notebook_set_show_border (GTK_NOTEBOOK (display->priv->attributes_book), FALSE);
  gtk_notebook_set_show_tabs (GTK_NOTEBOOK (display->priv->attributes_book), FALSE);
  gtk_box_pack_start (GTK_BOX (display), display->priv->attributes_book, TRUE, TRUE, 0);
 
  /* Blank page */
  page = gtk_label_new ("");
  gtk_notebook_append_page (GTK_NOTEBOOK (display->priv->attributes_book), page, NULL);

  /* Generic Secret page. */
  page = generic_secret_contruct_page (display->priv->attribute_labels);
  alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (alignment), page);
  gtk_notebook_append_page (GTK_NOTEBOOK (display->priv->attributes_book), alignment, NULL);

  /* Network Secret page */
  page = network_password_construct_page (display->priv->attribute_labels);
  alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (alignment), page);
  gtk_notebook_append_page (GTK_NOTEBOOK (display->priv->attributes_book), alignment, NULL);

  /* Note page. */
  page = note_construct_page (display->priv->attribute_labels);
  alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (alignment), page);
  gtk_notebook_append_page (GTK_NOTEBOOK (display->priv->attributes_book), alignment, NULL);

  /* The non-standard attributes warning */
  display->priv->unknown_attributes_warning = alignment = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 0, 0, 0);
  gtk_box_pack_start (GTK_BOX (display), alignment, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_add (GTK_CONTAINER (alignment), hbox);
 
  image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);

  label = gtk_label_new (_("This item has non-standard attributes.\nUse 'Edit' to view them."));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
}

static void
gkm_attribute_display_finalize (GObject *object)
{
  GKMAttributeDisplay *display;

  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (object));

  display = GKM_ATTRIBUTE_DISPLAY (object);

  gnome_keyring_attribute_list_free (display->priv->attributes);
  g_free (display->priv->attribute_labels);
  g_free (display->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_attribute_display_destroy (GtkObject *object)
{
  GKMAttributeDisplay *display;

  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (object));

  display = GKM_ATTRIBUTE_DISPLAY (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (GTK_OBJECT (display));
}

GtkWidget *
gkm_attribute_display_new (void)
{
  GKMAttributeDisplay *display;

  display = g_object_new (GKM_TYPE_ATTRIBUTE_DISPLAY, NULL);

  return GTK_WIDGET (display);
}

/**************************************************
 * Generic secrets
 */

/* FIXME: What are the standard attributes for this? Does it have any? */
/* FIXME: These need to do something useful */

static GtkWidget *
generic_secret_contruct_page (GtkWidget **labels G_GNUC_UNUSED)
{
  GtkWidget *table;

  table = gtk_table_new (1, 2, TRUE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);

  return table;
}

static gboolean
generic_secret_update_page (GKMAttributeDisplay *display G_GNUC_UNUSED)
{
  return TRUE;
}

/**************************************************
 * Network passwords
 */

static GtkWidget *
network_password_construct_page (GtkWidget **labels)
{
  static const struct 
    {
      char *name;
      int idx;
    } field[] =
    {
      { N_("User:"), USER_LABEL },
      { N_("Domain:"),  DOMAIN_LABEL },
      { N_("Server:"), SERVER_LABEL },
      { N_("Object:"), OBJECT_LABEL },
      { N_("Authentication Method:"), AUTHTYPE_LABEL },
      { N_("Protocol:"), PROTOCOL_LABEL },
      { N_("Port:"), PORT_LABEL }
    };

  GtkWidget *table;
  GtkWidget *label;
  char *text;
  unsigned int i;

  table = gtk_table_new (7, 4, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);

  for (i = 0; i < G_N_ELEMENTS (field); i++)
    {
      label = gtk_label_new (NULL);
      text = g_strdup_printf ("<span weight='bold'>%s</span>", _(field[i].name));
      gtk_label_set_markup (GTK_LABEL (label), text);
      g_free (text);
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5); 
      gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, i, i+1);
      
      label = gtk_label_new ("");
      gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
      labels[field[i].idx] = label; 
      gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, i, i+1);
    }

  return table;
}

static gboolean
network_password_update_page (GKMAttributeDisplay *display)
{
  guint32 i;
  gboolean non_standard_attributes;
 
  non_standard_attributes = FALSE;
  for (i = 0; i < display->priv->attributes->len; i++)
    {
      GnomeKeyringAttribute attribute;
      
      attribute = gnome_keyring_attribute_list_index(display->priv->attributes, i);

      if (strcmp (attribute.name, "user") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[USER_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "domain") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[DOMAIN_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "server") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[SERVER_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "object") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[OBJECT_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "authtype") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[AUTHTYPE_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "protocol") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING)
	{
	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[PROTOCOL_LABEL]), attribute.value.string);
	}
      else if (strcmp (attribute.name, "port") == 0 && attribute.type == GNOME_KEYRING_ATTRIBUTE_TYPE_UINT32)
	{
	  gchar *port_num;
	  
	  port_num = g_strdup_printf ("%d", attribute.value.integer);
 	  gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[PORT_LABEL]), port_num);
	  g_free (port_num);
	}
      else
        {
          non_standard_attributes = TRUE;
	}
    }

  return non_standard_attributes;
}

/**************************************************
 * Notes
 */

static GtkWidget *
note_construct_page (GtkWidget **labels G_GNUC_UNUSED)
{
  GtkWidget *table;

  table = gtk_table_new (1, 2, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);

  return table;
}

static gboolean
note_update_page (GKMAttributeDisplay *display)
{
  if (display->priv->attributes->len > 0)
    {
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

static void
update_attribute_display (GKMAttributeDisplay *display)
{
  gboolean non_standard_attributes;
  int page;

  g_return_if_fail (display->priv->attributes != NULL);
  
  gkm_attribute_display_clear (display);
  gtk_widget_hide (display->priv->unknown_attributes_warning);

  switch (display->priv->item_type)
    {
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        non_standard_attributes = generic_secret_update_page (display);
        page = SECRET_PAGE;
        break;
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        non_standard_attributes = network_password_update_page (display);
        page = NETWORK_PAGE;
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        non_standard_attributes = note_update_page (display);
        page = NOTE_PAGE;
	break;
      default:
        non_standard_attributes = TRUE;
	page = BLANK_PAGE; 
    }

  gtk_notebook_set_current_page (GTK_NOTEBOOK (display->priv->attributes_book), page);
  if (non_standard_attributes)
    {
      gtk_widget_show (display->priv->unknown_attributes_warning);
    }
}

/**************************************************
 * Get and set the type
 */

void
gkm_attribute_display_clear (GKMAttributeDisplay *display)
{
  int i;

  for (i = 0; i < NUM_LABELS; i++)
    {
      gtk_label_set_text (GTK_LABEL (display->priv->attribute_labels[i]), "");
    }
}

void 
gkm_attribute_display_set (GKMAttributeDisplay       *display, 
			   GnomeKeyringItemType       item_type,
                           GnomeKeyringAttributeList *attributes)
{
  g_return_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (display));

  display->priv->item_type = item_type;

  gnome_keyring_attribute_list_free (display->priv->attributes);
  display->priv->attributes = gnome_keyring_attribute_list_copy (attributes);
  
  update_attribute_display (display);
}

GnomeKeyringItemType 
gkm_attribute_display_get_item_type (GKMAttributeDisplay *display)
{
  g_return_val_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (display), GNOME_KEYRING_ITEM_NO_TYPE);

  return display->priv->item_type;
}

GnomeKeyringAttributeList *
gkm_attribute_display_get_attributes (GKMAttributeDisplay *display)
{
  g_return_val_if_fail (GKM_IS_ATTRIBUTE_DISPLAY (display), NULL);

  return gnome_keyring_attribute_list_copy (display->priv->attributes);
}

