/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2008 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <locale.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <glade/glade.h>
#include <gtk/gtk.h>
#include <dbus/dbus-glib.h>
#include <gconf/gconf-client.h>

#include "egg-debug.h"
#include "egg-color.h"
#include "egg-string.h"
#include "egg-obj-list.h"
#include "egg-array-float.h"
#include "egg-unique.h"

#include "gpm-common.h"
#include "gpm-stock-icons.h"
#include "gpm-devicekit.h"
#include "gpm-graph-widget.h"
#include "dkp-client.h"
#include "dkp-history-obj.h"
#include "dkp-stats-obj.h"
#include "dkp-device.h"
#include "dkp-wakeups.h"

static GladeXML *glade_xml = NULL;
static GtkListStore *list_store_info = NULL;
static GtkListStore *list_store_devices = NULL;
static GtkListStore *list_store_wakeups = NULL;
gchar *current_device = NULL;
static const gchar *history_type;
static const gchar *stats_type;
static guint history_time;
static GConfClient *gconf_client;
static gfloat sigma_smoothing = 0.0f;
static DkpWakeups *wakeups = NULL;

enum {
	GPM_INFO_COLUMN_TEXT,
	GPM_INFO_COLUMN_VALUE,
	GPM_INFO_COLUMN_LAST
};

enum {
	GPM_DEVICES_COLUMN_ICON,
	GPM_DEVICES_COLUMN_TEXT,
	GPM_DEVICES_COLUMN_ID,
	GPM_DEVICES_COLUMN_LAST
};

enum {
	GPM_WAKEUPS_COLUMN_ICON,
	GPM_WAKEUPS_COLUMN_ID,
	GPM_WAKEUPS_COLUMN_VALUE,
	GPM_WAKEUPS_COLUMN_CMDLINE,
	GPM_WAKEUPS_COLUMN_DETAILS,
	GPM_WAKEUPS_COLUMN_LAST
};

#define GPM_HISTORY_RATE_TEXT			_("Rate")
#define GPM_HISTORY_CHARGE_TEXT			_("Charge")
#define GPM_HISTORY_TIME_FULL_TEXT		_("Time to full")
#define GPM_HISTORY_TIME_EMPTY_TEXT		_("Time to empty")

#define GPM_HISTORY_RATE_VALUE			"rate"
#define GPM_HISTORY_CHARGE_VALUE		"charge"
#define GPM_HISTORY_TIME_FULL_VALUE		"time-full"
#define GPM_HISTORY_TIME_EMPTY_VALUE		"time-empty"

#define GPM_HISTORY_MINUTE_TEXT			_("10 minutes")
#define GPM_HISTORY_HOUR_TEXT			_("2 hours")
#define GPM_HISTORY_DAY_TEXT			_("1 day")
#define GPM_HISTORY_WEEK_TEXT			_("1 week")

#define GPM_HISTORY_MINUTE_VALUE		10*60
#define GPM_HISTORY_HOUR_VALUE			2*60*60
#define GPM_HISTORY_DAY_VALUE			24*60*60
#define GPM_HISTORY_WEEK_VALUE			7*24*60*60

#define GPM_STATS_CHARGE_DATA_TEXT		_("Charge profile")
#define GPM_STATS_CHARGE_ACCURACY_TEXT		_("Charge accuracy")
#define GPM_STATS_DISCHARGE_DATA_TEXT		_("Discharge profile")
#define GPM_STATS_DISCHARGE_ACCURACY_TEXT	_("Discharge accuracy")

#define GPM_STATS_CHARGE_DATA_VALUE		"charge-data"
#define GPM_STATS_CHARGE_ACCURACY_VALUE		"charge-accuracy"
#define GPM_STATS_DISCHARGE_DATA_VALUE		"discharge-data"
#define GPM_STATS_DISCHARGE_ACCURACY_VALUE	"discharge-accuracy"

/**
 * gpm_stats_button_help_cb:
 **/
static void
gpm_stats_button_help_cb (GtkWidget *widget, gboolean data)
{
	//gpm_gnome_help ("update-log");
}

/**
 * gpm_stats_add_info_columns:
 **/
static void
gpm_stats_add_info_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	/* image */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Attribute"), renderer,
							   "markup", GPM_INFO_COLUMN_TEXT, NULL);
	gtk_tree_view_column_set_sort_column_id (column, GPM_INFO_COLUMN_TEXT);
	gtk_tree_view_append_column (treeview, column);

	/* column for text */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Value"), renderer,
							   "markup", GPM_INFO_COLUMN_VALUE, NULL);
	gtk_tree_view_append_column (treeview, column);
}

/**
 * gpm_stats_add_devices_columns:
 **/
static void
gpm_stats_add_devices_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	/* image */
	renderer = gtk_cell_renderer_pixbuf_new ();
	g_object_set (renderer, "stock-size", GTK_ICON_SIZE_DIALOG, NULL);
	column = gtk_tree_view_column_new_with_attributes (_("Image"), renderer,
							   "icon-name", GPM_DEVICES_COLUMN_ICON, NULL);
	gtk_tree_view_append_column (treeview, column);

	/* column for text */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Description"), renderer,
							   "markup", GPM_DEVICES_COLUMN_TEXT, NULL);
	gtk_tree_view_column_set_sort_column_id (column, GPM_INFO_COLUMN_TEXT);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_expand (column, TRUE);
}

/**
 * gpm_stats_add_wakeups_columns:
 **/
static void
gpm_stats_add_wakeups_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	/* image */
	renderer = gtk_cell_renderer_pixbuf_new ();
	g_object_set (renderer, "stock-size", GTK_ICON_SIZE_BUTTON, NULL);
	column = gtk_tree_view_column_new_with_attributes (_("Type"), renderer,
							   "icon-name", GPM_WAKEUPS_COLUMN_ICON, NULL);
	gtk_tree_view_append_column (treeview, column);

	/* column for id */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("ID"), renderer,
							   "markup", GPM_WAKEUPS_COLUMN_ID, NULL);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_expand (column, TRUE);

	/* column for value */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Wakeups"), renderer,
							   "markup", GPM_WAKEUPS_COLUMN_VALUE, NULL);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_expand (column, TRUE);

	/* column for cmdline */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Command"), renderer,
							   "markup", GPM_WAKEUPS_COLUMN_CMDLINE, NULL);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_expand (column, TRUE);

	/* column for details */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Details"), renderer,
							   "markup", GPM_WAKEUPS_COLUMN_DETAILS, NULL);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_expand (column, TRUE);
}

/**
 * gpm_stats_add_info_data:
 **/
static void
gpm_stats_add_info_data (const gchar *attr, const gchar *text)
{
	GtkTreeIter iter;
	gtk_list_store_append (list_store_info, &iter);
	gtk_list_store_set (list_store_info, &iter,
			    GPM_INFO_COLUMN_TEXT, attr,
			    GPM_INFO_COLUMN_VALUE, text, -1);
}

/**
 * gpm_stats_update_smooth_data:
 **/
static EggObjList *
gpm_stats_update_smooth_data (EggObjList *list)
{
	guint i;
	GpmPointObj *point;
	GpmPointObj point_new;
	EggObjList *new;
	EggArrayFloat *raw;
	EggArrayFloat *convolved;
	EggArrayFloat *outliers;
	EggArrayFloat *gaussian = NULL;

	/* convert the y data to a EggArrayFloat array */
	raw = egg_array_float_new (list->len);
	convolved = egg_array_float_new (list->len);
	for (i=0; i<list->len; i++) {
		point = (GpmPointObj *) egg_obj_list_index (list, i);
		egg_array_float_set (raw, i, point->y);
	}

	/* remove any outliers */
	outliers = egg_array_float_remove_outliers (raw, 3, 0.1);

	/* convolve with gaussian */
	gaussian = egg_array_float_compute_gaussian (15, sigma_smoothing);
	convolved = egg_array_float_convolve (outliers, gaussian);

	/* add the smoothed data back into a new array */
	new = egg_obj_list_new ();
	egg_obj_list_set_copy (new, (EggObjListCopyFunc) dkp_point_obj_copy);
	egg_obj_list_set_free (new, (EggObjListFreeFunc) dkp_point_obj_free);
	for (i=0; i<list->len; i++) {
		point = (GpmPointObj *) egg_obj_list_index (list, i);
		point_new.color = point->color;
		point_new.x = point->x;
		point_new.y = egg_array_float_get (convolved, i);
		egg_obj_list_add (new, &point_new);
	}

	/* free data */
	egg_array_float_free (gaussian);
	egg_array_float_free (raw);
	egg_array_float_free (convolved);
	egg_array_float_free (outliers);

	return new;
}

/**
 * gpm_stats_time_to_text:
 **/
static gchar *
gpm_stats_time_to_text (gint seconds)
{
	gfloat value = seconds;

	if (value < 0)
		return g_strdup ("unknown");
	if (value < 60)
		return g_strdup_printf ("%.0f seconds", value);
	value /= 60.0;
	if (value < 60)
		return g_strdup_printf ("%.1f minutes", value);
	value /= 60.0;
	if (value < 60)
		return g_strdup_printf ("%.1f hours", value);
	value /= 24.0;
	return g_strdup_printf ("%.1f days", value);
}

/**
 * gpm_stats_bool_to_text:
 **/
static const gchar *
gpm_stats_bool_to_text (gboolean ret)
{
	return ret ? _("Yes") : _("No");
}

/**
 * gpm_stats_update_info_page_details:
 **/
static void
gpm_stats_update_info_page_details (const DkpDevice *device)
{
	const DkpObject *obj;
	struct tm *time_tm;
	time_t t;
	gchar time_buf[256];
	gchar *text;

	gtk_list_store_clear (list_store_info);

	obj = dkp_device_get_object (device);

	/* get a human readable time */
	t = (time_t) obj->update_time;
	time_tm = localtime (&t);
	strftime (time_buf, sizeof time_buf, "%c", time_tm);

	gpm_stats_add_info_data (_("Device"), dkp_device_get_object_path (device));
	gpm_stats_add_info_data (_("Type"), gpm_device_type_to_localised_text (obj->type, 1));
	if (!egg_strzero (obj->vendor))
		gpm_stats_add_info_data (_("Vendor"), obj->vendor);
	if (!egg_strzero (obj->model))
		gpm_stats_add_info_data (_("Model"), obj->model);
	if (!egg_strzero (obj->serial))
		gpm_stats_add_info_data (_("Serial number"), obj->serial);
	gpm_stats_add_info_data (_("Supply"), gpm_stats_bool_to_text (obj->power_supply));

	text = g_strdup_printf ("%d seconds", (int) (time (NULL) - obj->update_time));
	gpm_stats_add_info_data (_("Refreshed"), text);
	g_free (text);

	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MOUSE ||
	    obj->type == DKP_DEVICE_TYPE_KEYBOARD ||
	    obj->type == DKP_DEVICE_TYPE_UPS)
		gpm_stats_add_info_data (_("Present"), gpm_stats_bool_to_text (obj->is_present));
	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MOUSE ||
	    obj->type == DKP_DEVICE_TYPE_KEYBOARD)
		gpm_stats_add_info_data (_("Rechargeable"), gpm_stats_bool_to_text (obj->is_rechargeable));
	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MOUSE ||
	    obj->type == DKP_DEVICE_TYPE_KEYBOARD)
		gpm_stats_add_info_data (_("State"), dkp_device_state_to_text (obj->state));
	if (obj->type == DKP_DEVICE_TYPE_BATTERY) {
		text = g_strdup_printf ("%.1f Wh", obj->energy);
		gpm_stats_add_info_data (_("Energy"), text);
		g_free (text);
		text = g_strdup_printf ("%.1f Wh", obj->energy_empty);
		gpm_stats_add_info_data (_("Energy when empty"), text);
		g_free (text);
		text = g_strdup_printf ("%.1f Wh", obj->energy_full);
		gpm_stats_add_info_data (_("Energy when full"), text);
		g_free (text);
		text = g_strdup_printf ("%.1f Wh", obj->energy_full_design);
		gpm_stats_add_info_data (_("Energy (design)"), text);
		g_free (text);
	}
	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MONITOR) {
		text = g_strdup_printf ("%.1f W", obj->energy_rate);
		gpm_stats_add_info_data (_("Rate"), text);
		g_free (text);
	}
	if (obj->type == DKP_DEVICE_TYPE_UPS ||
	    obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MONITOR) {
		text = g_strdup_printf ("%.1f V", obj->voltage);
		gpm_stats_add_info_data (_("Voltage"), text);
		g_free (text);
	}
	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_UPS) {
		if (obj->time_to_full >= 0) {
			text = gpm_stats_time_to_text (obj->time_to_full);
			gpm_stats_add_info_data (_("Time to full"), text);
			g_free (text);
		}
		if (obj->time_to_empty >= 0) {
			text = gpm_stats_time_to_text (obj->time_to_empty);
			gpm_stats_add_info_data (_("Time to empty"), text);
			g_free (text);
		}
	}
	if (obj->type == DKP_DEVICE_TYPE_BATTERY ||
	    obj->type == DKP_DEVICE_TYPE_MOUSE ||
	    obj->type == DKP_DEVICE_TYPE_KEYBOARD ||
	    obj->type == DKP_DEVICE_TYPE_UPS) {
		text = g_strdup_printf ("%.1f%%", obj->percentage);
		gpm_stats_add_info_data (_("Percentage"), text);
		g_free (text);
	}
	if (obj->type == DKP_DEVICE_TYPE_BATTERY) {
		text = g_strdup_printf ("%.1f%%", obj->capacity);
		gpm_stats_add_info_data (_("Capacity"), text);
		g_free (text);
	}
	if (obj->type == DKP_DEVICE_TYPE_BATTERY)
		gpm_stats_add_info_data (_("Technology"), gpm_device_technology_to_localised_text (obj->technology));
	if (obj->type == DKP_DEVICE_TYPE_LINE_POWER)
		gpm_stats_add_info_data (_("Online"), gpm_stats_bool_to_text (obj->online));
}

/**
 * gpm_stats_set_graph_data:
 **/
static void
gpm_stats_set_graph_data (GtkWidget *widget, EggObjList *data, gboolean use_smoothed, gboolean use_points)
{
	EggObjList *smoothed;

	gpm_graph_widget_data_clear (GPM_GRAPH_WIDGET (widget));

	/* add correct data */
	if (!use_smoothed) {
		if (use_points)
			gpm_graph_widget_data_assign (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_PLOT_BOTH, data);
		else
			gpm_graph_widget_data_assign (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_PLOT_LINE, data);
	} else {
		smoothed = gpm_stats_update_smooth_data (data);
		if (use_points)
			gpm_graph_widget_data_assign (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_PLOT_POINTS, data);
		gpm_graph_widget_data_assign (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_PLOT_LINE, smoothed);
		g_object_unref (smoothed);
	}

	/* show */
	gtk_widget_show (widget);
}

/**
 * gpm_stats_update_info_page_history:
 **/
static void
gpm_stats_update_info_page_history (const DkpDevice *device)
{
	EggObjList *array;
	guint i;
	DkpHistoryObj *hobj;
	GtkWidget *widget;
	gboolean checked;
	gboolean points;
	GpmPointObj *point;
	EggObjList *new;
	gint32 offset = 0;
	GTimeVal timeval;

	new = egg_obj_list_new ();
	egg_obj_list_set_copy (new, (EggObjListCopyFunc) dkp_point_obj_copy);
	egg_obj_list_set_free (new, (EggObjListFreeFunc) dkp_point_obj_free);

	widget = glade_xml_get_widget (glade_xml, "custom_graph_history");
	gpm_graph_widget_set_type_x (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_TIME);
	if (strcmp (history_type, GPM_HISTORY_CHARGE_VALUE) == 0)
		gpm_graph_widget_set_type_y (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_PERCENTAGE);
	else if (strcmp (history_type, GPM_HISTORY_RATE_VALUE) == 0)
		gpm_graph_widget_set_type_y (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_POWER);
	else
		gpm_graph_widget_set_type_y (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_TIME);

	array = dkp_device_get_history (device, history_type, history_time, 150);
	if (array == NULL) {
		gtk_widget_hide (widget);
		goto out;
	}
	gtk_widget_show (widget);

	g_get_current_time (&timeval);
	offset = timeval.tv_sec;

	for (i=0; i<array->len; i++) {
		hobj = (DkpHistoryObj *) egg_obj_list_index (array, i);

		/* abandon this point */
		if (hobj->state == DKP_DEVICE_STATE_UNKNOWN)
			continue;

		point = dkp_point_obj_new ();
		point->x = (gint32) hobj->time - offset;
		point->y = hobj->value;
		if (hobj->state == DKP_DEVICE_STATE_CHARGING)
			point->color = egg_color_from_rgb (255, 0, 0);
		else if (hobj->state == DKP_DEVICE_STATE_DISCHARGING)
			point->color = egg_color_from_rgb (0, 0, 255);
		else {
			if (strcmp (history_type, GPM_HISTORY_RATE_VALUE) == 0)
				point->color = egg_color_from_rgb (255, 255, 255);
			else
				point->color = egg_color_from_rgb (0, 255, 0);
		}
		egg_obj_list_add (new, point);
		dkp_point_obj_free (point);
	}

	/* render */
	sigma_smoothing = 2.0;
	widget = glade_xml_get_widget (glade_xml, "checkbutton_smooth_history");
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	widget = glade_xml_get_widget (glade_xml, "checkbutton_points_history");
	points = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));

	/* present data to graph */
	widget = glade_xml_get_widget (glade_xml, "custom_graph_history");
	gpm_stats_set_graph_data (widget, new, checked, points);

	g_object_unref (array);
	g_object_unref (new);
out:
	return;
}

/**
 * gpm_stats_update_info_page_stats:
 **/
static void
gpm_stats_update_info_page_stats (const DkpDevice *device)
{
	EggObjList *array;
	guint i;
	DkpStatsObj *sobj;
	GtkWidget *widget;
	gboolean checked;
	gboolean points;
	GpmPointObj *point;
	EggObjList *new;
	gboolean use_data = FALSE;
	const gchar *type = NULL;

	new = egg_obj_list_new ();
	egg_obj_list_set_copy (new, (EggObjListCopyFunc) dkp_point_obj_copy);
	egg_obj_list_set_free (new, (EggObjListFreeFunc) dkp_point_obj_free);

	widget = glade_xml_get_widget (glade_xml, "custom_graph_stats");
	if (strcmp (stats_type, GPM_STATS_CHARGE_DATA_VALUE) == 0) {
		type = "charging";
		use_data = TRUE;
	} else if (strcmp (stats_type, GPM_STATS_DISCHARGE_DATA_VALUE) == 0) {
		type = "discharging";
		use_data = TRUE;
	} else if (strcmp (stats_type, GPM_STATS_CHARGE_ACCURACY_VALUE) == 0) {
		type = "charging";
		use_data = FALSE;
	} else if (strcmp (stats_type, GPM_STATS_DISCHARGE_ACCURACY_VALUE) == 0) {
		type = "discharging";
		use_data = FALSE;
	} else {
		g_assert_not_reached ();
	}
	gpm_graph_widget_set_type_x (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_PERCENTAGE);

	if (use_data)
		gpm_graph_widget_set_type_y (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_FACTOR);
	else
		gpm_graph_widget_set_type_y (GPM_GRAPH_WIDGET (widget), GPM_GRAPH_WIDGET_TYPE_PERCENTAGE);

	array = dkp_device_get_statistics (device, type);
	if (array == NULL) {
		gtk_widget_hide (widget);
		goto out;
	}

	for (i=0; i<array->len; i++) {
		sobj = (DkpStatsObj *) egg_obj_list_index (array, i);
		point = dkp_point_obj_new ();
		point->x = i;
		if (use_data)
			point->y = sobj->value;
		else
			point->y = sobj->accuracy;
		point->color = egg_color_from_rgb (255, 0, 0);
		egg_obj_list_add (new, point);
		dkp_point_obj_free (point);
	}

	/* render */
	sigma_smoothing = 1.1;
	widget = glade_xml_get_widget (glade_xml, "checkbutton_smooth_stats");
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	widget = glade_xml_get_widget (glade_xml, "checkbutton_points_stats");
	points = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));

	/* present data to graph */
	widget = glade_xml_get_widget (glade_xml, "custom_graph_stats");
	gpm_stats_set_graph_data (widget, new, checked, points);

	g_object_unref (array);
	g_object_unref (new);
out:
	return;
}

/**
 * gpm_stats_update_info_data_page:
 **/
static void
gpm_stats_update_info_data_page (const DkpDevice *device, gint page)
{
	if (page == 0)
		gpm_stats_update_info_page_details (device);
	else if (page == 1)
		gpm_stats_update_info_page_history (device);
	else if (page == 2)
		gpm_stats_update_info_page_stats (device);
}

/**
 * gpm_stats_update_info_data:
 **/
static void
gpm_stats_update_info_data (const DkpDevice *device)
{
	gint page;
	GtkWidget *widget;
	GtkWidget *page_widget;
	const DkpObject	*obj;

	widget = glade_xml_get_widget (glade_xml, "notebook1");
	obj = dkp_device_get_object (device);

	/* show info page */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 0);
	gtk_widget_show (page_widget);

	/* hide history if no support */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 1);
	if (obj->has_history)
		gtk_widget_show (page_widget);
	else
		gtk_widget_hide (page_widget);

	/* hide statistics if no support */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 2);
	if (obj->has_statistics)
		gtk_widget_show (page_widget);
	else
		gtk_widget_hide (page_widget);

	/* hide wakeups page */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 3);
	gtk_widget_hide (page_widget);

	page = gtk_notebook_get_current_page (GTK_NOTEBOOK (widget));
	gpm_stats_update_info_data_page (device, page);

	return;
}

/**
 * gpm_stats_add_wakeups_obj:
 **/
static void
gpm_stats_add_wakeups_obj (const DkpWakeupsObj *obj)
{
	const gchar *icon;
	gchar *value;
	gchar *cmdline;
	gchar *id;
	const gchar *cmdline_ptr;
	gchar *cmdline_escaped;
	gchar *details;
	gchar *found;
	GtkTreeIter iter;

	if (obj->is_userspace) {
		icon = "application-x-executable";
		id = g_strdup_printf ("%i", obj->id);
	} else {
		icon = "applications-system";
		if (obj->id < 0xff0)
			id = g_strdup_printf ("IRQ%i", obj->id);
		else
			id = g_strdup ("IRQx");
	}

	/* formate value to one decimal place */
	value = g_strdup_printf ("%.1f", obj->value);

	/* truncate at first space or ':' */
	cmdline = g_strdup (obj->cmdline);
	found = strstr (cmdline, ":");
	if (found != NULL)
		*found = '\0';
	found = strstr (cmdline, " ");
	if (found != NULL)
		*found = '\0';

	/* remove ./ */
	found = g_strrstr (cmdline, "/");
	if (found != NULL)
		cmdline_ptr = found + 1;
	else
		cmdline_ptr = cmdline;

	if (strcmp (cmdline_ptr, "insmod") == 0)
		cmdline_ptr = _("Kernel module");
	else if (strcmp (cmdline_ptr, "modprobe") == 0)
		cmdline_ptr = _("Kernel module");
	else if (strcmp (cmdline_ptr, "swapper") == 0)
		cmdline_ptr = _("Kernel core");
	else if (strcmp (cmdline_ptr, "kernel-ipi") == 0)
		cmdline_ptr = _("Interprocessor interrupt");
	else if (strcmp (cmdline_ptr, "interrupt") == 0)
		cmdline_ptr = _("Interrupt");

	/* format command line */
	if (obj->is_userspace)
		cmdline_escaped = g_markup_escape_text (cmdline_ptr, -1);
	else
		cmdline_escaped = g_markup_printf_escaped ("<i>%s</i>", cmdline_ptr);

	/* replace common driver names */
	if (strcmp (obj->details, "i8042") == 0)
		details = g_strdup (_("PS/2 keyboard/mouse/touchpad"));
	else if (strcmp (obj->details, "acpi") == 0)
		details = g_strdup (_("ACPI"));
	else if (strcmp (obj->details, "ata_piix") == 0)
		details = g_strdup (_("Serial ATA"));
	else if (strcmp (obj->details, "libata") == 0)
		details = g_strdup (_("ATA host controller"));
	else if (strcmp (obj->details, "iwl3945") == 0)
		details = g_strdup (_("Intel wireless adaptor"));

	/* try to make the wakeup type nicer */
	else if (g_str_has_prefix (obj->details, "__mod_timer"))
		details = g_strdup_printf (_("Timer %s"), obj->details+12);
	else if (g_str_has_prefix (obj->details, "mod_timer"))
		details = g_strdup_printf (_("Timer %s"), obj->details+10);
	else if (g_str_has_prefix (obj->details, "do_setitimer"))
		details = g_strdup_printf (_("Timer %s"), obj->details+10);
	else if (g_str_has_prefix (obj->details, "do_nanosleep"))
		details = g_strdup_printf (_("Sleep %s"), obj->details+13);
	else if (g_str_has_prefix (obj->details, "futex_wait"))
		details = g_strdup_printf (_("Wait %s"), obj->details+11);
	else if (g_str_has_prefix (obj->details, "queue_delayed_work_on"))
		details = g_strdup_printf (_("Work queue %s"), obj->details+22);
	else if (g_str_has_prefix (obj->details, "queue_delayed_work"))
		details = g_strdup_printf (_("Work queue %s"), obj->details+19);
	else if (g_str_has_prefix (obj->details, "dst_run_gc"))
		details = g_strdup_printf (_("Network route flush %s"), obj->details+11);
	else if (g_str_has_prefix (obj->details, "usb_hcd_poll_rh_status"))
		details = g_strdup_printf (_("USB activity %s"), obj->details+23);
	else if (g_str_has_prefix (obj->details, "Local timer interrupts"))
		details = g_strdup (_("Local interrupts"));
	else if (g_str_has_prefix (obj->details, "Rescheduling interrupts"))
		details = g_strdup (_("Rescheduling interrupts"));
	else
		details = g_markup_escape_text (obj->details, -1);

	gtk_list_store_append (list_store_wakeups, &iter);
	gtk_list_store_set (list_store_wakeups, &iter,
			    GPM_WAKEUPS_COLUMN_ID, id,
			    GPM_WAKEUPS_COLUMN_VALUE, value,
			    GPM_WAKEUPS_COLUMN_CMDLINE, cmdline_escaped,
			    GPM_WAKEUPS_COLUMN_DETAILS, details,
			    GPM_WAKEUPS_COLUMN_ICON, icon, -1);
	g_free (cmdline);
	g_free (cmdline_escaped);
	g_free (details);
	g_free (value);
	g_free (id);
}

/**
 * gpm_stats_update_wakeups_data:
 **/
static void
gpm_stats_update_wakeups_data (void)
{
	GtkWidget *widget;
	GtkWidget *page_widget;
	guint total;
	DkpWakeupsObj *obj;
	gchar *text;
	guint i;
	GError *error = NULL;
	GPtrArray *array;

	widget = glade_xml_get_widget (glade_xml, "notebook1");

	/* hide other pages */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 0);
	gtk_widget_hide (page_widget);
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 1);
	gtk_widget_hide (page_widget);
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 2);
	gtk_widget_hide (page_widget);

	/* show wakeups page */
	page_widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK(widget), 3);
	gtk_widget_show (page_widget);

	/* show total */
	total = dkp_wakeups_get_total (wakeups, &error);
	widget = glade_xml_get_widget (glade_xml, "label_total_wakeups");
	if (error == NULL) {
		text = g_strdup_printf ("%i", total);
		gtk_label_set_label (GTK_LABEL(widget), text);
		g_free (text);
	} else {
		gtk_label_set_label (GTK_LABEL(widget), error->message);
		g_error_free (error);
	}

	/* get data */
	gtk_list_store_clear (list_store_wakeups);
	array = dkp_wakeups_get_data (wakeups, NULL);
	if (array == NULL)
		return;
	for (i=0; i<array->len; i++) {
		obj = g_ptr_array_index (array, i);
		gpm_stats_add_wakeups_obj (obj);
	}
	g_ptr_array_foreach (array, (GFunc) dkp_wakeups_obj_free, NULL);
	g_ptr_array_free (array, TRUE);
}

static void
gpm_stats_set_title (GtkWindow *window, gint page_num)
{
	gchar *title;
	const gchar * const page_titles[] = {
		N_("Device Information"),
		N_("Device History"),
		N_("Device Profile"),
		N_("Processor Wakeups")
	};

	title = g_strdup_printf ("%s - %s", _("Power Statistics"), _(page_titles[page_num]));
	gtk_window_set_title (window, title);
	g_free (title);
}

/**
 * gpm_stats_notebook_changed_cb:
 **/
static void
gpm_stats_notebook_changed_cb (GtkNotebook *notebook, GtkNotebookPage *page, gint page_num, gpointer user_data)
{
	DkpDevice *device;
	GtkWidget *widget;

	/* set the window title depending on the mode */
	widget = glade_xml_get_widget (glade_xml, "dialog_stats");
	gpm_stats_set_title (GTK_WINDOW (widget), page_num);

	/* save page in gconf */
	gconf_client_set_int (gconf_client, GPM_CONF_INFO_PAGE_NUMBER, page_num, NULL);

	if (current_device == NULL)
		return;

	if (egg_strequal (current_device, "wakeups"))
		return;

	device = dkp_device_new ();
	dkp_device_set_object_path (device, current_device);
	gpm_stats_update_info_data_page (device, page_num);
	gpm_stats_update_info_data (device);
	g_object_unref (device);
}

/**
 * gpm_stats_button_refresh_cb:
 **/
static void
gpm_stats_button_refresh_cb (GtkWidget *widget, gpointer data)
{
	DkpDevice *device;
	device = dkp_device_new ();
	dkp_device_set_object_path (device, current_device);
	dkp_device_refresh (device);
	gpm_stats_update_info_data (device);
	g_object_unref (device);
}

/**
 * gpm_stats_button_update_ui:
 **/
static void
gpm_stats_button_update_ui (void)
{
	DkpDevice *device;
	device = dkp_device_new ();
	dkp_device_set_object_path (device, current_device);
	gpm_stats_update_info_data (device);
	g_object_unref (device);
}

/**
 * gpm_stats_devices_treeview_clicked_cb:
 **/
static void
gpm_stats_devices_treeview_clicked_cb (GtkTreeSelection *selection, gboolean data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	DkpDevice *device;

	/* This will only work in single or browse selection mode! */
	if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
		g_free (current_device);
		gtk_tree_model_get (model, &iter, GPM_DEVICES_COLUMN_ID, &current_device, -1);

		/* save device in gconf */
		gconf_client_set_string (gconf_client, GPM_CONF_INFO_LAST_DEVICE, current_device, NULL);

		/* show transaction_id */
		egg_debug ("selected row is: %s", current_device);

		/* is special device */
		if (egg_strequal (current_device, "wakeups")) {
			gpm_stats_update_wakeups_data ();
		} else {
			device = dkp_device_new ();
			dkp_device_set_object_path (device, current_device);
			gpm_stats_update_info_data (device);
			g_object_unref (device);
		}

	} else {
		egg_debug ("no row selected");
	}
}

/**
 * gpm_stats_create_custom_widget:
 **/
static GtkWidget *
gpm_stats_create_custom_widget (GladeXML *xml, gchar *func_name, gchar *name,
				gchar *string1, gchar *string2, gint int1, gint int2, gpointer user_data)
{
	GtkWidget *widget = NULL;
	if (strcmp ("gpm_graph_widget_new", func_name) == 0) {
		widget = gpm_graph_widget_new ();
		return widget;
	}
	egg_warning ("name unknown=%s", name);
	return NULL;
}

/**
 * gpm_stats_window_activated_cb
 **/
static void
gpm_stats_window_activated_cb (EggUnique *egg_unique, gpointer data)
{
	GtkWidget *widget;
	widget = glade_xml_get_widget (glade_xml, "dialog_stats");
	gtk_window_present (GTK_WINDOW (widget));
}

/**
 * gpm_stats_add_device:
 **/
static void
gpm_stats_add_device (const DkpDevice *device)
{
	const gchar *id;
	GtkTreeIter iter;
	const DkpObject *obj;
	const gchar *text;
	const gchar *icon;

	obj = dkp_device_get_object (device);
	id = dkp_device_get_object_path (device);
	text = gpm_device_type_to_localised_text (obj->type, 1);
	icon = gpm_devicekit_get_object_icon (obj);

	gtk_list_store_append (list_store_devices, &iter);
	gtk_list_store_set (list_store_devices, &iter,
			    GPM_DEVICES_COLUMN_ID, id,
			    GPM_DEVICES_COLUMN_TEXT, text,
			    GPM_DEVICES_COLUMN_ICON, icon, -1);
}

/**
 * gpm_stats_data_changed_cb:
 **/
static void
gpm_stats_data_changed_cb (DkpClient *client, gpointer user_data)
{
	if (egg_strequal (current_device, "wakeups"))
		gpm_stats_update_wakeups_data ();
}

/**
 * gpm_stats_device_added_cb:
 **/
static void
gpm_stats_device_added_cb (DkpClient *client, const DkpDevice *device, gpointer user_data)
{
	const gchar *object_path;
	object_path = dkp_device_get_object_path (device);
	egg_debug ("added:     %s", object_path);
	gpm_stats_add_device (device);
}

/**
 * gpm_stats_device_changed_cb:
 **/
static void
gpm_stats_device_changed_cb (DkpClient *client, const DkpDevice *device, gpointer user_data)
{
	const gchar *object_path;
	object_path = dkp_device_get_object_path (device);
	if (object_path == NULL || current_device == NULL)
		return;
	egg_debug ("changed:   %s", object_path);
	if (strcmp (current_device, object_path) == 0)
		gpm_stats_update_info_data (device);
}

/**
 * gpm_stats_device_removed_cb:
 **/
static void
gpm_stats_device_removed_cb (DkpClient *client, const DkpDevice *device, gpointer user_data)
{
	const gchar *object_path;
	GtkTreeIter iter;
	gchar *id = NULL;
	gboolean ret;

	object_path = dkp_device_get_object_path (device);
	egg_debug ("removed:   %s", object_path);
	if (strcmp (current_device, object_path) == 0) {
		gtk_list_store_clear (list_store_info);
	}

	/* search the list and remove the object path entry */
	ret = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (list_store_devices), &iter);
	do {
		gtk_tree_model_get (GTK_TREE_MODEL (list_store_devices), &iter, GPM_DEVICES_COLUMN_ID, &id, -1);
		if (strcmp (id, object_path) == 0) {
			gtk_list_store_remove (list_store_devices, &iter);
			break;
		}
		g_free (id);
		ret = gtk_tree_model_iter_next (GTK_TREE_MODEL (list_store_devices), &iter);
	} while (ret);
}

/**
 * gpm_stats_history_type_combo_changed_cb:
 **/
static void
gpm_stats_history_type_combo_changed_cb (GtkWidget *widget, gpointer data)
{
	gchar *value;
	const gchar *axis_x = NULL;
	const gchar *axis_y = NULL;
	value = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
	if (strcmp (value, GPM_HISTORY_RATE_TEXT) == 0) {
		history_type = GPM_HISTORY_RATE_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Time elapsed");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Power");
	} else if (strcmp (value, GPM_HISTORY_CHARGE_TEXT) == 0) {
		history_type = GPM_HISTORY_CHARGE_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Time elapsed");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Cell charge");
	} else if (strcmp (value, GPM_HISTORY_TIME_FULL_TEXT) == 0) {
		history_type = GPM_HISTORY_TIME_FULL_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Time elapsed");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Predicted time");
	} else if (strcmp (value, GPM_HISTORY_TIME_EMPTY_TEXT) == 0) {
		history_type = GPM_HISTORY_TIME_EMPTY_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Time elapsed");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Predicted time");
	} else {
		g_assert (FALSE);
	}

	/* set axis */
	widget = glade_xml_get_widget (glade_xml, "label_axis_history_x");
	gtk_label_set_label (GTK_LABEL(widget), axis_x);
	widget = glade_xml_get_widget (glade_xml, "label_axis_history_y");
	gtk_label_set_label (GTK_LABEL(widget), axis_y);

	gpm_stats_button_update_ui ();
	g_free (value);

	/* save to gconf */
	gconf_client_set_string (gconf_client, GPM_CONF_INFO_HISTORY_TYPE, history_type, NULL);
}

/**
 * gpm_stats_type_combo_changed_cb:
 **/
static void
gpm_stats_type_combo_changed_cb (GtkWidget *widget, gpointer data)
{
	gchar *value;
	const gchar *axis_x = NULL;
	const gchar *axis_y = NULL;
	value = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
	if (strcmp (value, GPM_STATS_CHARGE_DATA_TEXT) == 0) {
		stats_type = GPM_STATS_CHARGE_DATA_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Cell charge");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Correction factor");
	} else if (strcmp (value, GPM_STATS_CHARGE_ACCURACY_TEXT) == 0) {
		stats_type = GPM_STATS_CHARGE_ACCURACY_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Cell charge");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Prediction accuracy");
	} else if (strcmp (value, GPM_STATS_DISCHARGE_DATA_TEXT) == 0) {
		stats_type = GPM_STATS_DISCHARGE_DATA_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Cell charge");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Correction factor");
	} else if (strcmp (value, GPM_STATS_DISCHARGE_ACCURACY_TEXT) == 0) {
		stats_type = GPM_STATS_DISCHARGE_ACCURACY_VALUE;
		/* TRANSLATORS: this is the X axis on the graph */
		axis_x = _("Cell charge");
		/* TRANSLATORS: this is the Y axis on the graph */
		axis_y = _("Prediction accuracy");
	} else {
		g_assert (FALSE);
	}

	/* set axis */
	widget = glade_xml_get_widget (glade_xml, "label_axis_stats_x");
	gtk_label_set_label (GTK_LABEL(widget), axis_x);
	widget = glade_xml_get_widget (glade_xml, "label_axis_stats_y");
	gtk_label_set_label (GTK_LABEL(widget), axis_y);

	gpm_stats_button_update_ui ();
	g_free (value);

	/* save to gconf */
	gconf_client_set_string (gconf_client, GPM_CONF_INFO_STATS_TYPE, stats_type, NULL);
}

/**
 * gpm_stats_range_combo_changed:
 **/
static void
gpm_stats_range_combo_changed (GtkWidget *widget, gpointer data)
{
	gchar *value;
	value = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
	if (strcmp (value, GPM_HISTORY_MINUTE_TEXT) == 0)
		history_time = GPM_HISTORY_MINUTE_VALUE;
	else if (strcmp (value, GPM_HISTORY_HOUR_TEXT) == 0)
		history_time = GPM_HISTORY_HOUR_VALUE;
	else if (strcmp (value, GPM_HISTORY_DAY_TEXT) == 0)
		history_time = GPM_HISTORY_DAY_VALUE;
	else if (strcmp (value, GPM_HISTORY_WEEK_TEXT) == 0)
		history_time = GPM_HISTORY_WEEK_VALUE;
	else
		g_assert (FALSE);

	/* save to gconf */
	gconf_client_set_int (gconf_client, GPM_CONF_INFO_HISTORY_TIME, history_time, NULL);

	gpm_stats_button_update_ui ();
	g_free (value);
}

/**
 * gpm_stats_smooth_checkbox_history_cb:
 * @widget: The GtkWidget object
 **/
static void
gpm_stats_smooth_checkbox_history_cb (GtkWidget *widget, gpointer data)
{
	gboolean checked;
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gconf_client_set_bool (gconf_client, GPM_CONF_INFO_HISTORY_GRAPH_SMOOTH, checked, NULL);
	gpm_stats_button_update_ui ();
}

/**
 * gpm_stats_smooth_checkbox_stats_cb:
 * @widget: The GtkWidget object
 **/
static void
gpm_stats_smooth_checkbox_stats_cb (GtkWidget *widget, gpointer data)
{
	gboolean checked;
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gconf_client_set_bool (gconf_client, GPM_CONF_INFO_STATS_GRAPH_SMOOTH, checked, NULL);
	gpm_stats_button_update_ui ();
}

/**
 * gpm_stats_points_checkbox_history_cb:
 * @widget: The GtkWidget object
 **/
static void
gpm_stats_points_checkbox_history_cb (GtkWidget *widget, gpointer data)
{
	gboolean checked;
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gconf_client_set_bool (gconf_client, GPM_CONF_INFO_HISTORY_GRAPH_POINTS, checked, NULL);
	gpm_stats_button_update_ui ();
}

/**
 * gpm_stats_points_checkbox_stats_cb:
 * @widget: The GtkWidget object
 **/
static void
gpm_stats_points_checkbox_stats_cb (GtkWidget *widget, gpointer data)
{
	gboolean checked;
	checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gconf_client_set_bool (gconf_client, GPM_CONF_INFO_STATS_GRAPH_POINTS, checked, NULL);
	gpm_stats_button_update_ui ();
}

/**
 * main:
 **/
int
main (int argc, char *argv[])
{
	gboolean verbose = FALSE;
	GOptionContext *context;
	GtkWidget *widget;
	GtkTreeSelection *selection;
	EggUnique *egg_unique;
	gboolean ret;
	DkpClient *client;
	GPtrArray *devices;
	DkpDevice *device;
	guint i;
	gint page;
	const gchar *object_path;
	gboolean checked;
	gchar *last_device;

	const GOptionEntry options[] = {
		{ "verbose", 'v', 0, G_OPTION_ARG_NONE, &verbose,
		  N_("Show extra debugging information"), NULL },
		{ NULL}
	};

	setlocale (LC_ALL, "");

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	if (! g_thread_supported ())
		g_thread_init (NULL);
	dbus_g_thread_init ();
	g_type_init ();

	context = g_option_context_new (NULL);
	g_option_context_set_summary (context, _("Power Statistics"));
	g_option_context_add_main_entries (context, options, NULL);
	g_option_context_parse (context, &argc, &argv, NULL);
	g_option_context_free (context);

	egg_debug_init (verbose);
	gtk_init (&argc, &argv);

	/* are we already activated? */
	egg_unique = egg_unique_new ();
	ret = egg_unique_assign (egg_unique, "org.freedesktop.DeviceKit.Gnome");
	if (!ret)
		goto unique_out;
	g_signal_connect (egg_unique, "activated",
			  G_CALLBACK (gpm_stats_window_activated_cb), NULL);

	/* add application specific icons to search path */
	gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (),
                                           GPM_DATA G_DIR_SEPARATOR_S "icons");

	/* get data from gconf */
	gconf_client = gconf_client_get_default ();

	/* add application specific icons to search path */
	gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (),
                                           GPM_DATA G_DIR_SEPARATOR_S "icons");

	/* use custom widgets */
	glade_set_custom_handler (gpm_stats_create_custom_widget, NULL);

	glade_xml = glade_xml_new (GPM_DATA "/gpm-statistics.glade", NULL, NULL);
	widget = glade_xml_get_widget (glade_xml, "dialog_stats");
	gtk_window_set_default_size (GTK_WINDOW(widget), 800, 500);
	gtk_window_set_default_icon_name (GPM_STOCK_APP_ICON);

	/* Get the main window quit */
	g_signal_connect_swapped (widget, "delete_event", G_CALLBACK (gtk_main_quit), NULL);

	widget = glade_xml_get_widget (glade_xml, "button_close");
	g_signal_connect_swapped (widget, "clicked", G_CALLBACK (gtk_main_quit), NULL);
	gtk_widget_grab_default (widget);

	widget = glade_xml_get_widget (glade_xml, "button_help");
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_button_help_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "button_refresh");
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_button_refresh_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "checkbutton_smooth_history");
	checked = gconf_client_get_bool (gconf_client, GPM_CONF_INFO_HISTORY_GRAPH_SMOOTH, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), checked);
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_smooth_checkbox_history_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "checkbutton_smooth_stats");
	checked = gconf_client_get_bool (gconf_client, GPM_CONF_INFO_STATS_GRAPH_SMOOTH, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), checked);
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_smooth_checkbox_stats_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "checkbutton_points_history");
	checked = gconf_client_get_bool (gconf_client, GPM_CONF_INFO_HISTORY_GRAPH_POINTS, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), checked);
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_points_checkbox_history_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "checkbutton_points_stats");
	checked = gconf_client_get_bool (gconf_client, GPM_CONF_INFO_STATS_GRAPH_POINTS, NULL);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), checked);
	g_signal_connect (widget, "clicked",
			  G_CALLBACK (gpm_stats_points_checkbox_stats_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "notebook1");
	page = gconf_client_get_int (gconf_client, GPM_CONF_INFO_PAGE_NUMBER, NULL);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (widget), page);
	g_signal_connect (widget, "switch-page",
			  G_CALLBACK (gpm_stats_notebook_changed_cb), NULL);

	/* create list stores */
	list_store_info = gtk_list_store_new (GPM_INFO_COLUMN_LAST, G_TYPE_STRING, G_TYPE_STRING);
	list_store_devices = gtk_list_store_new (GPM_DEVICES_COLUMN_LAST, G_TYPE_STRING,
						 G_TYPE_STRING, G_TYPE_STRING);
	list_store_wakeups = gtk_list_store_new (GPM_WAKEUPS_COLUMN_LAST, G_TYPE_STRING,
						 G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);

	/* create transaction_id tree view */
	widget = glade_xml_get_widget (glade_xml, "treeview_info");
	gtk_tree_view_set_model (GTK_TREE_VIEW (widget),
				 GTK_TREE_MODEL (list_store_info));

	/* add columns to the tree view */
	gpm_stats_add_info_columns (GTK_TREE_VIEW (widget));
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (widget)); /* show */

	/* create transaction_id tree view */
	widget = glade_xml_get_widget (glade_xml, "treeview_devices");
	gtk_tree_view_set_model (GTK_TREE_VIEW (widget),
				 GTK_TREE_MODEL (list_store_devices));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
	g_signal_connect (selection, "changed",
			  G_CALLBACK (gpm_stats_devices_treeview_clicked_cb), NULL);

	/* add columns to the tree view */
	gpm_stats_add_devices_columns (GTK_TREE_VIEW (widget));
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (widget)); /* show */

	/* create wakeups tree view */
	widget = glade_xml_get_widget (glade_xml, "treeview_wakeups");
	gtk_tree_view_set_model (GTK_TREE_VIEW (widget),
				 GTK_TREE_MODEL (list_store_wakeups));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));

	/* add columns to the tree view */
	gpm_stats_add_wakeups_columns (GTK_TREE_VIEW (widget));
	gtk_tree_view_columns_autosize (GTK_TREE_VIEW (widget)); /* show */

	history_type = gconf_client_get_string (gconf_client, GPM_CONF_INFO_HISTORY_TYPE, NULL);
	history_time = gconf_client_get_int (gconf_client, GPM_CONF_INFO_HISTORY_TIME, NULL);
	if (history_type == NULL)
		history_type = GPM_HISTORY_CHARGE_VALUE;
	if (history_time == 0)
		history_time = GPM_HISTORY_HOUR_VALUE;

	stats_type = gconf_client_get_string (gconf_client, GPM_CONF_INFO_STATS_TYPE, NULL);
	if (stats_type == NULL)
		stats_type = GPM_STATS_CHARGE_DATA_VALUE;

	widget = glade_xml_get_widget (glade_xml, "combobox_history_type");
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_RATE_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_CHARGE_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_TIME_FULL_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_TIME_EMPTY_TEXT);
	if (strcmp (history_type, GPM_HISTORY_RATE_VALUE) == 0)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 0);
	else
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 1);
	g_signal_connect (G_OBJECT (widget), "changed",
			  G_CALLBACK (gpm_stats_history_type_combo_changed_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "combobox_stats_type");
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_STATS_CHARGE_DATA_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_STATS_CHARGE_ACCURACY_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_STATS_DISCHARGE_DATA_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_STATS_DISCHARGE_ACCURACY_TEXT);
	if (strcmp (stats_type, GPM_STATS_CHARGE_DATA_VALUE) == 0)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 0);
	else if (strcmp (stats_type, GPM_STATS_CHARGE_DATA_VALUE) == 0)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 1);
	else if (strcmp (stats_type, GPM_STATS_CHARGE_DATA_VALUE) == 0)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 2);
	else if (strcmp (stats_type, GPM_STATS_CHARGE_DATA_VALUE) == 0)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 0);
	else
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 3);
	g_signal_connect (G_OBJECT (widget), "changed",
			  G_CALLBACK (gpm_stats_type_combo_changed_cb), NULL);

	widget = glade_xml_get_widget (glade_xml, "combobox_history_time");
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_MINUTE_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_HOUR_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_DAY_TEXT);
	gtk_combo_box_append_text (GTK_COMBO_BOX (widget), GPM_HISTORY_WEEK_TEXT);
	gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 1);
	if (history_time == GPM_HISTORY_MINUTE_VALUE)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 0);
	else if (history_time == GPM_HISTORY_HOUR_VALUE)
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 1);
	else
		gtk_combo_box_set_active (GTK_COMBO_BOX (widget), 2);
	g_signal_connect (G_OBJECT (widget), "changed",
			  G_CALLBACK (gpm_stats_range_combo_changed), NULL);

	widget = glade_xml_get_widget (glade_xml, "custom_graph_history");
	gtk_widget_set_size_request (widget, 700, 400);
	gtk_widget_show (widget);
	widget = glade_xml_get_widget (glade_xml, "custom_graph_stats");
	gtk_widget_set_size_request (widget, 700, 400);
	gtk_widget_show (widget);

	client = dkp_client_new ();
	g_signal_connect (client, "device-added", G_CALLBACK (gpm_stats_device_added_cb), NULL);
	g_signal_connect (client, "device-removed", G_CALLBACK (gpm_stats_device_removed_cb), NULL);
	g_signal_connect (client, "device-changed", G_CALLBACK (gpm_stats_device_changed_cb), NULL);

	wakeups = dkp_wakeups_new ();
	g_signal_connect (wakeups, "data-changed", G_CALLBACK (gpm_stats_data_changed_cb), NULL);

	/* coldplug */
	devices = dkp_client_enumerate_devices (client);
	if (devices == NULL)
		goto out;
	for (i=0; i < devices->len; i++) {
		device = g_ptr_array_index (devices, i);
		gpm_stats_add_device (device);
		if (i == 0) {
			gpm_stats_update_info_data (device);
			current_device = g_strdup (dkp_device_get_object_path (device));
		}
	}

	last_device = gconf_client_get_string (gconf_client, GPM_CONF_INFO_LAST_DEVICE, NULL);

	/* can we get wakeup data? */
	i = dkp_wakeups_get_total (wakeups, NULL);
	if (i > 0) {
		GtkTreeIter iter;
		gtk_list_store_append (list_store_devices, &iter);
		gtk_list_store_set (list_store_devices, &iter,
				    GPM_DEVICES_COLUMN_ID, "wakeups",
				    GPM_DEVICES_COLUMN_TEXT, _("Processor"),
				    GPM_DEVICES_COLUMN_ICON, "computer", -1);
	}

	/* set the correct focus on the last device */
	for (i=0; i < devices->len; i++) {
		object_path = (const gchar *) g_ptr_array_index (devices, i);
		if (last_device == NULL || object_path == NULL)
			break;
		if (strcmp (last_device, object_path) == 0) {
			GtkTreePath *path;
			gchar *path_str;
			path_str = g_strdup_printf ("%i", i);
			path = gtk_tree_path_new_from_string (path_str);
			widget = glade_xml_get_widget (glade_xml, "treeview_devices");
			gtk_tree_view_set_cursor_on_cell (GTK_TREE_VIEW (widget), path, NULL, NULL, FALSE);
			g_free (path_str);
			gtk_tree_path_free (path);
		}
	}

	g_ptr_array_foreach (devices, (GFunc) g_object_unref, NULL);
	g_ptr_array_free (devices, TRUE);

	/* set axis */
	widget = glade_xml_get_widget (glade_xml, "combobox_history_type");
	gpm_stats_history_type_combo_changed_cb (widget, NULL);
	widget = glade_xml_get_widget (glade_xml, "combobox_stats_type");
	gpm_stats_type_combo_changed_cb (widget, NULL);

	widget = glade_xml_get_widget (glade_xml, "dialog_stats");
	gtk_widget_show (widget);

	gtk_main ();

out:
	g_object_unref (gconf_client);
	g_object_unref (client);
	g_object_unref (wakeups);
	g_object_unref (glade_xml);
	g_object_unref (list_store_info);
unique_out:
	g_object_unref (egg_unique);
	return 0;
}
