/* GStreamer
 * Copyright (C) <2005> Jan Schmidt <jan@fluendo.com>
 * Copyright (C) <2002> Wim Taymans <wim@fluendo.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*#define DEBUG_ENABLED */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <string.h>
#include <gst/gst.h>

#define GST_TYPE_DVDSUBDEC \
          (gst_dvdsubdec_get_type())
#define GST_DVDSUBDEC(obj) \
          (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DVDSUBDEC,GstDVDSubDec))
#define GST_DVDSUBDEC_CLASS(klass) \
          (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DVDSUBDEC,GstDVDSubDec))
#define GST_IS_DVDSUBDEC(obj) \
          (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DVDSUBDEC))
#define GST_IS_DVDSUBDEC_CLASS(obj) \
          (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DVDSUBDEC))

GType gst_dvdsubdec_get_type (void);

/* Hold premultimplied colour values */
typedef struct YUVA_val
{
  guchar Y;
  guchar U;
  guchar V;
  guchar A;
} YUVA_val;

struct _GstDVDSubDec
{
  GstElement element;

  GstPad *subtitlepad;
  GstPad *srcpad;

  gint in_width, in_height;

  /* Collect together subtitle buffers until we have a full control sequence */
  GstBuffer *partialbuf;
  gboolean have_title;

  guchar subtitle_index[4];
  guchar menu_index[4];
  guchar subtitle_alpha[4];
  guchar menu_alpha[4];

  guint32 current_clut[16];
  YUVA_val palette_cache[4];
  YUVA_val hl_palette_cache[4];

  GstClockTime next_ts;

  /*
   * State info for the current subpicture
   * buffer
   */
  guchar *parse_pos;

  guint16 packet_size;
  guint16 data_size;

  gint offset[2];

  gboolean forced_display;
  gboolean visible;

  gint left, top, right, bottom;
  gint hl_left, hl_top, hl_right, hl_bottom;

  gint current_button;

  GstClockTime next_event_ts;

  GstBuffer *out_buffer;
  gboolean buf_dirty;
};

struct _GstDVDSubDecClass
{
  GstElementClass parent_class;
};

typedef struct _GstDVDSubDec GstDVDSubDec;
typedef struct _GstDVDSubDecClass GstDVDSubDecClass;

static void gst_dvdsubdec_class_init (GstDVDSubDecClass * klass);
static void gst_dvdsubdec_base_init (GstDVDSubDecClass * klass);
static void gst_dvdsubdec_init (GstDVDSubDec * dvdsubdec);

static GstCaps *gst_dvdsubdec_getcaps_video (GstPad * pad);
static GstPadLinkReturn gst_dvdsubdec_link_video (GstPad * pad,
    const GstCaps * caps);
static gboolean gst_dvdsubdec_src_event (GstPad * pad, GstEvent * event);
static void gst_dvdsubdec_handle_subtitle (GstPad * pad, GstData * _data);

static void gst_dvdsubdec_handle_dvd_event (GstDVDSubDec * dvdsubdec,
    GstEvent * event);
static void gst_dvdsubdec_finalize (GObject * gobject);
static void gst_setup_palette (GstDVDSubDec * dvdsubdec);
static void gst_dvdsubdec_merge_title (GstDVDSubDec * dvdsubdec,
    GstBuffer * buf);
static GstClockTime dvdsubdec_get_event_delay (GstDVDSubDec * dvdsubdec);

static void gst_send_subtitle_frame (GstDVDSubDec * dvdsubdec,
    GstClockTime end_ts);

/* elementfactory information */
static GstElementDetails dvdsubdec_details = {
  "DVD subtitle Decoder",
  "Codec/Decoder/Video",
  "Decodes DVD subtitles into AYUV video frames",
  "Wim Taymans <wim.taymans@chello.be>\n"
      "Jan Schmidt <thaytan@mad.scientist.com>"
};

static GstStaticPadTemplate src_template = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-yuv, format = (fourcc) AYUV, "
        "width = (int) 720, height = (int) 576")
    );

static GstStaticPadTemplate subtitle_template = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-dvd-subpicture")
    );

GST_DEBUG_CATEGORY_STATIC (dvdsubdec_debug);
#define GST_CAT_DEFAULT (dvdsubdec_debug)

/* GstDVDSubDec signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  ARG_0
      /* FILL ME */
};

enum
{
  SPU_FORCE_DISPLAY = 0x00,
  SPU_SHOW = 0x01,
  SPU_HIDE = 0x02,
  SPU_SET_PALETTE = 0x03,
  SPU_SET_ALPHA = 0x04,
  SPU_SET_SIZE = 0x05,
  SPU_SET_OFFSETS = 0x06,
  SPU_WIPE = 0x07,
  SPU_END = 0xff
};

static guint32 default_clut[16] = {
  0xb48080, 0x248080, 0x628080, 0xd78080,
  0x808080, 0x808080, 0x808080, 0x808080,
  0x808080, 0x808080, 0x808080, 0x808080,
  0x808080, 0x808080, 0x808080, 0x808080
};

typedef struct RLE_state
{
  gint id;
  gint aligned;
  gint offset[2];
  gint hl_left;
  gint hl_right;

  guchar *target;

  guchar next;
}
RLE_state;

static GstElementClass *parent_class = NULL;

GType
gst_dvdsubdec_get_type (void)
{
  static GType dvdsubdec_type = 0;

  if (!dvdsubdec_type) {
    static const GTypeInfo dvdsubdec_info = {
      sizeof (GstDVDSubDecClass),
      (GBaseInitFunc) gst_dvdsubdec_base_init,
      NULL,
      (GClassInitFunc) gst_dvdsubdec_class_init,
      NULL,
      NULL,
      sizeof (GstDVDSubDec),
      0,
      (GInstanceInitFunc) gst_dvdsubdec_init,
    };

    dvdsubdec_type =
        g_type_register_static (GST_TYPE_ELEMENT, "GstDVDSubDec",
        &dvdsubdec_info, 0);

    GST_DEBUG_CATEGORY_INIT (dvdsubdec_debug, "dvdsubdec", 0,
        "DVD subtitle decoder element");
  }

  return dvdsubdec_type;
}

static void
gst_dvdsubdec_base_init (GstDVDSubDecClass * klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&subtitle_template));

  gst_element_class_set_details (element_class, &dvdsubdec_details);
}

static void
gst_dvdsubdec_class_init (GstDVDSubDecClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->finalize = gst_dvdsubdec_finalize;
}

static void
gst_dvdsubdec_init (GstDVDSubDec * dvdsubdec)
{
  dvdsubdec->subtitlepad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&subtitle_template), "sink");
  gst_pad_set_chain_function (GST_PAD (dvdsubdec->subtitlepad),
      (GstPadChainFunction) gst_dvdsubdec_handle_subtitle);
  gst_element_add_pad (GST_ELEMENT (dvdsubdec), dvdsubdec->subtitlepad);

  dvdsubdec->srcpad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&src_template), "src");
  gst_pad_set_getcaps_function (dvdsubdec->srcpad,
      GST_DEBUG_FUNCPTR (gst_dvdsubdec_getcaps_video));
  gst_pad_set_link_function (dvdsubdec->srcpad,
      GST_DEBUG_FUNCPTR (gst_dvdsubdec_link_video));
  gst_pad_set_event_function (dvdsubdec->srcpad,
      GST_DEBUG_FUNCPTR (gst_dvdsubdec_src_event));
  gst_element_add_pad (GST_ELEMENT (dvdsubdec), dvdsubdec->srcpad);

  GST_FLAG_SET (GST_ELEMENT (dvdsubdec), GST_ELEMENT_EVENT_AWARE);

  dvdsubdec->partialbuf = NULL;
  dvdsubdec->have_title = FALSE;
  dvdsubdec->parse_pos = NULL;
  dvdsubdec->forced_display = FALSE;
  dvdsubdec->visible = FALSE;

  memset (dvdsubdec->menu_index, 0, sizeof (dvdsubdec->menu_index));
  memset (dvdsubdec->menu_alpha, 0, sizeof (dvdsubdec->menu_alpha));
  memset (dvdsubdec->subtitle_index, 0, sizeof (dvdsubdec->subtitle_index));
  memset (dvdsubdec->subtitle_alpha, 0, sizeof (dvdsubdec->subtitle_alpha));
  memcpy (dvdsubdec->current_clut, default_clut, sizeof (guint32) * 16);

  gst_setup_palette (dvdsubdec);

  dvdsubdec->next_ts = 0;
  dvdsubdec->next_event_ts = GST_CLOCK_TIME_NONE;

  dvdsubdec->out_buffer = NULL;
  dvdsubdec->buf_dirty = TRUE;
}

static void
gst_dvdsubdec_finalize (GObject * gobject)
{
  GstDVDSubDec *dvdsubdec = GST_DVDSUBDEC (gobject);

  if (dvdsubdec->partialbuf)
    gst_buffer_unref (dvdsubdec->partialbuf);
}

static GstCaps *
gst_dvdsubdec_getcaps_video (GstPad * pad)
{
  return GST_PAD_TEMPLATE_CAPS (gst_static_pad_template_get (&src_template));
}

static GstPadLinkReturn
gst_dvdsubdec_link_video (GstPad * pad, const GstCaps * caps)
{
  GstDVDSubDec *dvdsubdec = GST_DVDSUBDEC (gst_pad_get_parent (pad));
  GstStructure *structure;
  gint width, height;

  structure = gst_caps_get_structure (caps, 0);

  if (!gst_structure_get_int (structure, "width", &width) ||
      !gst_structure_get_int (structure, "height", &height)) {
    return GST_PAD_LINK_REFUSED;
  }

  dvdsubdec->in_width = width;
  dvdsubdec->in_height = height;

  return GST_PAD_LINK_OK;
}

static gboolean
gst_dvdsubdec_src_event (GstPad * pad, GstEvent * event)
{
  GstDVDSubDec *dvdsubdec = GST_DVDSUBDEC (gst_pad_get_parent (pad));

  return gst_pad_send_event (GST_PAD_PEER (dvdsubdec->subtitlepad), event);
}

static GstClockTime
dvdsubdec_get_event_delay (GstDVDSubDec * dvdsubdec)
{
  guchar *start = GST_BUFFER_DATA (dvdsubdec->partialbuf);
  guchar *buf;
  GstClockTime event_delay;

  /* If starting a new buffer, follow the first DCSQ ptr */
  if (dvdsubdec->parse_pos == start) {
    buf = dvdsubdec->parse_pos + dvdsubdec->data_size;
  } else {
    buf = dvdsubdec->parse_pos;
  }

  event_delay = GST_READ_UINT16_BE (buf) * GST_SECOND * 1024 / 90000;

  GST_DEBUG_OBJECT (dvdsubdec, "gst_dvdsubdec_get_event_delay returning delay %"
      G_GINT64_FORMAT " from offset %d\n", event_delay,
      buf - dvdsubdec->parse_pos);

  return event_delay;
}

/*
 * Parse the next event time in the current subpicture buffer, stopping
 * when time advances to the next state. 
 */
static void
gst_dvdsubdec_parse_subpic (GstDVDSubDec * dvdsubdec)
{
#define PARSE_BYTES_NEEDED(x) if ((buf+(x)) >= end) \
  { GST_WARNING("Subtitle stream broken parsing %c", *buf); \
    broken = TRUE; break; }

  guchar *start = GST_BUFFER_DATA (dvdsubdec->partialbuf);
  guchar *buf;
  guchar *end;
  gboolean broken = FALSE;
  gboolean last_seq = FALSE;
  guchar *next_seq = NULL;
  guint event_time;

  /* nothing to do if we finished this buffer already */
  if (dvdsubdec->parse_pos == NULL)
    return;

  g_return_if_fail (dvdsubdec->packet_size >= 4);

  end = start + dvdsubdec->packet_size;
  if (dvdsubdec->parse_pos == start) {
    buf = dvdsubdec->parse_pos + dvdsubdec->data_size;
  } else {
    buf = dvdsubdec->parse_pos;
  }

  g_assert (buf >= start && buf < end);

  /* If the next control sequence is at the current offset, this is 
   * the last one */
  next_seq = start + GST_READ_UINT16_BE (buf + 2);
  last_seq = (next_seq == buf);
  buf += 4;

  while ((buf < end) && (!broken)) {
    switch (*buf) {
      case SPU_FORCE_DISPLAY:  /* Forced display menu subtitle */
        dvdsubdec->forced_display = TRUE;
        dvdsubdec->buf_dirty = TRUE;
        GST_DEBUG_OBJECT (dvdsubdec, "SPU FORCE_DISPLAY");
        buf++;
        break;
      case SPU_SHOW:           /* Show the subtitle in this packet */
        dvdsubdec->visible = TRUE;
        dvdsubdec->buf_dirty = TRUE;
        GST_DEBUG_OBJECT (dvdsubdec, "SPU SHOW at %" G_GUINT64_FORMAT,
            dvdsubdec->next_event_ts);
        buf++;
        break;
      case SPU_HIDE:
        /* 02 ff (ff) is the end of the packet, hide the subpicture */
        dvdsubdec->visible = FALSE;
        dvdsubdec->buf_dirty = TRUE;

        GST_DEBUG_OBJECT (dvdsubdec, "SPU HIDE at %" G_GUINT64_FORMAT,
            dvdsubdec->next_event_ts);
        buf++;
        break;
      case SPU_SET_PALETTE:    /* palette */
        PARSE_BYTES_NEEDED (3);

        GST_DEBUG_OBJECT (dvdsubdec, "SPU SET_PALETTE");

        dvdsubdec->subtitle_index[3] = buf[1] >> 4;
        dvdsubdec->subtitle_index[2] = buf[1] & 0xf;
        dvdsubdec->subtitle_index[1] = buf[2] >> 4;
        dvdsubdec->subtitle_index[0] = buf[2] & 0xf;
        gst_setup_palette (dvdsubdec);

        dvdsubdec->buf_dirty = TRUE;
        buf += 3;
        break;
      case SPU_SET_ALPHA:      /* transparency palette */
        PARSE_BYTES_NEEDED (3);

        GST_DEBUG_OBJECT (dvdsubdec, "SPU SET_ALPHA");

        dvdsubdec->subtitle_alpha[3] = buf[1] >> 4;
        dvdsubdec->subtitle_alpha[2] = buf[1] & 0xf;
        dvdsubdec->subtitle_alpha[1] = buf[2] >> 4;
        dvdsubdec->subtitle_alpha[0] = buf[2] & 0xf;
        gst_setup_palette (dvdsubdec);

        dvdsubdec->buf_dirty = TRUE;
        buf += 3;
        break;
      case SPU_SET_SIZE:       /* image coordinates */
        PARSE_BYTES_NEEDED (7);

        dvdsubdec->left =
            CLAMP ((((unsigned int) buf[1]) << 4) | (buf[2] >> 4), 0,
            (dvdsubdec->in_width - 1));
        dvdsubdec->top =
            CLAMP ((((unsigned int) buf[4]) << 4) | (buf[5] >> 4), 0,
            (dvdsubdec->in_height - 1));
        dvdsubdec->right =
            CLAMP ((((buf[2] & 0x0f) << 8) | buf[3]), 0,
            (dvdsubdec->in_width - 1));
        dvdsubdec->bottom =
            CLAMP ((((buf[5] & 0x0f) << 8) | buf[6]), 0,
            (dvdsubdec->in_height - 1));

        GST_DEBUG_OBJECT (dvdsubdec, "SPU SET_SIZE left %d, top %d, right %d, "
            "bottom %d", dvdsubdec->left, dvdsubdec->top, dvdsubdec->right,
            dvdsubdec->bottom);

        dvdsubdec->buf_dirty = TRUE;
        buf += 7;
        break;
      case SPU_SET_OFFSETS:    /* image 1 / image 2 offsets */
        PARSE_BYTES_NEEDED (5);

        dvdsubdec->offset[0] = (((unsigned int) buf[1]) << 8) | buf[2];
        dvdsubdec->offset[1] = (((unsigned int) buf[3]) << 8) | buf[4];
        GST_DEBUG_OBJECT (dvdsubdec, "Offset1 %d, Offset2 %d",
            dvdsubdec->offset[0], dvdsubdec->offset[1]);

        dvdsubdec->buf_dirty = TRUE;
        buf += 5;
        break;
      case SPU_WIPE:
      {
        guint length;

        PARSE_BYTES_NEEDED (3);

        GST_WARNING ("SPU_WIPE not yet implemented");

        length = (buf[1] << 8) | (buf[2]);
        buf += 1 + length;

        dvdsubdec->buf_dirty = TRUE;
        break;
      }
      case SPU_END:
        buf = (last_seq) ? end : next_seq;

        /* Start a new control sequence */
        if (buf + 4 < end) {
          gint ticks = GST_READ_UINT16_BE (buf);

          event_time = GST_SECOND * ticks * 1024 / 90000;

          GST_DEBUG_OBJECT (dvdsubdec,
              "Next DCSQ at offset %d, delay %g secs (%d ticks)", buf - start,
              (gdouble) event_time / GST_SECOND, ticks);

          dvdsubdec->parse_pos = buf;
          if (event_time > 0) {
            dvdsubdec->next_event_ts += event_time;

            GST_LOG_OBJECT (dvdsubdec, "Exiting parse loop with time %g",
                (gdouble) dvdsubdec->next_event_ts / GST_SECOND);
            return;
          }
        } else {
          dvdsubdec->parse_pos = NULL;
          dvdsubdec->next_event_ts = GST_CLOCK_TIME_NONE;
          GST_LOG_OBJECT (dvdsubdec, "Finished all cmds. Exiting parse loop");
          return;
        }
      default:
        GST_ERROR
            ("Invalid sequence in subtitle packet header (%.2x). Skipping",
            *buf);
        broken = TRUE;
        dvdsubdec->parse_pos = NULL;
        break;
    }
  }
}

inline int
gst_get_nibble (guchar * buffer, RLE_state * state)
{
  if (state->aligned) {
    state->next = buffer[state->offset[state->id]++];
    state->aligned = 0;
    return state->next >> 4;
  } else {
    state->aligned = 1;
    return state->next & 0xf;
  }
}

/* Premultiply the current lookup table into the "target" cache */
static void
gst_setup_palette (GstDVDSubDec * dvdsubdec)
{
  gint i;
  guint32 col;
  YUVA_val *target = dvdsubdec->palette_cache;
  YUVA_val *target2 = dvdsubdec->hl_palette_cache;

  for (i = 0; i < 4; i++, target2++, target++) {
    col = dvdsubdec->current_clut[dvdsubdec->subtitle_index[i]];
    target->Y = (col >> 16) & 0xff;
    target->V = (col >> 8) & 0xff;
    target->U = col & 0xff;
    target->A = dvdsubdec->subtitle_alpha[i] * 0xff / 0xf;

    col = dvdsubdec->current_clut[dvdsubdec->menu_index[i]];
    target2->Y = (col >> 16) & 0xff;
    target2->V = (col >> 8) & 0xff;
    target2->U = col & 0xff;
    target2->A = dvdsubdec->menu_alpha[i] * 0xff / 0xf;
  }
}

inline guint
gst_get_rle_code (guchar * buffer, RLE_state * state)
{
  gint code;

  code = gst_get_nibble (buffer, state);
  if (code < 0x4) {             /* 4 .. f */
    code = (code << 4) | gst_get_nibble (buffer, state);
    if (code < 0x10) {          /* 1x .. 3x */
      code = (code << 4) | gst_get_nibble (buffer, state);
      if (code < 0x40) {        /* 04x .. 0fx */
        code = (code << 4) | gst_get_nibble (buffer, state);
      }
    }
  }
  return code;
}

#define DRAW_RUN(target,len,c)                  \
G_STMT_START {                                  \
  if ((c)->A) {                                 \
    gint i;                                     \
    for (i = 0; i < (len); i++) {               \
      *(target)++ = (c)->A;                     \
      *(target)++ = (c)->Y;                     \
      *(target)++ = (c)->U;                     \
      *(target)++ = (c)->V;                     \
    }                                           \
  } else {                                      \
    (target) += 4 * (len);                      \
  }                                             \
} G_STMT_END

/* 
 * This function steps over each run-length segment, drawing 
 * into the YUVA buffers as it goes. UV are composited and then output
 * at half width/height
 */
static void
gst_draw_rle_line (GstDVDSubDec * dvdsubdec, guchar * buffer, RLE_state * state)
{
  gint length, colourid;
  guint code;
  gint x, right;
  guchar *target;

  target = state->target;

  x = dvdsubdec->left;
  right = dvdsubdec->right + 1;

  while (x < right) {
    gboolean in_hl;
    const YUVA_val *colour_entry;

    code = gst_get_rle_code (buffer, state);
    length = code >> 2;
    colourid = code & 3;
    colour_entry = dvdsubdec->palette_cache + colourid;

    /* Length = 0 implies fill to the end of the line */
    /* Restrict the colour run to the end of the line */
    if (length == 0 || x + length > right)
      length = right - x;

    /* Check if this run of colour touches the highlight region */
    in_hl = ((x <= state->hl_right) && (x + length) >= state->hl_left);
    if (in_hl) {
      gint run;

      /* Draw to the left of the highlight */
      if (x <= state->hl_left) {
        run = MIN (length, state->hl_left - x + 1);

        DRAW_RUN (target, run, colour_entry);
        length -= run;
        x += run;
      }

      /* Draw across the highlight region */
      if (x <= state->hl_right) {
        const YUVA_val *hl_colour = dvdsubdec->hl_palette_cache + colourid;

        run = MIN (length, state->hl_right - x + 1);

        DRAW_RUN (target, run, hl_colour);
        length -= run;
        x += run;
      }
    }

    /* Draw the rest of the run */
    if (length > 0) {
      DRAW_RUN (target, length, colour_entry);
      x += length;
    }
  }
}

/*
 * Decode the RLE subtitle image and blend with the current
 * frame buffer.
 */
static void
gst_dvdsubdec_merge_title (GstDVDSubDec * dvdsubdec, GstBuffer * buf)
{
  gint y;
  gint Y_stride = 4 * dvdsubdec->in_width;
  guchar *buffer = GST_BUFFER_DATA (dvdsubdec->partialbuf);

  gint hl_top, hl_bottom;
  gint last_y;
  RLE_state state;

  GST_DEBUG ("Merging subtitle on frame at time %" G_GUINT64_FORMAT,
      GST_BUFFER_TIMESTAMP (buf));

  state.id = 0;
  state.aligned = 1;
  state.next = 0;
  state.offset[0] = dvdsubdec->offset[0];
  state.offset[1] = dvdsubdec->offset[1];

  if (dvdsubdec->current_button) {
    hl_top = dvdsubdec->hl_top;
    hl_bottom = dvdsubdec->hl_bottom;
  } else {
    hl_top = -1;
    hl_bottom = -1;
  }
  last_y = MIN (dvdsubdec->bottom, dvdsubdec->in_height);

  y = dvdsubdec->top;
  state.target = GST_BUFFER_DATA (buf) + 4 * dvdsubdec->left + (y * Y_stride);

  /* Now draw scanlines until we hit last_y or end of RLE data */
  for (; ((state.offset[1] < dvdsubdec->data_size + 2) && (y <= last_y)); y++) {
    /* Set up to draw the highlight if we're in the right scanlines */
    if (y > hl_bottom || y < hl_top) {
      state.hl_left = -1;
      state.hl_right = -1;
    } else {
      state.hl_left = dvdsubdec->hl_left;
      state.hl_right = dvdsubdec->hl_right;
    }
    gst_draw_rle_line (dvdsubdec, buffer, &state);

    state.target += Y_stride;

    /* Realign the RLE state for the next line */
    if (!state.aligned)
      gst_get_nibble (buffer, &state);
    state.id = !state.id;
  }
}

static void
gst_send_empty_fill (GstDVDSubDec * dvdsubdec, GstClockTime ts)
{
  if (!GST_PAD_IS_USABLE (dvdsubdec->srcpad)) {
    dvdsubdec->next_ts = ts;
    return;
  }

  if (dvdsubdec->next_ts < ts) {
    GstClockTime cur_ts = dvdsubdec->next_ts;

    GstEvent *event = gst_event_new_filler_stamped (cur_ts,
        GST_CLOCK_DIFF (ts, cur_ts));

    gst_pad_push (dvdsubdec->srcpad, GST_DATA (event));

    GST_LOG_OBJECT (dvdsubdec, "Sending padding filler, ts %"
        G_GUINT64_FORMAT ", dur %" G_GINT64_FORMAT,
        cur_ts, GST_CLOCK_DIFF (ts, cur_ts));
  }
  dvdsubdec->next_ts = ts;
}

static void
gst_send_subtitle_frame (GstDVDSubDec * dvdsubdec, GstClockTime end_ts)
{
  GstBuffer *out_buf;
  gint x, y;

  if (!GST_PAD_IS_USABLE (dvdsubdec->srcpad)) {
    dvdsubdec->next_ts = end_ts;
    return;
  }

  g_assert (dvdsubdec->have_title);
  g_assert (dvdsubdec->next_ts <= end_ts);

  /* Check if we need to redraw the output buffer */
  if (dvdsubdec->buf_dirty) {
    if (dvdsubdec->out_buffer) {
      gst_buffer_unref (dvdsubdec->out_buffer);
      dvdsubdec->out_buffer = NULL;
    }

    out_buf = gst_pad_alloc_buffer (dvdsubdec->srcpad, 0,
        4 * dvdsubdec->in_width * dvdsubdec->in_height);

    /* Clear the buffer */
    /* FIXME - move this into the buffer rendering code */
    for (y = 0; y < dvdsubdec->in_height; y++) {
      guchar *line = GST_BUFFER_DATA (out_buf) + 4 * dvdsubdec->in_width * y;

      for (x = 0; x < dvdsubdec->in_width; x++) {
        line[0] = 0;            // A
        line[1] = 16;           // Y
        line[2] = 128;          // U
        line[3] = 128;          // V

        line += 4;
      }
    }

    if (dvdsubdec->visible || dvdsubdec->forced_display) {
      gst_dvdsubdec_merge_title (dvdsubdec, out_buf);
    }

    dvdsubdec->out_buffer = out_buf;
    dvdsubdec->buf_dirty = FALSE;
  }

  out_buf = gst_buffer_create_sub (dvdsubdec->out_buffer, 0,
      GST_BUFFER_SIZE (dvdsubdec->out_buffer));

  GST_BUFFER_TIMESTAMP (out_buf) = dvdsubdec->next_ts;
  GST_BUFFER_DURATION (out_buf) = GST_CLOCK_DIFF (end_ts, dvdsubdec->next_ts);

  GST_DEBUG_OBJECT (dvdsubdec, "Sending subtitle buffer with ts %"
      G_GINT64_FORMAT " dur %" G_GINT64_FORMAT,
      GST_BUFFER_TIMESTAMP (out_buf), GST_BUFFER_DURATION (out_buf));

  gst_pad_push (dvdsubdec->srcpad, GST_DATA (out_buf));

  dvdsubdec->next_ts = end_ts;
}

/* Walk time forward, processing any subtitle events as needed. */
static void
dvdsubdec_advance_time (GstDVDSubDec * dvdsubdec, GstClockTime new_ts)
{
  GST_LOG_OBJECT (dvdsubdec, "Advancing time to %" G_GINT64_FORMAT, new_ts);

  if (!dvdsubdec->have_title) {
    gst_send_empty_fill (dvdsubdec, new_ts);
    return;
  }

  while (dvdsubdec->next_ts < new_ts) {
    GstClockTime next_ts = new_ts;

    if (GST_CLOCK_TIME_IS_VALID (dvdsubdec->next_event_ts) &&
        dvdsubdec->next_event_ts < next_ts) {
      /* We might need to process the subtitle cmd queue */
      next_ts = dvdsubdec->next_event_ts;
    }

    /* 
     * Now, either output a filler or a frame spanning
     * dvdsubdec->next_ts to next_ts
     */
    if (dvdsubdec->visible || dvdsubdec->forced_display) {
      gst_send_subtitle_frame (dvdsubdec, next_ts);
    } else {
      gst_send_empty_fill (dvdsubdec, next_ts);
    }

    /*
     * and then process some subtitle cmds if we need
     */
    if (next_ts == dvdsubdec->next_event_ts)
      gst_dvdsubdec_parse_subpic (dvdsubdec);
  }
}

static void
gst_dvdsubdec_handle_subtitle (GstPad * pad, GstData * _data)
{
  GstDVDSubDec *dvdsubdec;

  g_return_if_fail (_data != NULL);
  g_return_if_fail (pad != NULL);

  dvdsubdec = GST_DVDSUBDEC (gst_pad_get_parent (pad));
  g_return_if_fail (dvdsubdec != NULL);

  if (GST_IS_BUFFER (_data)) {
    GstBuffer *buf = GST_BUFFER (_data);
    guchar *data;
    glong size = 0;

    GST_DEBUG_OBJECT (dvdsubdec, "Have buffer of size %d, ts %"
        G_GINT64_FORMAT ", dur %" G_GINT64_FORMAT, GST_BUFFER_SIZE (buf),
        (gint64) GST_BUFFER_TIMESTAMP (buf),
        (gint64) GST_BUFFER_DURATION (buf));

    if (GST_CLOCK_TIME_IS_VALID (GST_BUFFER_TIMESTAMP (buf))) {
      if (!GST_CLOCK_TIME_IS_VALID (dvdsubdec->next_ts)) {
        dvdsubdec->next_ts = GST_BUFFER_TIMESTAMP (buf);
      }

      /* Move time forward to the start of the new buffer */
      dvdsubdec_advance_time (dvdsubdec, GST_BUFFER_TIMESTAMP (buf));
    }

    if (dvdsubdec->have_title) {
      gst_buffer_unref (dvdsubdec->partialbuf);
      dvdsubdec->partialbuf = NULL;
      dvdsubdec->have_title = FALSE;
    }

    GST_DEBUG ("Got subtitle buffer, pts %" G_GINT64_FORMAT,
        GST_BUFFER_TIMESTAMP (buf));

    /* deal with partial frame from previous buffer */
    if (dvdsubdec->partialbuf) {
      GstBuffer *merge;

      merge = gst_buffer_join (dvdsubdec->partialbuf, buf);
      dvdsubdec->partialbuf = merge;
    } else {
      dvdsubdec->partialbuf = buf;
    }

    data = GST_BUFFER_DATA (dvdsubdec->partialbuf);
    size = GST_BUFFER_SIZE (dvdsubdec->partialbuf);

    if (size > 4) {
      dvdsubdec->packet_size = GST_READ_UINT16_BE (data);

      if (dvdsubdec->packet_size == size) {
        GST_LOG ("Subtitle packet size %d, current size %ld",
            dvdsubdec->packet_size, size);

        dvdsubdec->data_size = GST_READ_UINT16_BE (data + 2);

        /*
         * Reset parameters for a new subtitle buffer
         */
        dvdsubdec->parse_pos = data;
        dvdsubdec->forced_display = FALSE;
        dvdsubdec->visible = FALSE;

        dvdsubdec->have_title = TRUE;
        dvdsubdec->next_event_ts = GST_BUFFER_TIMESTAMP (dvdsubdec->partialbuf);

        if (!GST_CLOCK_TIME_IS_VALID (dvdsubdec->next_event_ts))
          dvdsubdec->next_event_ts = dvdsubdec->next_ts;

        dvdsubdec->next_event_ts += dvdsubdec_get_event_delay (dvdsubdec);
      }
    }
  } else if (GST_IS_EVENT (_data)) {
    GstEvent *event = GST_EVENT (_data);

    switch (GST_EVENT_TYPE (GST_EVENT (_data))) {
      case GST_EVENT_ANY:{
        GstClockTime ts = GST_EVENT_TIMESTAMP (event);

        GST_LOG ("DVD event on subtitle pad with timestamp %llu",
            GST_EVENT_TIMESTAMP (event));

        if (GST_CLOCK_TIME_IS_VALID (ts))
          dvdsubdec_advance_time (dvdsubdec, ts);

        gst_dvdsubdec_handle_dvd_event (dvdsubdec, GST_EVENT (_data));
        // dvdsubdec_advance_time (dvdsubdec, dvdsubdec->next_ts + GST_SECOND / 30.0);

        gst_data_unref (_data);
        break;
      }
      case GST_EVENT_FILLER:{
        GstClockTime ts = dvdsubdec->next_ts;

        if (GST_CLOCK_TIME_IS_VALID (GST_EVENT_TIMESTAMP (event))) {
          ts = GST_EVENT_TIMESTAMP (event);
        }

        if (GST_CLOCK_TIME_IS_VALID (gst_event_filler_get_duration (event))) {
          ts += gst_event_filler_get_duration (event);
        }

        GST_DEBUG ("Got filler, advancing time from %" G_GINT64_FORMAT
            " to %" G_GINT64_FORMAT, dvdsubdec->next_ts, ts);

        dvdsubdec_advance_time (dvdsubdec, ts);

        gst_data_unref (_data);
        break;
      }
      case GST_EVENT_DISCONTINUOUS:{
        gint64 ts;

        /* Turn off forced highlight display */
        // dvdsubdec->forced_display = 0;
        // dvdsubdec->current_button = 0;
        if (dvdsubdec->partialbuf) {
          gst_buffer_unref (dvdsubdec->partialbuf);
          dvdsubdec->partialbuf = NULL;
          dvdsubdec->have_title = FALSE;
        }

        if (gst_event_discont_get_value (event, GST_FORMAT_TIME, &ts))
          dvdsubdec->next_ts = ts;
        else
          dvdsubdec->next_ts = GST_CLOCK_TIME_NONE;

        GST_DEBUG_OBJECT (dvdsubdec, "Got discont, new time = %"
            G_GINT64_FORMAT, dvdsubdec->next_ts);

        gst_pad_event_default (dvdsubdec->subtitlepad, event);
        break;
      }
      case GST_EVENT_FLUSH:{
        /* Turn off forced highlight display */
        dvdsubdec->forced_display = 0;
        dvdsubdec->current_button = 0;

        if (dvdsubdec->partialbuf) {
          gst_buffer_unref (dvdsubdec->partialbuf);
          dvdsubdec->partialbuf = NULL;
          dvdsubdec->have_title = FALSE;
        }

        gst_pad_event_default (dvdsubdec->subtitlepad, event);
        break;
      }
      default:
        gst_pad_event_default (dvdsubdec->subtitlepad, event);
        break;
    }
  } else
    gst_data_unref (_data);
}

static void
gst_dvdsubdec_handle_dvd_event (GstDVDSubDec * dvdsubdec, GstEvent * event)
{
  GstStructure *structure;
  const gchar *event_type;

  structure = event->event_data.structure.structure;

  event_type = gst_structure_get_string (structure, "event");
  g_return_if_fail (event_type != NULL);

  if (!strcmp (event_type, "dvd-spu-highlight")) {
    gint button;
    gint palette, sx, sy, ex, ey;
    gint i;

    /* Details for the highlight region to display */
    if (!gst_structure_get_int (structure, "button", &button) ||
        !gst_structure_get_int (structure, "palette", &palette) ||
        !gst_structure_get_int (structure, "sx", &sx) ||
        !gst_structure_get_int (structure, "sy", &sy) ||
        !gst_structure_get_int (structure, "ex", &ex) ||
        !gst_structure_get_int (structure, "ey", &ey)) {
      GST_ERROR ("Invalid dvd-spu-highlight event received");
      return;
    }
    dvdsubdec->current_button = button;
    dvdsubdec->hl_left = sx;
    dvdsubdec->hl_top = sy;
    dvdsubdec->hl_right = ex;
    dvdsubdec->hl_bottom = ey;
    for (i = 0; i < 4; i++) {
      dvdsubdec->menu_alpha[i] = ((guint32) (palette) >> (i * 4)) & 0x0f;
      dvdsubdec->menu_index[i] = ((guint32) (palette) >> (16 + (i * 4))) & 0x0f;
    }

    GST_DEBUG ("New button activated highlight=(%d,%d) to (%d,%d) "
        "palette 0x%x", sx, sy, ex, ey, palette);
    gst_setup_palette (dvdsubdec);

    dvdsubdec->buf_dirty = TRUE;
  } else if (!strcmp (event_type, "dvd-spu-clut-change")) {
    /* Take a copy of the colour table */
    gchar name[16];
    int i;
    gint value;

    GST_LOG ("New colour table recieved");
    for (i = 0; i < 16; i++) {
      sprintf (name, "clut%02d", i);
      if (!gst_structure_get_int (structure, name, &value)) {
        GST_ERROR ("dvd-spu-clut-change event did not contain %s field", name);
        break;
      }
      dvdsubdec->current_clut[i] = (guint32) (value);
    }

    gst_setup_palette (dvdsubdec);

    dvdsubdec->buf_dirty = TRUE;
  } else if (!strcmp (event_type, "dvd-spu-stream-change")
      || !strcmp (event_type, "dvd-spu-reset-highlight")) {
    /* Turn off forced highlight display */
    dvdsubdec->current_button = 0;

    GST_LOG ("Clearing button state");
    dvdsubdec->buf_dirty = TRUE;
  } else if (!strcmp (event_type, "dvd-spu-still-frame")) {
    /* Handle a still frame */
    GST_LOG ("Received still frame notification");
  } else {
    /* Ignore all other unknown events */
    GST_LOG ("Ignoring DVD event %s from %s pad", event_type);
  }
}

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dvdsubdec", GST_RANK_PRIMARY,
      GST_TYPE_DVDSUBDEC);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "dvdsubdec",
    "Decode DVD subtitles to AYUV video frames", plugin_init,
    VERSION, "LGPL", GST_PACKAGE, GST_ORIGIN)
