/* $Id: gdict-pref-dialog.c,v 1.63 2005/01/18 19:33:52 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict preferences window
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include "gdict-pref-dialog.h"
#include "gdict-app.h"
#include "gdict-utils.h"
#include "gdict-speller.h"
#include "gdict-web-search.h"

static void gdict_pref_dialog_class_init (GDictPrefDialogClass *class);
static void gdict_pref_dialog_init (GDictPrefDialog *pref_dialog);
static void gdict_pref_dialog_finalize (GObject *object);

static void pref_dialog_add_db      (GDictPrefDialog *pref_dialog, 
                                     gchar *db, gchar *desc);
static void pref_dialog_add_strat   (GDictPrefDialog *pref_dialog, 
                                     gchar *strat, gchar *desc);
static void pref_dialog_reset_db    (GDictPrefDialog *pref_dialog);
static void pref_dialog_reset_strat (GDictPrefDialog *pref_dialog);

static void pref_set_strat_cb       (GtkWidget *widget, gpointer data);
static void pref_set_db_cb          (GtkWidget *widget, gpointer data);

static void pref_error_cb           (dict_command_t *command, 
                                     DictStatusCode code, 
                                     gchar *message, gpointer data);
static void pref_status_cb          (dict_command_t *command, 
                                     DictStatusCode code, 
                                     int num_found, gpointer data);
static void pref_data_cb            (dict_command_t *command, dict_res_t *res,
                                     gpointer data);

static void response_cb		    (GtkDialog *dialog, gint response);
static gboolean	set_server_cb 	    (GtkWidget *widget, GdkEventFocus *event, gpointer data);
static gboolean set_port_cb	    (GtkWidget *widget, GdkEventFocus *event, gpointer data);
static void set_default_server      (GtkButton *button, gpointer data);
static void set_default_port        (GtkButton *button, gpointer data);

enum {
  SOCKET_ERROR_SIGNAL,
  LAST_SIGNAL
};

enum {
	ENABLED = 0,
	WEBSITE,
	URL,
	POINTER_COLUMN,
	NUM_COLUMNS
};

static gint gdict_pref_dialog_signals[LAST_SIGNAL] = { 0 };

static GtkDialogClass *parent_class = NULL;

GtkWidget *error_label1 = NULL;
GtkWidget *error_label2 = NULL;
GtkWidget *alignment1 = NULL;
GtkWidget *alignment2 = NULL;

GType
gdict_pref_dialog_get_type (void)
{
  static GType gdict_pref_dialog_type = 0;
    
  g_type_init ();
    
  if (!gdict_pref_dialog_type) {
    static const GTypeInfo gdict_pref_dialog_info = {
      sizeof (GDictPrefDialogClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gdict_pref_dialog_class_init,
      NULL,
      NULL,
      sizeof (GDictPrefDialog),
      0,
      (GInstanceInitFunc) gdict_pref_dialog_init,
    };
        
    gdict_pref_dialog_type = g_type_register_static (GTK_TYPE_DIALOG, 
                                                     "GDictPrefDialog", 
                                                     &gdict_pref_dialog_info,
                                                     0);
  }
    
  return gdict_pref_dialog_type;
}

static void 
gdict_pref_dialog_class_init (GDictPrefDialogClass *class) 
{
  GObjectClass *object_class;
    
  object_class = G_OBJECT_CLASS (class);
  parent_class = g_type_class_peek_parent (class);

  gdict_pref_dialog_signals[SOCKET_ERROR_SIGNAL] =
    g_signal_new ("socket_error",
                  G_TYPE_FROM_CLASS (object_class),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (GDictPrefDialogClass, socket_error),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE,
                  1,
                  G_TYPE_STRING);

  object_class->finalize = gdict_pref_dialog_finalize;
}

static web_site_edited (GtkCellRendererText *cell,
                        gchar *path_str, gchar *new_text,
                        gpointer data)
{
	GDictPrefDialog *pref_dialog = data;
	GDictWindow *gdict = pref_dialog->gdict;
	GtkTreeModel *model;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
	GtkTreeIter iter;
	GDictWebSearchSite *site;
	gint column = (gint) g_object_get_data (G_OBJECT (cell), "column");
	
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (pref_dialog->view));
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, POINTER_COLUMN, &site, -1);

	if (column == WEBSITE) {
		g_free (site->name);
		site->name = g_strdup_printf ("%s", new_text);
	} else {
		g_free (site->url);
		site->url = g_strdup_printf ("%s", new_text);
	}
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, column, new_text, -1);
	
}

static web_site_toggled (GtkCellRendererToggle *cell,
                         gchar                 *path_str,
                         gpointer               data)
{
	GDictPrefDialog *pref_dialog = data;
	GDictWindow *gdict = pref_dialog->gdict;
	GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean toggle_item;
	GDictWebSearchSite *site;
	
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (pref_dialog->view));
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, ENABLED, &toggle_item, POINTER_COLUMN, &site, -1);

	toggle_item ^= 1;
	site->enabled = toggle_item;

	gtk_cell_renderer_toggle_set_active (cell, toggle_item);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, ENABLED, toggle_item, -1);
}

static web_site_add (GtkButton *widget, gpointer data)
{
	GDictWebSearchSite *site;
	GDictPrefDialog *pref_dialog = data;
	GDictWindow *gdict = pref_dialog->gdict;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreePath *path;
	GtkTreeViewColumn *column;

	site = gdict_web_search_new ("", "http://", TRUE);
	gdict->web_search_sites = g_slist_prepend (gdict->web_search_sites, site);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (pref_dialog->view));
	gtk_list_store_prepend (GTK_LIST_STORE(model), &iter);
	gtk_list_store_set (GTK_LIST_STORE(model), &iter, 
                      ENABLED, site->enabled,
                      WEBSITE, site->name,
                      URL, site->url, 
                      POINTER_COLUMN, site,
                      -1);

	path = gtk_tree_model_get_path (model, &iter);
	column = gtk_tree_view_get_column (GTK_TREE_VIEW (pref_dialog->view), WEBSITE);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (pref_dialog->view), path, column, TRUE);
	gtk_tree_path_free (path);
}

static web_site_remove (GtkButton *widget, gpointer data)
{
	GDictPrefDialog *pref_dialog = data;
	GDictWindow *gdict = pref_dialog->gdict;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GDictWebSearchSite *site;

	model = gtk_tree_view_get_model (GTK_TREE_VIEW (pref_dialog->view));
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pref_dialog->view));
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, POINTER_COLUMN, &site, -1);
	gdict_web_search_remove (gdict, site);
	gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
}

static web_site_selected (GtkTreeSelection *selection, gpointer data)
{
	GDictPrefDialog *pref_dialog = data;
	if (gtk_tree_selection_count_selected_rows (selection) > 0) 
		gtk_widget_set_sensitive (pref_dialog->remove_button, TRUE);
	else
		gtk_widget_set_sensitive (pref_dialog->remove_button, FALSE);
}

static GtkWidget *
create_dictionary_panel (GDictPrefDialog *pref_dialog)
{
  GtkWidget *vbox, *hbox;

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_set_spacing (GTK_BOX (vbox), 2);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
    
  pref_dialog->table = GTK_TABLE (gtk_table_new (4, 2, FALSE));
  gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (pref_dialog->table), FALSE, FALSE, 0);
  gtk_table_set_row_spacings (pref_dialog->table, 6);
  gtk_table_set_col_spacings (pref_dialog->table, 12);
    
  hbox = gtk_hbox_new (FALSE, 6);
  pref_dialog->server_entry = GTK_ENTRY (gtk_entry_new ());	
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET(pref_dialog->server_entry), TRUE, TRUE, 0);
  g_signal_connect (G_OBJECT (pref_dialog->server_entry), "focus_out_event",
                    G_CALLBACK (set_server_cb), pref_dialog);

  pref_dialog->server_button = gtk_button_new_with_mnemonic (_("De_fault Server"));
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET(pref_dialog->server_button), FALSE, FALSE, 0);
  g_signal_connect (G_OBJECT (pref_dialog->server_button), "clicked",
                    G_CALLBACK (set_default_server), pref_dialog);

  gtk_table_attach (pref_dialog->table, 
                    hbox,
                    1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);
		    
  pref_dialog->server_label = gtk_label_new_with_mnemonic (_("_Server:"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (pref_dialog->server_label), 
                                 GTK_WIDGET (pref_dialog->server_entry));
  gtk_label_set_justify (GTK_LABEL (pref_dialog->server_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (pref_dialog->server_label), 0, 0.5);
  gtk_table_attach_defaults (pref_dialog->table, pref_dialog->server_label, 0, 1, 0, 1);
    
  if (gail_loaded)
    gdict_add_atk_namedesc (GTK_WIDGET (pref_dialog->server_button), NULL, _("Reset server to default"));

  pref_dialog->port_label = gtk_label_new_with_mnemonic (_("_Port:"));
  gtk_label_set_justify (GTK_LABEL (pref_dialog->port_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (pref_dialog->port_label), 0, 0.5);
  gtk_table_attach_defaults (pref_dialog->table, pref_dialog->port_label, 0, 1, 1, 2);

  pref_dialog->port_entry = GTK_ENTRY (gtk_entry_new ());
  gtk_label_set_mnemonic_widget (GTK_LABEL (pref_dialog->port_label), 
                                 GTK_WIDGET (pref_dialog->port_entry));
  g_signal_connect (G_OBJECT (pref_dialog->port_entry), "focus_out_event",
                    G_CALLBACK (set_port_cb), pref_dialog);
  pref_dialog->port_button = gtk_button_new_with_mnemonic (_("Def_ault Port"));
  g_signal_connect (G_OBJECT (pref_dialog->port_button), "clicked",
                    G_CALLBACK (set_default_port), pref_dialog);

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET(pref_dialog->port_entry), TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), pref_dialog->port_button, FALSE, FALSE, 0);
  gtk_table_attach (pref_dialog->table, 
                    hbox,
                    1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (0), 0, 0);


  if (gail_loaded) {
    gdict_add_atk_namedesc (GTK_WIDGET(pref_dialog->port_button), NULL, _("Reset port to default"));
    gdict_add_atk_namedesc (pref_dialog->server_label, _("Server"), _("Server"));
    gdict_add_atk_namedesc (GTK_WIDGET (pref_dialog->server_entry),
                            _("Server Entry"), _("Enter the Server Name"));
    gdict_add_atk_namedesc (pref_dialog->port_label, _("Port"), _("Port"));
    gdict_add_atk_namedesc (GTK_WIDGET (pref_dialog->port_entry),
                            _("Port Entry"), _("Enter the Port Number"));
  }

  pref_dialog->db_label = gtk_label_new_with_mnemonic (_("_Database:"));
  gtk_label_set_justify (GTK_LABEL (pref_dialog->db_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (pref_dialog->db_label), 0, 0.5);
  gtk_table_attach_defaults (pref_dialog->table, pref_dialog->db_label, 0, 1, 2, 3);
  pref_dialog->db_list = GTK_MENU (gtk_menu_new ());
  gtk_widget_show (GTK_WIDGET (pref_dialog->db_list));
    
  pref_dialog->strat_label = gtk_label_new_with_mnemonic (_("Strat_egy:"));
  gtk_label_set_justify (GTK_LABEL (pref_dialog->strat_label), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (pref_dialog->strat_label), 0, 0.5);
  gtk_table_attach_defaults (pref_dialog->table, pref_dialog->strat_label, 0, 1, 3, 4);
  pref_dialog->strat_list = GTK_MENU (gtk_menu_new ());
  gtk_widget_show (GTK_WIDGET (pref_dialog->strat_list));
    
  gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));

  return vbox;
}

static GtkWidget *
create_websites_panel (GDictPrefDialog *pref_dialog)
{
	int i;

	GtkWidget *vbox, *hbox;
	GtkWidget *button;
	GtkWidget *scrolled;
	GtkListStore *store;
	GtkTreeSelection *selection;
	GtkWidget *view;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
		
	const gchar *column_titles[] = { NULL, N_("Web Site"), N_("Search Address"), NULL };

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);

	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled), 
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
	                                     GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (vbox), scrolled, TRUE, TRUE, 0);

	store = gtk_list_store_new (NUM_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING,
                              G_TYPE_POINTER, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);	
	view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (view), TRUE);
	g_object_unref (G_OBJECT (store));
	
	renderer = gtk_cell_renderer_toggle_new ();
	g_object_set (G_OBJECT (renderer), "activatable", TRUE, NULL);
	g_signal_connect (renderer, "toggled", G_CALLBACK (web_site_toggled), pref_dialog);

	column = gtk_tree_view_column_new_with_attributes (_(column_titles[0]), renderer, 
                                                     "active", ENABLED, NULL);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_column_set_resizable (column, FALSE);
	gtk_tree_view_column_set_clickable (column, FALSE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

	for (i=1; column_titles[i]; i++) {
		renderer = gtk_cell_renderer_text_new ();
		g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
		g_object_set_data (G_OBJECT (renderer), "column", (gint *) i);
		g_signal_connect (renderer, "edited", G_CALLBACK (web_site_edited), pref_dialog);
		column = gtk_tree_view_column_new_with_attributes (_(column_titles[i]),
                                                       renderer,
                                                       "text", i,
                                                       NULL);
		gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
		gtk_tree_view_column_set_resizable (column, FALSE);
		gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);
  }
	gtk_widget_set_size_request (GTK_WIDGET (view), 400,200);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (view));

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
	g_signal_connect (selection, "changed", G_CALLBACK(web_site_selected), pref_dialog);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

	hbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 5);
	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	g_signal_connect (button, "clicked", G_CALLBACK (web_site_add), pref_dialog);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
	gtk_widget_set_sensitive (button, FALSE);
	g_signal_connect (button, "clicked", G_CALLBACK (web_site_remove), pref_dialog);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	
	gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	pref_dialog->view = view;
	pref_dialog->remove_button = button;

	return vbox;
}

static void
gdict_pref_dialog_set (GDictPrefDialog *pref_dialog)
{
	gchar *port_str;
	GDictPref *pref = pref_dialog->gdict->pref;
	GDictWindow *gdict = pref_dialog->gdict;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GSList *site_list;
	GDictWebSearchSite *site;

	if (pref->server != NULL)
		gtk_entry_set_text (GTK_ENTRY (pref_dialog->server_entry),
                        pref->server);
	port_str = g_strdup_printf("%d", (int) pref->port);
	gtk_entry_set_text (GTK_ENTRY (pref_dialog->port_entry), port_str);
	g_free(port_str);
  if (! gdict_is_gconf_key_writable (pref_dialog->gdict, "port")) {
    gtk_widget_set_sensitive (GTK_WIDGET (pref_dialog->port_entry), FALSE);
    gtk_widget_set_sensitive (pref_dialog->port_label, FALSE);
    gtk_widget_set_sensitive (pref_dialog->port_button, FALSE);
  }
  if (! gdict_is_gconf_key_writable (pref_dialog->gdict, "server")) {
    gtk_widget_set_sensitive (GTK_WIDGET (pref_dialog->server_entry), FALSE);
    gtk_widget_set_sensitive (pref_dialog->server_label, FALSE);
    gtk_widget_set_sensitive (pref_dialog->server_button, FALSE);
  }

	/* add the websites to the list */
	
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (pref_dialog->view));
	for (site_list = gdict->web_search_sites; site_list; site_list = g_slist_next (site_list)) {
		site = site_list->data;
		gtk_list_store_prepend (GTK_LIST_STORE(model), &iter);
		gtk_list_store_set (GTK_LIST_STORE(model), &iter, 
                        ENABLED, site->enabled,
                        WEBSITE, site->name,
                        URL, site->url, 
                        POINTER_COLUMN, site,
                        -1);
	}

}

static void 
gdict_pref_dialog_init (GDictPrefDialog *pref_dialog)
{
  GtkWidget *vbox;
  GtkWidget *notebook;
  GtkWidget *label;

  gtk_window_set_title (GTK_WINDOW (pref_dialog), _("Dictionary Preferences"));
  gtk_window_set_resizable (GTK_WINDOW (pref_dialog), TRUE);
  gtk_container_set_border_width (GTK_CONTAINER (pref_dialog), 5);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (pref_dialog)->vbox), 2);

  notebook = gtk_notebook_new ();
 
  vbox = create_dictionary_panel (pref_dialog);
  label = gtk_label_new (_("Dictionary"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, label);

  vbox = create_websites_panel (pref_dialog);
  label = gtk_label_new (_("Web Sites"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, label);

  gtk_container_add (GTK_CONTAINER(GTK_DIALOG (pref_dialog)->vbox), notebook);
  gtk_container_set_border_width (GTK_CONTAINER (notebook), 5);

  gtk_dialog_add_buttons (GTK_DIALOG (pref_dialog), GTK_STOCK_CLOSE,
                          GTK_RESPONSE_CLOSE, GTK_STOCK_HELP, GTK_RESPONSE_HELP, NULL);  

  gtk_dialog_set_default_response (GTK_DIALOG (pref_dialog), GTK_RESPONSE_CLOSE);
  gtk_dialog_set_has_separator (GTK_DIALOG (pref_dialog), FALSE);
    
  g_signal_connect (G_OBJECT (pref_dialog), "response",
                    G_CALLBACK (response_cb), NULL);

  /* To prevent dialog desctruction when pressing ESC */
  g_signal_connect (G_OBJECT (pref_dialog), "delete_event", 
                    G_CALLBACK (gtk_true), NULL);
}

static void
gdict_pref_dialog_finalize (GObject *object)
{
  GDictPrefDialog *pref_dialog;

  g_return_if_fail (GDICT_IS_PREF_DIALOG (object));

  pref_dialog = GDICT_PREF_DIALOG (object);
  dict_command_destroy (pref_dialog->get_db_cmd);
  dict_command_destroy (pref_dialog->get_strat_cmd);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
gdict_pref_dialog_new (GDictWindow *gdict)
{
  GDictPrefDialog *pref_dialog;
  dict_context_t *context = gdict->context;
    
  if (context == NULL)
    return NULL;

  pref_dialog = GDICT_PREF_DIALOG (g_object_new (GDICT_TYPE_PREF_DIALOG, NULL));
  pref_dialog->gdict = gdict;
  pref_dialog->context = context;
  gdict_pref_dialog_set (pref_dialog);

  pref_dialog_reset_db (pref_dialog);
  pref_dialog_reset_strat (pref_dialog);
    
  gtk_widget_show_all (GTK_WIDGET (pref_dialog));

  return GTK_WIDGET (pref_dialog);
}

/* pref_set_db_cb
 *
 * Sets the current search database to the one indicated
 */

static void
pref_set_db_cb (GtkWidget *widget, gpointer data) 
{
  GDictPrefDialog *pref_dialog;
  GDictWindow *gdict;

  pref_dialog = GDICT_PREF_DIALOG (data);
  gdict = pref_dialog->gdict;
  gdict->pref->database = g_object_get_data (G_OBJECT (widget), "db_name");

  gdict_set_gconf_string (gdict, "database", gdict->pref->database);
  if (gdict->speller_visible)
    gdict_speller_change_db (gdict->speller, gdict->pref->database);
}

/* pref_set_strat_cb
 *
 * Sets the current search strategy to the one indicated
 */

static void
pref_set_strat_cb (GtkWidget *widget, gpointer data) 
{
  GDictPrefDialog *pref_dialog;
  GDictWindow *gdict;
    
  pref_dialog = GDICT_PREF_DIALOG (data);
  gdict = pref_dialog->gdict;
  gdict->pref->dfl_strat = g_object_get_data (G_OBJECT (widget), "strat_name");

  gdict_set_gconf_string (gdict, "strategy", gdict->pref->dfl_strat);
  if (gdict->speller_visible)
    gdict_speller_change_strategy (gdict->speller, gdict->pref->dfl_strat);
}

/* pref_dialog_add_db
 *
 * Adds a database to the database list
 */

static void
pref_dialog_add_db (GDictPrefDialog *pref_dialog, gchar *db,
                    gchar *desc) 
{
  GtkWidget *menu_item;
  GDictWindow *gdict = pref_dialog->gdict;
    
  menu_item = gtk_menu_item_new_with_label (desc);
  g_signal_connect (G_OBJECT (menu_item), "activate", 
                    G_CALLBACK (pref_set_db_cb), pref_dialog);
  g_object_set_data (G_OBJECT (menu_item), "db_name", db);
  gtk_widget_show (menu_item);
  gtk_menu_append (pref_dialog->db_list, menu_item);

  if ((gdict->pref->database == NULL)
      || (!strcmp (gdict->pref->database, db)))
    gtk_menu_set_active (pref_dialog->db_list, pref_dialog->database_idx);
  pref_dialog->database_idx++;
}

/* pref_dialog_add_strat
 *
 * Adds a search strategy to the search strategy list
 */

static void
pref_dialog_add_strat (GDictPrefDialog *pref_dialog, gchar *strat,
                       gchar *desc) 
{
  GtkWidget *menu_item;
  GDictWindow *gdict = pref_dialog->gdict;
    
  menu_item = gtk_menu_item_new_with_label (desc);
  g_signal_connect (G_OBJECT (menu_item), "activate", 
                    G_CALLBACK (pref_set_strat_cb), pref_dialog);
  g_object_set_data (G_OBJECT (menu_item), "strat_name", strat);
  gtk_widget_show (menu_item);
  gtk_menu_append (pref_dialog->strat_list, menu_item);
    
  if ((gdict->pref->dfl_strat == NULL)
      || (!strcmp (gdict->pref->dfl_strat, strat)))
    gtk_menu_set_active (pref_dialog->strat_list,
                         pref_dialog->dfl_strat_idx);
  pref_dialog->dfl_strat_idx++;
}

/* pref_dialog_reset_db
 *
 * Resets the database option menu and begins a command to retrieve it from
 * the server again
 */

static void
pref_dialog_reset_db (GDictPrefDialog *pref_dialog)
{
  if (pref_dialog->get_db_cmd)
    dict_command_destroy (pref_dialog->get_db_cmd);
    
  if (pref_dialog->db_sel) {
    gtk_option_menu_remove_menu (pref_dialog->db_sel);
    gtk_widget_destroy (GTK_WIDGET (pref_dialog->db_sel));
    gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));
    pref_dialog->db_sel = NULL;
  }

  pref_dialog->db_list = GTK_MENU (gtk_menu_new ());
    
  pref_dialog->get_db_cmd = dict_show_db_command_new ();
  pref_dialog->get_db_cmd->error_notify_cb = pref_error_cb;
  pref_dialog->get_db_cmd->data_notify_cb = pref_data_cb;
  pref_dialog->get_db_cmd->status_notify_cb = pref_status_cb;
  pref_dialog->get_db_cmd->user_data = pref_dialog;
  pref_dialog->database_idx = 0;

  if ((alignment1) && (error_label1)) {
    gtk_container_remove (GTK_CONTAINER (alignment1), error_label1);
    gtk_widget_destroy (GTK_WIDGET (alignment1));
    error_label1 = NULL;
    alignment1 = NULL;
  }
    
  pref_dialog_add_db (pref_dialog, "*", _("Search all databases"));

  if (dict_command_invoke (pref_dialog->get_db_cmd,
                           pref_dialog->context) == -1)
    {
      /* Could not look up search strategies, so just display a
       * label; FIXME: Memory leak
       */
      error_label1 = GTK_WIDGET (gtk_label_new (_("Cannot connect to server")));
      alignment1 = GTK_WIDGET (gtk_alignment_new (0, 0.5, 0, 0));
      gtk_container_add (GTK_CONTAINER (alignment1), error_label1);
      gtk_table_attach_defaults (pref_dialog->table, alignment1, 1, 2, 2, 3);
      gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));
    }
}

/* pref_dialog_reset_strat
 *
 * Resets the strategies option menu and begins a command to retrieve it from
 * the server again
 */

static void
pref_dialog_reset_strat (GDictPrefDialog *pref_dialog)
{
  if (pref_dialog->get_strat_cmd)
    dict_command_destroy (pref_dialog->get_strat_cmd);
    
  if (pref_dialog->strat_sel) {
    gtk_option_menu_remove_menu (pref_dialog->strat_sel);
    gtk_widget_destroy (GTK_WIDGET (pref_dialog->strat_sel));
    gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));
    pref_dialog->strat_sel = NULL;
  }
    
  pref_dialog->strat_list = GTK_MENU (gtk_menu_new ());
    
  pref_dialog->get_strat_cmd = dict_show_strat_command_new ();
  pref_dialog->get_strat_cmd->error_notify_cb = pref_error_cb;
  pref_dialog->get_strat_cmd->data_notify_cb = pref_data_cb;
  pref_dialog->get_strat_cmd->status_notify_cb = pref_status_cb;
  pref_dialog->get_strat_cmd->user_data = pref_dialog;
  pref_dialog->dfl_strat_idx = 0;

  if ((alignment2) && (error_label2)) {
    gtk_container_remove (GTK_CONTAINER (alignment2), error_label2);
    gtk_widget_destroy (GTK_WIDGET (alignment2));
    error_label2 = NULL;
    alignment2 = NULL;
  }
    
  if (dict_command_invoke (pref_dialog->get_strat_cmd,
                           pref_dialog->context) == -1) 
    {
      /* Could not look up search strategies, so just display a
       * label; FIXME: Memory leak
       */
      error_label2 = GTK_WIDGET (gtk_label_new (_("Cannot connect to server")));
      alignment2 = GTK_WIDGET (gtk_alignment_new (0, 0.5, 0, 0));
      gtk_container_add (GTK_CONTAINER (alignment2), error_label2);
      gtk_table_attach_defaults (pref_dialog->table, alignment2, 1, 2, 3, 4);
      gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));
    }
}

/* pref_error_cb
 *
 * Callback used when there is a socket error
 */

static void
pref_error_cb (dict_command_t *command, DictStatusCode code, 
               gchar *message, gpointer data)
{
  GDictPrefDialog *pref_dialog;
    
  g_return_if_fail (data != NULL);
  g_return_if_fail (GDICT_IS_PREF_DIALOG (data));
    
  pref_dialog = GDICT_PREF_DIALOG (data);
    
  if (code != DICT_SOCKET_ERROR) {
    GtkWidget *dialog;
    dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                  	 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                                  	 "%s", message); 
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
  }
  else {
    g_signal_emit (G_OBJECT (pref_dialog),
                   gdict_pref_dialog_signals[SOCKET_ERROR_SIGNAL], 0,
                   message);
  }
}

/* pref_data_cb
 *
 * Callback used when a new database or strategy definition has arrived 
 * over the link
 */

static void
pref_data_cb (dict_command_t *command, dict_res_t *res, gpointer data)
{
  GDictPrefDialog *pref_dialog;
    
  g_return_if_fail (data != NULL);
  g_return_if_fail (GDICT_IS_PREF_DIALOG (data));
    
  pref_dialog = GDICT_PREF_DIALOG (data);
  if (command->cmd == C_SHOW_DB)
    pref_dialog_add_db (pref_dialog, res->name, res->desc);
  else if (command->cmd == C_SHOW_STRAT)
    pref_dialog_add_strat (pref_dialog, res->name, res->desc);
}

/* pref_status_cb
 *
 * Callback used when a status code has arrived over the link
 */

static void 
pref_status_cb (dict_command_t *command, DictStatusCode code, 
                int num_found, gpointer data)
{
  GDictPrefDialog *pref_dialog;
  GtkWidget *alignment;
  GtkOptionMenu *option_menu = NULL;
  GtkMenu *use_menu = NULL;
  gint row = 0;
  gboolean writable;
    
  g_return_if_fail (data != NULL);
  g_return_if_fail (GDICT_IS_PREF_DIALOG (data));
    
  pref_dialog = GDICT_PREF_DIALOG (data);
    
  if (code == DICT_STATUS_OK) {
    if (command->cmd == C_SHOW_DB) {
      row = 2;
      use_menu = pref_dialog->db_list;
    }
    else if (command->cmd == C_SHOW_STRAT) {
      row = 3;
      use_menu = pref_dialog->strat_list;
    }
        
    option_menu = GTK_OPTION_MENU (gtk_option_menu_new ());
    gtk_option_menu_set_menu (option_menu, GTK_WIDGET (use_menu));

    gtk_table_attach_defaults (pref_dialog->table, GTK_WIDGET (option_menu), 
                               1, 2, row, row + 1);
    gtk_widget_show_all (GTK_WIDGET (pref_dialog->table));
        
    if (command->cmd == C_SHOW_DB) {
	    pref_dialog->db_sel = option_menu;
      gtk_label_set_mnemonic_widget (GTK_LABEL (pref_dialog->db_label),
                                     GTK_WIDGET (pref_dialog->db_sel));

      if (gail_loaded) {
        gdict_add_atk_namedesc (pref_dialog->db_label, _("Database"), _("Database Name"));
        gdict_add_atk_relation (GTK_WIDGET (option_menu),
                                pref_dialog->db_label, ATK_RELATION_LABELLED_BY);
      }

      writable = gdict_is_gconf_key_writable (pref_dialog->gdict, "database");

	    gtk_widget_set_sensitive (pref_dialog->db_label, writable);
	    gtk_widget_set_sensitive (GTK_WIDGET (option_menu), writable);
    }
    else if (command->cmd == C_SHOW_STRAT) {
	    pref_dialog->strat_sel = option_menu;
      gtk_label_set_mnemonic_widget (GTK_LABEL (pref_dialog->strat_label),
                                     GTK_WIDGET (pref_dialog->strat_sel));

      if (gail_loaded) {
        gdict_add_atk_namedesc (pref_dialog->strat_label,
                                _("Default Strategy"), _("Default Strategy"));
        gdict_add_atk_relation (GTK_WIDGET (option_menu),
                                pref_dialog->strat_label, ATK_RELATION_LABELLED_BY);
      }

      writable = gdict_is_gconf_key_writable (pref_dialog->gdict, "strategy");

	    gtk_widget_set_sensitive (pref_dialog->strat_label, writable);
	    gtk_widget_set_sensitive (GTK_WIDGET (option_menu), writable);
    }
  }
}

static void 
response_cb (GtkDialog *dialog, gint response)
{
  if (response == GTK_RESPONSE_HELP) {
    GError *error = NULL;

    gnome_help_display_on_screen ("gnome-dictionary", "gdict-settings",
                                  gtk_widget_get_screen (GTK_WIDGET (dialog)), &error);
    if (error) {
      GtkWidget *msg_dialog;

      msg_dialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (dialog),
                                                       GTK_DIALOG_DESTROY_WITH_PARENT,
                                                       GTK_MESSAGE_ERROR,
                                                       GTK_BUTTONS_OK,
                                                       "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
                                                       _("Could not display help"),
                                                       error->message);

      g_signal_connect (G_OBJECT (msg_dialog), "response",
                        G_CALLBACK (gtk_widget_destroy), NULL);

      gtk_widget_show (msg_dialog);
      g_error_free (error);
    }

    return;
  }

  gtk_widget_hide (GTK_WIDGET (dialog));
}

static gboolean
set_server_cb (GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
  GDictPrefDialog *pref_dialog;
  GDictPref *pref;
  gchar *server;

  g_return_val_if_fail (GDICT_IS_PREF_DIALOG (data), FALSE);

  pref_dialog = data;
  pref = pref_dialog->gdict->pref;
  server = gtk_editable_get_chars (GTK_EDITABLE (pref_dialog->server_entry), 0, -1);
  if (!server)
    return FALSE;

  if ((pref->server == NULL) || (strcmp (pref->server, server))) {
    g_free (pref->server);
    pref->server = g_strdup(server);
    gdict_init_context (pref_dialog->gdict);
    pref_dialog->context = pref_dialog->gdict->context;
    pref_dialog_reset_db (pref_dialog);
    pref_dialog_reset_strat (pref_dialog);
            
    gdict_set_gconf_string (pref_dialog->gdict, "server", pref->server);

    if (pref_dialog->gdict->speller_visible)
      gdict_speller_reset (pref_dialog->gdict->speller, pref_dialog->gdict->context);
  }

  g_free (server);


  return FALSE;
}

static gboolean 
set_port_cb (GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
  GDictPrefDialog *pref_dialog;
  GDictPref *pref;
  gchar *port;

  g_return_val_if_fail (GDICT_IS_PREF_DIALOG (data), FALSE);

  pref_dialog = data;
  pref = pref_dialog->gdict->pref;
  port = gtk_editable_get_chars (GTK_EDITABLE (pref_dialog->port_entry), 0, -1);
  if (pref->port != atoi (port)) {
    
    pref->port = atoi (port);
    gdict_init_context (pref_dialog->gdict);
        
    pref_dialog->context = pref_dialog->gdict->context;
        
    pref_dialog_reset_db (pref_dialog);
    pref_dialog_reset_strat (pref_dialog);
            
    gdict_set_gconf_int (pref_dialog->gdict, "port", pref->port);
  }
    
  g_free (port);

  return FALSE;
}

static void
set_default_server (GtkButton *button, gpointer data)
{
  GDictPrefDialog *pref_dialog;

  g_return_if_fail (GDICT_IS_PREF_DIALOG (data));

  pref_dialog = data;
  gtk_entry_set_text (GTK_ENTRY (pref_dialog->server_entry), DICT_DEFAULT_SERVER);
  set_server_cb (GTK_WIDGET (pref_dialog->port_entry), NULL, pref_dialog);
}

static void
set_default_port (GtkButton *button, gpointer data)
{
  GDictPrefDialog *pref_dialog;
  gchar *port;

  g_return_if_fail (GDICT_IS_PREF_DIALOG (data));

  pref_dialog = data;
  port = g_strdup_printf ("%d", DICT_DEFAULT_PORT);
  gtk_entry_set_text (GTK_ENTRY (pref_dialog->port_entry), port);
  g_free (port);
  set_port_cb (GTK_WIDGET (pref_dialog->port_entry), NULL, pref_dialog);
}
