/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-pdf-t1.c: Tyep1 Font embeding for gnome-print-pdf
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useoful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Chema Celorio <chema@celorio.com>
 *
 *  Copyright 2000-2001 Ximian, Inc. and authors
 *
 *  References:
 *    [1] Portable Document Format Referece Manual, Version 1.3 (March 11, 1999)
 *    [2] Adobe Type 1 Font Format 
 */

#include "config.h"
#include <string.h>

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-font-private.h>
#include <libgnomeprint/gnome-print-pdf-private.h>

/**
 * text_utils_search_real:
 * @buffer: buffer to do the searching in
 * @buffer_length: length of the buffer
 * @search_text: text to search for
 * @search_text_length: lenght of the search text
 * @case_sensitive: TRUE if the search is case sensitive, FALSE otherwise
 * 
 * Searches for "search_text" in "buffer" and returns the location where it is found
 * it can search buffers that contain ascii ceros. That is why you need to provide
 * buffer & search text length
 *
 * Return Value: 
 **/
static gint
text_utils_search_real (const gchar *buffer, gint buffer_length,
			const gchar *search_text, gint search_text_length,
			gboolean case_sensitive)
{
	gint p1;
	gint p2 = 0;
	gint case_sensitive_mask;

	case_sensitive_mask = case_sensitive ? 0 : 32;
	
	for (p1 = 0; p1 < buffer_length; p1 ++) {
		if ((buffer[p1]      | case_sensitive_mask)==
		    (search_text[p2] | case_sensitive_mask)) {
			p2++;
			if (p2 == search_text_length)
				return p1 - search_text_length + 1;
		}
		else
			p2 = 0;
	}
	
	return -1;
}


/**
 * gnome_print_pdf_type1_get_length: 
 * @pfb: pfb buffer
 * 
 * returns the numeric value of 2 bytes starting at pfb
 *
 * Return Value: 
 **/
static gint
gp_t1_get_length (guchar *pfb)
{
	guchar msb;
	guchar lsb;
	gint resp;
	
	lsb = pfb [0];
	msb = pfb [1];
	
	resp = (msb * 256) + lsb;
	
	return resp;
}

#define PDF_FONT_TAG_1 "%!"
#define PDF_FONT_TAG_2 "currentfile eexec"
#define PDF_FONT_TAG_3 "0000000000"

/**
 * gnome_print_pdf_type1_determine_lengths:
 * @pfb: font's pfb buffer
 * @pfb_size: size of pfb
 * @length1: length of the 1st section
 * @length2: 2nd
 * @length3: 3rd
 * 
 * given a pfb, it determines the lengths of the 1st,2nd and 3rd sections by string
 * matching. It is use to verify that the lengths of the font file specified on
 * the block identifiers are correct.
 * 
 * Return Value: TRUE on success, FALSE on error
 **/
static gboolean
gp_t1_determine_lengths (guchar *pfb, gint pfb_size, gint *length1, gint *length2, gint *length3)
{
	gint pos1;
	gint pos2;
	gint pos3;
	gint temp;

	g_return_val_if_fail (pfb != NULL, FALSE);

	/* Search for %! */
	pos1 = text_utils_search_real (pfb, pfb_size,
				       PDF_FONT_TAG_1, strlen (PDF_FONT_TAG_1),
				       TRUE);
	if (pos1 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_1);
		return FALSE;
	}
	
	/* Search for "currentfile eexec"*/
	pos2 = text_utils_search_real (pfb, pfb_size,
				       PDF_FONT_TAG_2, strlen (PDF_FONT_TAG_2),
				       TRUE);
	if (pos2 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_2);
		return FALSE;
	}
	
	/* Search for 00000 */
	pos3 = text_utils_search_real (pfb, pfb_size,
				       PDF_FONT_TAG_3, strlen (PDF_FONT_TAG_3),
				       TRUE);
	
	if (pos3 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_3);
		return FALSE;
	}
	
	temp = pos2 + strlen (PDF_FONT_TAG_2) - pos1 + 1;
	
	*length1 = temp;
	*length2 = pos3 - pos2 - strlen (PDF_FONT_TAG_2) - 1 - 12;
	/* 12 are the 2 x 6 bytes block init */
	*length3 = 0;

	return TRUE;
}

/**
 * gnome_print_pdf_type1_get_lengths: 
 * @pfb: pfb buffer
 * @length1: returns the length of the 1st section
 * @length2: 2nd section
 * @length3: 3rd section
 * 
 * reads the lengths of the 3 sections of the pfb based on the blocks identifiers
 * it also performs basic checking of the blocks identiiers.
 *
 * Return Value: TRUE on success, FALSE otherwise
 **/
	
static gboolean
gp_t1_get_lengths (guchar *pfb, gint pfb_size, gint *len1, gint *len2, gint *len3)
{
	gint length1;
	gint length2;
	gint length3;
	
	g_return_val_if_fail (pfb != NULL, FALSE);
	
	if ((pfb [0] != 0x80) ||
	    (pfb [1] != 0x01)) {
		g_warning ("Expected 0x80,0x01 at the begining of the pfb (1)\n");
		return FALSE;
	}
	if ((pfb [4] != 0x00) ||
	    (pfb [5] != 0x00)) {
		g_warning ("Expected 0x00,0x00 at the begining of the pfb (2)\n");
		return FALSE;
	}
	
	*len1 = gp_t1_get_length (pfb + 2);
	
	if ((pfb [ *len1 + 6]     != 0x80) ||
	    (pfb [ *len1 + 6 + 1] != 0x02)) {
		g_warning ("Expected 0x80,0x02 at the midle of the pfb (3)\n");
		return FALSE;
	}
	if ((pfb [ *len1 + 6 + 4] != 0x00) ||
	    (pfb [ *len1 + 6 + 5] != 0x00)) {
		g_warning ("Expected 0x00,0x00 at the middle of the pfb (4)\n");
		return FALSE;
	}
	
	*len2 = gp_t1_get_length (pfb + *len1 + 2 + 6);
	*len3 = 0;
	
	if (!gp_t1_determine_lengths (pfb, pfb_size, &length1, &length2, &length3)) {
		g_warning ("Could not determine lengths from font file");
		return -1;
	}
	if ((*len1 != length1) ||
	    (*len2 != length2) ||
	    (*len3 != length3) ) {
		g_warning ("The lengths of the font do not match [%i,%i,%i] [%i,%i,%i]",
			   *len1, *len2, *len3, length1, length2, length3);
		return -1;
	}
	
	
	return TRUE;
}


/**
 * gnome_print_pdf_type1_read_pfb:
 * @file_name: full path of the .pfb file
 * @pfb_size: return the length read here
 * 
 * reads a pfb from filename 
 * 
 * Return Value: pointer to the malloced buffer, caller must free it
 *               NULL on error
 **/
static guchar*
gp_t1_read_pfb (const gchar* file_name, gint *pfb_size_)
{
	FILE *f;
	gint pfb_size_max;
	gint bytes_read;
	gint pfb_size;
	guchar *pfb;
	
	*pfb_size_ = 0;
	pfb_size = 0;
	
	g_return_val_if_fail (file_name != NULL, NULL);
	
	f = fopen (file_name, "r");
	if (f == NULL) {
		g_warning ("Couldn't open font file %s\n", file_name);
		return NULL;
	}
	
	pfb_size = 0;
	pfb_size_max = 32768;
	pfb = g_new (guchar, pfb_size_max);
	while (1) {
		bytes_read = fread (pfb + pfb_size, 1, pfb_size_max - pfb_size, f);
		if (bytes_read == 0)
			break;
		pfb_size += bytes_read;
		pfb = g_realloc (pfb, pfb_size_max <<= 1);
	}
	
	*pfb_size_ = pfb_size;
	
	if (0 != fclose (f)) {
		g_warning ("Could not close %s", file_name);
		return NULL;
	}
	
	return pfb;
}


/**
 * gnome_print_pdf_font_parse:
 * @file_name: full path of the pfb of the font
 * @body_: return the cleaned structure here, the caller must free it later
 *         by cleaned I mean that it does not contain block identifiers and
 *         has the trailing ceros removed. So it is only section 1 & 2.
 * @length: the length of the block
 * @length1: length of the 1st section of the font
 * @length2: "" 2nd ""
 * @length3: "" 3rd "", which will always be 0 for now.
 * @body_length: The length of @body
 * 
 * Parses the type1 font pointed by *file_name and cleans it by removing
 * the block identifiers and the 3rd seccion (section of 512 0's). It also
 * calculates the lengths of the sections of the font. See [2] Sec 7.11.1 page 223.
 *
 * Return Value: TRUE on success, FALSE otherwise
 **/
static gboolean
gp_t1_font_parse (const gchar *file_name,
		  gchar **body_,
		  gint *length, gint *len1, gint *len2, gint *len3,
		  gint *body_length)
{
	guchar *pfb;
	guchar *pfb_clean;
	gint pfb_size;
	
	*length = 0;
	*len1 = 0;
	*len2 = 0;
	*len3 = 0;

	pfb = gp_t1_read_pfb (file_name, &pfb_size);

	if ((pfb == NULL) || (!pfb_size)) {
		g_warning ("Could not read the font from \"%s\"\n", file_name);
		return FALSE;
	}
	
	if (!gp_t1_get_lengths (pfb, pfb_size, len1, len2, len3)) {
		g_warning ("Could not get lengths from font file [%s]", file_name);
		return FALSE;
	}
	
	*length = *len1 + *len2 + strlen (EOL);
	*len3 = 0;
	
	*body_length = *length;
	
	pfb_clean = g_new (guchar, *len1 + *len2 + 1);
	
	/* The block identifiers are 6 bytes long */
	memcpy (pfb_clean, pfb + 6, *len1);
	memcpy (pfb_clean + *len1, pfb + 6 + *len1 + 6, *len2);
	
	*body_ = pfb_clean;
	*body_length = *len1 + *len2;

	g_free (pfb);
	
	return TRUE;
}

/**
 * gnome_print_pdf_font_type1_embed:
 * @pc: Print Context to embed in
 * @font: Font to embed
 * 
 * Embeds in the print context a the type1 font
 *
 * Return Value: 0 on success, -1 on error
 **/
gint
gnome_print_pdf_t1_embed (GnomePrintPdf *pdf,
			  GnomePrintPdfFont *font,
			  gint object_number)
{
	GnomeFontFace *face;
	GPFontEntryT1 *t1;
	gchar *file_name = NULL;
	gchar *body = NULL;
	gint length, length1, length2, length3;
	gint body_length;
	gboolean ascii85 = FALSE;

	face = font->face;
	t1 = (GPFontEntryT1 *) face->entry;
	g_assert (face->entry->type == GP_FONT_ENTRY_TYPE1 ||
		  face->entry->type == GP_FONT_ENTRY_TYPE1_ALIAS);
	file_name = t1->pfb.name;
	
	if (!file_name) {
		g_warning ("Could not find file_name to open");
		goto pdf_type1_error;
	}
	
	if (!gp_t1_font_parse (file_name, &body, &length,
			       &length1, &length2,
			       &length3, &body_length)) {
		g_warning ("Could not Parse %s", file_name);
		goto pdf_type1_error;
	}

	if ((length == 0) || (length1 == 0)) {
		g_warning ("Could not embed the font. Invalid length or length1 for %s\n",
			   file_name);
		goto pdf_type1_error;
	}

	/* Write the object */
	gnome_print_pdf_object_start (pdf, object_number, FALSE);
	if (ascii85)
		gnome_print_pdf_fprintf  (pdf,"/Filter /ASCII85Decode" EOL);
	gnome_print_pdf_fprintf    (pdf,
				    "/Length %d" EOL
				    "/Length1 %d" EOL
				    "/Length2 %d" EOL
				    "/Length3 %d" EOL
				    ">>" EOL
				    "stream" EOL,
				    length,
				    length1,
				    length2,
				    length3);
	gnome_print_pdf_print_sized (pdf, body, body_length);
	gnome_print_pdf_fprintf     (pdf, EOL
				     "endstream" EOL
				     "endobj" EOL);
	gnome_print_pdf_object_end  (pdf, object_number, TRUE);

	if (file_name)
		g_free (file_name);
	if (body)
		g_free (body);
	
	return GNOME_PRINT_OK;
pdf_type1_error:
	if (file_name)
		g_free (file_name);
	if (body)
		g_free (body);

	return GNOME_PRINT_ERROR_UNKNOWN;
}

