/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-model.c: 
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Jose M. Celorio <chema@ximian.com>
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */

#include <config.h>

#include <string.h>
#include <libxml/parser.h>
#include <libxml/xmlmemory.h>

#include "gpa-utils.h"
#include "gpa-value.h"
#include "gpa-reference.h"
#include "gpa-vendor.h"
#include "gpa-option.h"
#include "gpa-model.h"

/* GPAModel */

static void gpa_model_class_init (GPAModelClass *klass);
static void gpa_model_init (GPAModel *model);
static void gpa_model_finalize (GObject *object);

static gboolean  gpa_model_verify    (GPANode *node);
static guchar *  gpa_model_get_value (GPANode *node);
static GPANode * gpa_model_get_child (GPANode *node, GPANode *previous_child);
static GPANode * gpa_model_lookup    (GPANode *node, const guchar *path);
static void      gpa_model_modified  (GPANode *node, guint flags);

static GHashTable *modeldict = NULL;
static GPANodeClass *parent_class = NULL;

GType
gpa_model_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPAModelClass),
			NULL, NULL,
			(GClassInitFunc) gpa_model_class_init,
			NULL, NULL,
			sizeof (GPAModel),
			0,
			(GInstanceInitFunc) gpa_model_init
		};
		type = g_type_register_static (GPA_TYPE_NODE, "GPAModel", &info, 0);
	}
	return type;
}

static void
gpa_model_class_init (GPAModelClass *klass)
{
	GObjectClass *object_class;
	GPANodeClass *node_class;

	object_class = (GObjectClass *) klass;
	node_class = (GPANodeClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gpa_model_finalize;

	node_class->verify    = gpa_model_verify;
	node_class->get_value = gpa_model_get_value;
	node_class->get_child = gpa_model_get_child;
	node_class->lookup    = gpa_model_lookup;
	node_class->modified  = gpa_model_modified;
}

static void
gpa_model_init (GPAModel *model)
{
	model->loaded = FALSE;
	model->name    = NULL;
	model->vendor  = NULL;
	model->options = NULL;
}

static void
gpa_model_vendor_gone (gpointer data, GObject *gone)
{
	GPAModel *model;

	model = GPA_MODEL (data);

	model->vendor = NULL;
}

static void
gpa_model_vendor_modified (GPANode *node, guint flags, GPANode *root)
{
	gpa_node_request_modified (root, flags);
}

static void
gpa_model_finalize (GObject *object)
{
	GPAModel *model;

	model = GPA_MODEL (object);

	if (GPA_NODE_ID_EXISTS (model)) {
		g_assert (modeldict != NULL);
#if 0
		g_assert (g_hash_table_lookup (modeldict, GPA_NODE_ID (model)) != NULL);
#endif
		g_hash_table_remove (modeldict, GPA_NODE_ID (model));
	}

	model->name = gpa_node_detach_unref (GPA_NODE (model), GPA_NODE (model->name));
	if (model->vendor) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (model->vendor), gpa_model_vendor_modified, model);
		g_object_weak_unref (G_OBJECT (model->vendor), gpa_model_vendor_gone, model);
		model->vendor = NULL;
	}
	if (model->options) {
		model->options = gpa_node_detach_unref (GPA_NODE (model), GPA_NODE (model->options));
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
gpa_model_verify (GPANode *node)
{
	GPAModel *model;

	model = GPA_MODEL (node);

	if (!model->name)
		return FALSE;
	if (!gpa_node_verify (model->name))
		return FALSE;
	if (model->loaded) {
		if (model->vendor && !gpa_node_verify (model->vendor))
			return FALSE;
		if (!model->options)
			return FALSE;
		if (!gpa_node_verify (GPA_NODE (model->options)))
			return FALSE;
	}

	return TRUE;
}

static guchar *
gpa_model_get_value (GPANode *node)
{
	GPAModel *model;

	model = GPA_MODEL (node);

	if (GPA_NODE_ID_EXISTS (node))
		return g_strdup (gpa_node_id (node));

	return NULL;
}

static GPANode *
gpa_model_get_child (GPANode *node, GPANode *previous_child)
{
	GPAModel *model;
	GPANode *child = NULL;

	model = GPA_MODEL (node);

	if (previous_child == NULL) {
		child = model->name;
	} else if (previous_child == model->name) {
		child = model->vendor;
	} else if (previous_child == model->vendor) {
		child = GPA_NODE (model->options);
	}

	if (child)
		gpa_node_ref (child);

	return child;
}

static GPANode *
gpa_model_lookup (GPANode *node, const guchar *path)
{
	GPAModel *model;
	GPANode *child;

	model = GPA_MODEL (node);

	child = NULL;

	if (gpa_node_lookup_helper (GPA_NODE (model->name),   path, "Name", &child))
		return child;
	if (gpa_node_lookup_helper (GPA_NODE (model->vendor), path, "Vendor", &child))
		return child;
	if (gpa_node_lookup_helper (GPA_NODE (model->options), path, "Options", &child))
		return child;

	return NULL;
}

static void
gpa_model_modified (GPANode *node, guint flags)
{
	GPAModel *model;

	model = GPA_MODEL (node);

	if (model->name && (GPA_NODE_FLAGS (model->name) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (model->name, 0);
	}
	if (model->vendor && (GPA_NODE_FLAGS (model->vendor) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (model->vendor, 0);
	}
	if (model->options && (GPA_NODE_FLAGS (model->options) & GPA_NODE_MODIFIED_FLAG)) {
		gpa_node_emit_modified (GPA_NODE (model->options), 0);
	}
}

/* Public Methods */
/* FIXME:  _from_info & _from_info_tree, they share a lot of code (Chema) */
static GPANode *
gpa_model_new_from_info_tree (xmlNodePtr tree)
{
	xmlNodePtr xml_node;
	GPAModel *model;
	GPANode *name;
	xmlChar *model_id;
	guchar *model_file;

	g_return_val_if_fail (tree != NULL, NULL);

	/* Check that tree is <Model> */
	if (strcmp (tree->name, "Model")) {
		g_warning ("file %s: line %d: Base node is <%s>, should be <Model>", __FILE__, __LINE__, tree->name);
		return NULL;
	}
	
	/* Check that model has Id */
	model_id = xmlGetProp (tree, "Id");
	if (!model_id) {
		g_warning ("file %s: line %d: Model node does not have Id", __FILE__, __LINE__);
		return NULL;
	}
	
	/* Check for model file */
	model_file = g_strdup_printf (GNOME_PRINT_DATA_DIR "/models/%s.model", model_id);
	if (!g_file_test (model_file, G_FILE_TEST_IS_REGULAR)) {
		g_warning ("Model description file is missing %s", model_id);
		xmlFree (model_id);
		g_free (model_file);
		return NULL;
	}
	g_free (model_file);

	/* Create modeldict, and check if this model is already loaded */
	if (!modeldict)
		modeldict = g_hash_table_new (g_str_hash, g_str_equal);
	model = g_hash_table_lookup (modeldict, model_id);
	if (model != NULL) {
		gpa_node_ref (GPA_NODE (model));
		return GPA_NODE (model);
	}
	
	model = NULL;
	name = NULL;

	for (xml_node = tree->xmlChildrenNode; xml_node != NULL; xml_node = xml_node->next) {
		/* <Name> */
		if (strcmp (xml_node->name, "Name") == 0) {
			name = gpa_value_new_from_tree ("Name", xml_node);
			continue;
		}
	}

	if (!name) {
		g_warning ("Incomplete model description. \"name\" missing.");
		goto gpa_model_new_from_info_tree_error;
	}
	
	model = (GPAModel *) gpa_node_new (GPA_TYPE_MODEL, model_id);
	model->name = name;
	name->parent = GPA_NODE (model);
	g_hash_table_insert (modeldict, (gpointer) GPA_NODE_ID (model), model);

gpa_model_new_from_info_tree_error:
	
	xmlFree (model_id);

	return (GPANode *) model;
}

/**
 * gpa_model_new_from_tree:
 * @tree: 
 * 
 * Load a GPAModel from an XML node
 * 
 * Return Value: 
 **/
GPANode *
gpa_model_new_from_tree (xmlNodePtr tree)
{
	xmlNodePtr xml_node;
	xmlChar *xml_version;
	xmlChar *model_id;
	GPAModel *model;
	GPANode *name;
	GPANode *vendor;
	GPAList *options;

	g_return_val_if_fail (tree != NULL, NULL);

	/* Check that tree is <Model> */
	if (strcmp (tree->name, "Model")) {
		g_warning ("file %s: line %d: Base node is <%s>, should be <Model>", __FILE__, __LINE__, tree->name);
		return NULL;
	}
	
	/* Check that model has Id */
	model_id = xmlGetProp (tree, "Id");
	if (!model_id) {
		g_warning ("file %s: line %d: Model node does not have Id", __FILE__, __LINE__);
		return NULL;
	}
	
	/* Check Model definition version */
	xml_version = xmlGetProp (tree, "Version");
	if (!xml_version || strcmp (xml_version, "1.0")) {
		g_warning ("file %s: line %d: Wrong model version %s, should be 1.0.", __FILE__, __LINE__, xml_version);
		xmlFree (model_id);
		if (xml_version)
			xmlFree (xml_version);
		return NULL;
	}
	xmlFree (xml_version);

	/* Create modeldict, and check if this model is already loaded */
	if (!modeldict)
		modeldict = g_hash_table_new (g_str_hash, g_str_equal);
	model = g_hash_table_lookup (modeldict, model_id);
	if (model != NULL) {
		gpa_node_ref (GPA_NODE (model));
		return GPA_NODE (model);
	}

	model = NULL;
	name = NULL;
	vendor = NULL;
	options = NULL;

	for (xml_node = tree->xmlChildrenNode; xml_node != NULL; xml_node = xml_node->next) {
		/* <Name> */
		if (strcmp (xml_node->name, "Name") == 0) {
			name = gpa_value_new_from_tree ("Name", xml_node);
			continue;
		}

		/* <Vendor> */
		if (strcmp (xml_node->name, "Vendor") == 0) {
			xmlChar *vendor_id;
			vendor_id = xmlNodeGetContent (xml_node);
			if (!vendor_id || !*vendor_id)
				continue; /* The error will be caught below */
			vendor = gpa_vendor_get_by_id (vendor_id);
			xmlFree (vendor_id);
			continue;
		}

		/* <Options> */
		if (strcmp (xml_node->name, "Options") == 0) {
			options = gpa_option_list_new_from_tree (xml_node);
			continue;
		}
	}

	/* Something went wrong */
	if (!name || !vendor || !options) {
		
		if (!name)
			g_warning ("Model \"%s\"does not have a valid name",   model_id);
		if (!vendor)
			g_warning ("Model \"%s\"does not have a valid vendor", model_id);
		if (!options)
			g_warning ("Model \"%s\"does not have valid options",  model_id);
		
		if (name)
			gpa_node_unref (name);
		if (vendor)
			gpa_node_unref (vendor);
		if (options)
			gpa_node_unref (GPA_NODE (options));
		
		xmlFree (model_id);
		return NULL;
	}

	/* Everything is OK, create the model */
	model = (GPAModel *) gpa_node_new (GPA_TYPE_MODEL, model_id);
	model->name    = gpa_node_attach (GPA_NODE (model), name);
	model->vendor  = gpa_reference_new (vendor);
	model->options = gpa_node_attach (GPA_NODE (model), GPA_NODE (options));
	model->loaded  = TRUE;
		
	g_hash_table_insert (modeldict, (gpointer) GPA_NODE_ID (model), model);

	xmlFree (model_id);
	
	return (GPANode *) model;
}

/* GPAModelList */

GPAList *
gpa_model_list_new_from_info_tree (xmlNodePtr tree)
{
	xmlNodePtr xml_node;
	GPAList *models;
	GSList *list = NULL;

	g_return_val_if_fail (!strcmp (tree->name, "Models"), NULL);

	for (xml_node = tree->xmlChildrenNode; xml_node != NULL; xml_node = xml_node->next) {
		if (strcmp (xml_node->name, "Model") == 0) {
			GPANode *model;
			model = gpa_model_new_from_info_tree (xml_node);
			if (model)
				list = g_slist_prepend (list, model);
		}
	}

	
	models = gpa_list_new (GPA_TYPE_MODEL, "Models", FALSE);

	while (list) {
		GPANode *model;
		
		model = GPA_NODE (list->data);
		model->parent = GPA_NODE (models);
		model->next = models->children;
		models->children = model;

		list = g_slist_remove (list, model);
	}

	return models;
}

GPANode *
gpa_model_get_by_id (const guchar *id)
{
	GPAModel *model;
	gchar *path;
	xmlDocPtr doc;
	xmlNodePtr root;

	g_return_val_if_fail (id != NULL, NULL);
	g_return_val_if_fail (*id != '\0', NULL);

	if (!modeldict)
		modeldict = g_hash_table_new (g_str_hash, g_str_equal);
	model = g_hash_table_lookup (modeldict, id);
	if (model) {
		gpa_node_ref (GPA_NODE (model));
		return GPA_NODE (model);
	}

	path = g_strdup_printf (GNOME_PRINT_DATA_DIR "/models/%s.model", id);
	if (!g_file_test (path, G_FILE_TEST_IS_REGULAR)) {
		g_warning ("Could not get model by id '%s' from '%s'\n", id, path);
		g_free (path);
		return NULL;
	}

	doc = xmlParseFile (path);
	if (!doc) {
		g_warning ("Could not parse XML. Model by id '%s' from '%s'\n", id, path);
		g_free (path);
		return NULL;
	}
	g_free (path);

	model = NULL;
	root = doc->xmlRootNode;
	if (!strcmp (root->name, "Model")) {
		model = GPA_MODEL (gpa_model_new_from_tree (root));
	}
	xmlFreeDoc (doc);

	return GPA_NODE (model);
}


gboolean
gpa_model_load (GPAModel *model)
{
	xmlNodePtr root;
	xmlNodePtr xmlc;
	xmlDocPtr doc;
	xmlChar *model_id;
	GPANode *vendor;
	GPAList *options;
	gchar *path;

	g_return_val_if_fail (model != NULL, FALSE);
	g_return_val_if_fail (GPA_IS_MODEL (model), FALSE);

	if (model->loaded)
		return TRUE;

	path = g_strdup_printf (GNOME_PRINT_DATA_DIR "/models/%s.model", GPA_NODE_ID (model));
	if (!g_file_test (path, G_FILE_TEST_IS_REGULAR)) {
		g_warning ("Model description file missing %s", GPA_NODE_ID (model));
		g_free (path);
		return FALSE;
	}

	doc = xmlParseFile (path);
	g_free (path);
	if (!doc) {
		g_warning ("Invalid model description file %s", GPA_NODE_ID (model));
		return FALSE;
	}

	root = doc->xmlRootNode;
	if (strcmp (root->name, "Model")) {
		g_warning ("Invalid model description file %s", GPA_NODE_ID (model));
		return FALSE;
	}

	model_id = xmlGetProp (root, "Id");
	if (!model_id || !GPA_NODE_ID_COMPARE (model, model_id)) {
		g_warning ("Missing \"Id\" node in model description %s", GPA_NODE_ID (model));
		return FALSE;
	}

	vendor = NULL;
	options = NULL;

	for (xmlc = root->xmlChildrenNode; xmlc != NULL; xmlc = xmlc->next) {
		/* FIXME: vendor should be initialized from info tree (Lauris) */
		if (!strcmp (xmlc->name, "Vendor")) {
			xmlChar *vendorid;
			vendorid = xmlNodeGetContent (xmlc);
			if (vendorid) {
				vendor = gpa_vendor_get_by_id (vendorid);
				xmlFree (vendorid);
			}
		} else if (!strcmp (xmlc->name, "Options")) {
			options = gpa_option_list_new_from_tree (xmlc);
		}
	}

	if (!vendor || !options) {
		g_warning ("Incomplete model description");
		if (vendor)
			gpa_node_unref (vendor);
		if (options)
			gpa_node_unref (GPA_NODE (options));
		
		return FALSE;
	}
	
	model->options = gpa_node_attach (GPA_NODE (model), GPA_NODE (options));
	model->vendor = gpa_reference_new (vendor);
	model->loaded = TRUE;
	
	xmlFree (model_id);
	xmlFreeDoc (doc);

	return TRUE;
}
