/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

#include <locale.h>
#include <config.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gnome.h>
#include <gst/gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include "gstmediaplay.h"
#include "gstvideowidget.h"
#include "gstpreferences.h"
#include "gstmediainfo.h"
#include "gstcontrol.h"
#include "gststatusarea.h"
#include "gstplayer-marshal.h"

static void 	gst_media_play_class_init      	 (GstMediaPlayClass *klass);
static void 	gst_media_play_init       	 (GstMediaPlay *play);

/* signal handlers for GstControl */
static void 	gst_media_play_slider_changed    (GtkAdjustment *adj, GstMediaPlay *mplay);
static void     gst_media_play_toggle_mute       (GtkWidget *widget, GstMediaPlay *mplay);
static void     gst_media_play_seek_to_pos       (GtkWidget *widget, gdouble value, GstMediaPlay *mplay);
static void     gst_media_play_toggle_play_pause (GtkWidget *widget, GstMediaPlay *mplay);
static void     gst_media_play_stop              (GtkWidget *widget, GstMediaPlay *mplay);
static void	gst_media_play_display_mode_change (GtkWidget *widget, GstMediaPlayMode display_mode, GstMediaPlay *mplay);
static void	gst_media_play_toggle_display_mode (GtkWidget *widget, GstMediaPlay *mplay);
static void	gst_media_play_fullscreen	 (GtkWidget *widget, GstMediaPlay *mplay);
static void	gst_media_play_volume_changed    (GtkWidget *widget, gdouble value, GstMediaPlay *mplay);

/* signal handlers for GstPlay */
static void     gst_media_play_stream_end        (GstPlay *play, GstMediaPlay *mplay);
static void     gst_media_play_time_tick         (GstPlay *play, gint64 time_nanos, GstMediaPlay *mplay);
static void     gst_media_play_got_length        (GstPlay *play, gint64 length_nanos, GstMediaPlay *mplay);
static void	gst_media_play_have_xid          (GstPlay *play, gint xid, GstMediaPlay *mplay);
static void	gst_media_play_have_video_size   (GstPlay *play, gint width, gint height, GstMediaPlay *mplay);
static void	gst_media_play_state_change      (GstPlay *play, GstElementState old_state, GstElementState new_state, GstMediaPlay *mplay);
static void	gst_media_play_information       (GstPlay *play, GstElement* element, GParamSpec *param, GstMediaPlay *mplay);
static GladeXML* gst_media_play_get_glade_xml    (const char *fname, const char *root, const char *domain);

/* FIXME: maybe this function should be done somewhere else */
#define GST_MEDIA_GCONF_DIR "/apps/gst-player"


static gst_media_error_dialog (const gchar *message)
{
	GtkWidget *dialog;
	/* throw up a dialog box; FIXME: we don't have a parent */
	/* FIXME: maybe even fallback the GError and do error
	* handling higher up ? */
	dialog = gtk_message_dialog_new (NULL, 0,
			                 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 message);
	gtk_dialog_run (GTK_DIALOG (dialog));
	/*
	gtk_widget_show (dialog);
	g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
				  G_CALLBACK (gtk_widget_destroy),
				  GTK_OBJECT (dialog));
				  */
}

static GObject *parent_class = NULL;

enum {
	STATE_CHANGE,
	DISPLAY_MODE_CHANGE,
	CURRENT_LOCATION,
	LAST_SIGNAL
};

GType
gst_media_play_get_type (void)
{
	static GType play_type = 0;
  
	if (!play_type)
	{
		static const GTypeInfo style_info =
			{
				sizeof (GstMediaPlayClass),
				(GBaseInitFunc) NULL,
				(GBaseFinalizeFunc) NULL,
				(GClassInitFunc) gst_media_play_class_init,
				NULL,           /* class_finalize */
				NULL,           /* class_data */
				sizeof (GstMediaPlay),
				0,              /* n_preallocs */
				(GInstanceInitFunc) gst_media_play_init,
			};
		
		play_type = g_type_register_static (GTK_TYPE_VBOX,
						    "GstMediaPlay",
						    &style_info, 0);
	}
	
	return play_type;
}
static gint gst_media_play_signals [LAST_SIGNAL] = { 0 };

static void
gst_media_play_class_init (GstMediaPlayClass *klass)
{
	GtkWidgetClass  *widget_class = (GtkWidgetClass *) klass;
	GtkObjectClass  *object_class = (GtkObjectClass *) klass;
	GObjectClass    *gobject_class = (GObjectClass *) klass;

	gst_media_play_signals [STATE_CHANGE] = 
		g_signal_new ("state_change", 
			      G_TYPE_FROM_CLASS (klass), 
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstMediaPlayClass, state_changed), 
			      NULL, NULL,
			      gst_marshal_VOID__INT_INT,
			      G_TYPE_NONE, 2, 
			      G_TYPE_INT, G_TYPE_INT);
	gst_media_play_signals [DISPLAY_MODE_CHANGE] =
		g_signal_new ("display_mode_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstMediaPlayClass, display_mode_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__INT,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT);
	gst_media_play_signals [CURRENT_LOCATION] =
		g_signal_new ("current_location", 
			      G_TYPE_FROM_CLASS (klass), 
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstMediaPlayClass, current_location), 
			      NULL, NULL,
			      gst_play_marshal_VOID__STRING_STRING,
			      G_TYPE_NONE, 2, 
			      G_TYPE_STRING, G_TYPE_STRING);
}
static void
gst_media_play_init (GstMediaPlay *mplay)
{
	GstElement *audio_sink, *video_sink;
	GError *error = NULL;
	
	/* play object */
	mplay->play = gst_play_new (GST_PLAY_PIPE_VIDEO, &error);
	if (error != NULL)
	{
		gst_media_error_dialog (error->message);
		g_error_free (error);
	}
	if (mplay->play == NULL) return;

#ifdef HAVE_GSTREAMER_CVS
	audio_sink = gst_gconf_get_default_audio_sink ();
#else
	audio_sink = gst_gconf_render_bin_from_key ("default/audiosink");
#endif
	if (!GST_IS_ELEMENT (audio_sink))
	{
		gst_media_error_dialog (
			"Could not render default GStreamer audio output sink "
			"from GConf /system/gstreamer/default/audiosink key. "
		        "Check if it is set correctly."	
		);
		return;
	}
	gst_play_set_audio_sink (mplay->play, audio_sink);

#ifdef HAVE_GSTREAMER_CVS
	video_sink = gst_gconf_get_default_video_sink ();
#else
	video_sink = gst_gconf_render_bin_from_key ("default/videosink");
#endif
	if (!GST_IS_ELEMENT (video_sink))
	{
		gst_media_error_dialog (
			"Could not render default GStreamer video output sink "
			"from GConf /system/gstreamer/default/videosink key. "
		        "Check if it is set correctly."	
		);
		return;
	}
	gst_play_set_video_sink (mplay->play, video_sink);

	g_signal_connect (G_OBJECT (mplay->play), "stream_end",
			  G_CALLBACK (gst_media_play_stream_end), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "time_tick", 
			  G_CALLBACK (gst_media_play_time_tick), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "stream_length", 
			  G_CALLBACK (gst_media_play_got_length), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "have_xid", 
			  G_CALLBACK (gst_media_play_have_xid), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "have_video_size", 
			  G_CALLBACK (gst_media_play_have_video_size), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "state_change", 
			  G_CALLBACK (gst_media_play_state_change), mplay);
	g_signal_connect (G_OBJECT (mplay->play), "information", 
			  G_CALLBACK (gst_media_play_information), mplay);
	
	gtk_box_set_homogeneous (GTK_BOX(mplay), FALSE);
	gtk_box_set_spacing (GTK_BOX(mplay), 0);
	/* player widgets */

	mplay->video_widget = gst_video_widget_new ();
	g_return_if_fail(mplay->video_widget != NULL);
	gtk_box_pack_start (GTK_BOX (mplay), mplay->video_widget, TRUE, TRUE, 0);

	mplay->control = gst_control_new (gst_media_play_get_glade_xml("mediacontrol.glade", "vbox_media_control", NULL));
	g_return_if_fail(mplay->control != NULL);
	gtk_box_pack_start (GTK_BOX (mplay), GTK_WIDGET (mplay->control), FALSE, FALSE, 0);
	
	mplay->statusarea = gst_status_area_new(gst_media_play_get_glade_xml("mediacontrol.glade", "vbox_media_info", NULL));
	gtk_box_pack_start (GTK_BOX (mplay), mplay->statusarea, FALSE, FALSE, 0);

	g_signal_connect (G_OBJECT (mplay->control), "button_play",
			  G_CALLBACK (gst_media_play_toggle_play_pause), mplay);
	g_signal_connect (G_OBJECT (mplay->control), "button_stop",
			  G_CALLBACK (gst_media_play_stop), mplay);
	g_signal_connect (G_OBJECT (mplay->control), "button_mute",
			  G_CALLBACK (gst_media_play_toggle_mute), mplay);
	g_signal_connect (G_OBJECT (mplay->control), "volume_change",
			  G_CALLBACK (gst_media_play_volume_changed), mplay);
	g_signal_connect (G_OBJECT (mplay->control), "seek_change",
			  G_CALLBACK (gst_media_play_seek_to_pos), mplay);
	g_signal_connect (G_OBJECT (mplay->control), "display_mode_change",
			  G_CALLBACK (gst_media_play_display_mode_change), mplay);
	
	mplay->display_mode = GST_MEDIA_PLAY_NORMAL;

	gtk_widget_show (mplay->statusarea);
	gtk_widget_show (mplay->control);

	mplay->location = NULL;
}

GstMediaPlay*
gst_media_play_new (void)
{
	return GST_MEDIA_PLAY (g_object_new (GST_TYPE_MEDIA_PLAY, NULL));
}
  


void
gst_media_play_set_state (GstMediaPlay *mplay, GstElementState state)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	gst_play_set_state (mplay->play, state);
}

static void
gst_media_play_toggle_play_pause (GtkWidget *widget, GstMediaPlay *mplay)
{
	GstElementState playstate;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	playstate = gst_play_get_state (mplay->play);

	if (playstate == GST_STATE_PLAYING) {
		gst_media_play_set_state (mplay, GST_STATE_PAUSED);
	}
	else {
		gst_media_play_set_state (mplay, GST_STATE_PLAYING);
	}
}

static void
gst_media_play_stop (GtkWidget *widget, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	gst_media_play_set_state (mplay, GST_STATE_READY);
}

static void
gst_media_play_volume_changed (GtkWidget *widget, gdouble value, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	gst_play_set_volume (mplay->play, value);
}

void
gst_media_play_set_volume (GstMediaPlay *mplay, gdouble value)
{
	GtkAdjustment *adj;
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	adj = gst_control_get_volume_adjustment(GST_CONTROL(mplay->control));
	
	value = value > 1.0 ? 1.0 : value;
	value = value < 0.0 ? 0.0 : value;
	gtk_adjustment_set_value(adj, value);
}

gdouble
gst_media_play_get_volume (GstMediaPlay *mplay)
{
	GtkAdjustment *adj;
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	adj = gst_control_get_volume_adjustment(GST_CONTROL(mplay->control));
	return gtk_adjustment_get_value(adj);
}

void
gst_media_play_set_location (GstMediaPlay *mplay, const gchar *location)
{
	GnomeVFSFileInfo info;
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	g_return_if_fail(location != NULL);

	mplay->location_short = NULL;
	mplay->location = location;
	
	/* get a short filename if we can */
	if (GNOME_VFS_OK != gnome_vfs_get_file_info(location, &info, GNOME_VFS_FILE_INFO_DEFAULT)){
		mplay->location_short = location;
	}
	else {
		mplay->location_short = info.name;
	}
	
	g_signal_emit (mplay, gst_media_play_signals [CURRENT_LOCATION], 0, mplay->location, mplay->location_short);

	gst_status_area_reset_media_info(GST_STATUS_AREA(mplay->statusarea));
	gst_media_play_set_state (mplay, GST_STATE_READY);

	gst_status_area_set_location(GST_STATUS_AREA(mplay->statusarea), mplay->location_short);
	gst_play_set_location (mplay->play, mplay->location);
}

const gchar*
gst_media_play_get_location (GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	return mplay->location;
}


static void
gst_media_play_toggle_mute (GtkWidget *widget, GstMediaPlay *mplay)
{
	gboolean muted;
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	muted = !gst_play_get_mute(mplay->play);
	gst_play_set_mute(mplay->play, muted);
	gst_control_set_mute(GST_CONTROL(mplay->control), muted);
}

static void
gst_media_play_seek_to_pos (GtkWidget *widget, gdouble value, GstMediaPlay *mplay)
{
	gdouble length_nanos;
	gint64 seek_time;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	length_nanos = (gdouble)mplay->length_nanos;
	seek_time = (gint64)(length_nanos * value);
	gst_play_seek_to_time (mplay->play, seek_time);
}

void
gst_media_play_set_video_scale (GstMediaPlay *mplay, gfloat scale_factor)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	if (mplay->video_widget) {
		g_object_set (G_OBJECT (mplay->video_widget), "scale_factor", scale_factor, NULL);
	}
}

void
gst_media_play_show_media_info (GstMediaPlay *mplay)
{
	GtkWidget *media_info;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	if (mplay->location != NULL) {
		media_info = gst_media_info_new (mplay->location);
	}
	else {
		media_info = gst_media_info_new ("no file loaded");
	}

	gtk_widget_show (media_info);
}

void
gst_media_play_show_preferences (GstMediaPlay *mplay)
{
	GtkWidget *preferences;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	preferences = gst_preferences_new ();
	
	gtk_widget_show (preferences);
}

void
gst_media_play_show_about (GstMediaPlay *mplay)
{
	static GtkWidget *about;

	const gchar *authors[] = {
		"Steve Baker <stevebaker_org@yahoo.co.uk>", 
		"Richard Boulton <richard@tartarus.org>",
		"Ronald Bultje <rbultje@ronald.bitfreak.net>",
		"Arik Devens <arik@gnome.org>",
		"Chris Emerson (PPC port)",
		"Benjamin Otte <in7y118@public.uni-hamburg.de>",
		"Wim Taymans <wim.taymans@tvd.be>",
		"Erik Walthinsen <omega@cse.ogi.edu>", 
		"Thomas Vander Stichele <thomas@apestaart.org>", 
		NULL};

	if (about != NULL) 
	{
                gdk_window_raise (about->window);
                gdk_window_show (about->window);
                return;
        }

	about = gnome_about_new ("GStreamer Player", GST_PLAYER_VERSION, 
				 "(C) 1999-2002 The GStreamer Team",
				 "A generic media player for the GStreamer media framework.",
				 (const char **)authors,
				 NULL,
				 NULL,
				 NULL);

	g_signal_connect (G_OBJECT (about), "destroy",
			  G_CALLBACK (gtk_widget_destroyed), &about);
	
        gtk_widget_show (about);
}

void
gst_media_play_fullscreen (GtkWidget *widget, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	gst_media_play_set_display_mode (mplay, GST_MEDIA_PLAY_FULLSCREEN);
}

void
gst_media_play_toggle_display_mode (GtkWidget *widget, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	if (mplay->display_mode == GST_MEDIA_PLAY_NORMAL){
		gst_media_play_set_display_mode(mplay, GST_MEDIA_PLAY_MINI);
	}
	else if (mplay->display_mode == GST_MEDIA_PLAY_MINI ||
	         mplay->display_mode == GST_MEDIA_PLAY_FULLSCREEN){
		gst_media_play_set_display_mode(mplay, GST_MEDIA_PLAY_NORMAL);
	}
}

static void
gst_media_play_display_mode_change (GtkWidget *widget, GstMediaPlayMode display_mode, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	gst_media_play_set_display_mode(mplay, display_mode);
}

static gboolean
hide_control_window (GstMediaPlay *mplay)
{
	static int height = 0;

	if (height == 0) {
		height = gdk_screen_height ();
	}

	gtk_window_move (GTK_WINDOW (mplay->control_window_fs),
			 0, height - mplay->control_y);
	mplay->control_y -= 2;

	if (mplay->control_y == 2) {
		return FALSE;
	} else {
		return TRUE;
	}
}

static gboolean
control_entered (GtkWidget *control,
		 GdkEventCrossing *event,
		 GstMediaPlay *mplay)
{
	if (mplay->move_id > 0) {
		g_source_remove (mplay->move_id);
		mplay->move_id = 0;
	}
	
	gtk_window_move (GTK_WINDOW (mplay->control_window_fs), 0,
			 gdk_screen_height () - mplay->control->allocation.height);
	mplay->control_y = mplay->control->allocation.height;
	return FALSE;
}

static gboolean
control_left (GtkWidget *control,
	      GdkEventCrossing *event,
	      GstMediaPlay *mplay)
{
	mplay->move_id = g_timeout_add (25, (GSourceFunc) hide_control_window, mplay);
}

static void
gst_media_play_set_fullscreen (GstMediaPlay *mplay, gboolean fullscreen)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
  
	if (fullscreen){
		GdkRectangle screen;
		GtkRequisition req;
		int height;

		/* return if we're in the right state */
		if (mplay->display_mode == GST_MEDIA_PLAY_FULLSCREEN){
			return;
		}
		

		/* get position of screen */
		/* FIXME: make me work with Xinerama, please.
		 * see galeon/.../window.c for an example */
		screen.x = 0;
		screen.y = 0;
		screen.height = gdk_screen_height();
		screen.width = gdk_screen_width();

		/* create new fullscreen widget */
		mplay->window_fs = GTK_WINDOW (gtk_window_new (GTK_WINDOW_POPUP));
		if (mplay->window_fs == NULL)
		{
			g_warning ("failed to create window for fullscreen\n");
			return;
		}
		gtk_window_move (mplay->window_fs, screen.x, screen.y);
		gtk_window_resize (mplay->window_fs, screen.width, screen.height);
		
		mplay->fs_vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (mplay->window_fs), mplay->fs_vbox);
/*  		gtk_grab_add (GTK_WIDGET (mplay->window_fs)); */

		g_object_ref(mplay->video_widget);
		gst_video_widget_reset_socket(GST_VIDEO_WIDGET(mplay->video_widget));
		gtk_container_remove(GTK_CONTAINER(mplay), mplay->video_widget);
		gtk_box_pack_start (GTK_BOX (mplay->fs_vbox), mplay->video_widget, TRUE, TRUE, 0);
		g_object_unref(mplay->video_widget);

		gst_play_need_new_video_window(mplay->play);

		/* Create a floating control bar */
		mplay->control_window_fs = gtk_window_new (GTK_WINDOW_POPUP);
		g_signal_connect (G_OBJECT (mplay->control_window_fs),
				  "enter-notify-event", G_CALLBACK (control_entered),
				  mplay);
		g_signal_connect (G_OBJECT (mplay->control_window_fs),
				  "leave-notify-event", G_CALLBACK (control_left),
				  mplay);
		
		g_object_ref(G_OBJECT(mplay->control));
		height = mplay->control->allocation.height;
		gtk_container_remove(GTK_CONTAINER(mplay), mplay->control);

		mplay->control_vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (mplay->control_window_fs),
				   mplay->control_vbox);
		gtk_box_pack_start (GTK_BOX (mplay->control_vbox),
				    mplay->control, FALSE, FALSE, 0);
				
		g_object_unref(mplay->control);

		gtk_widget_show (mplay->fs_vbox);
		gtk_widget_show (GTK_WIDGET (mplay->window_fs));
		gtk_widget_show (mplay->control_vbox);
		gtk_widget_show (mplay->control_window_fs);

		height = mplay->control->allocation.height;
		gtk_window_resize (GTK_WINDOW (mplay->control_window_fs),
				   screen.width, height);
		gtk_window_move (GTK_WINDOW (mplay->control_window_fs),
				 0, screen.height - height);

		mplay->control_y = height;
		/* Slide the control panel offscreen */
		mplay->move_id = g_timeout_add (25, (GSourceFunc)hide_control_window, mplay);
	}
	else {
		gtk_grab_remove (GTK_WIDGET (mplay->window_fs));
		gtk_widget_hide (GTK_WIDGET (mplay->window_fs));

		g_object_ref(mplay->video_widget);
		gst_video_widget_reset_socket(GST_VIDEO_WIDGET(mplay->video_widget));
		gtk_container_remove(GTK_CONTAINER(mplay->fs_vbox), mplay->video_widget);
		gtk_box_pack_start (GTK_BOX (mplay), mplay->video_widget, TRUE, TRUE, 0);
		g_object_unref(mplay->video_widget);

		gst_play_need_new_video_window(mplay->play);

		g_object_ref(G_OBJECT(mplay->control));
		gtk_container_remove (GTK_CONTAINER (mplay->control_vbox), mplay->control);
		gtk_box_pack_start (GTK_BOX (mplay), mplay->control, FALSE, FALSE, 0);
		g_object_unref(mplay->control);

		gtk_box_reorder_child(GTK_BOX (mplay), mplay->video_widget, 0);
		gtk_box_reorder_child(GTK_BOX (mplay), mplay->control, 1);

		gtk_widget_show (mplay->control);

		gtk_widget_destroy (mplay->control_window_fs);
		gtk_widget_destroy (GTK_WIDGET (mplay->window_fs));
		mplay->window_fs = NULL;
		mplay->fs_vbox = NULL;
		mplay->control_window_fs = NULL;
	}
}

void
gst_media_play_set_display_mode (GstMediaPlay *mplay, GstMediaPlayMode display_mode)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	gst_control_set_display_mode(GST_CONTROL(mplay->control), display_mode);
	switch (display_mode){
	case GST_MEDIA_PLAY_FULLSCREEN:
		gst_media_play_set_fullscreen (mplay, TRUE);
		break;
	case GST_MEDIA_PLAY_NORMAL:
		if (mplay->display_mode == GST_MEDIA_PLAY_FULLSCREEN){
			gst_media_play_set_fullscreen (mplay, FALSE);
		}
		gtk_widget_show (mplay->statusarea);
		break;
	case GST_MEDIA_PLAY_MINI:
		gtk_widget_hide (mplay->statusarea);
		break;
	default:
		g_warning("unknown mode %d\n", display_mode);
		break;
	}
	mplay->display_mode = display_mode;
	g_signal_emit (mplay, gst_media_play_signals [DISPLAY_MODE_CHANGE], 0, display_mode);
}

gfloat
gst_media_play_get_video_scale (GstMediaPlay *mplay)
{
	gfloat scale_factor = 1.0;
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	if (mplay->video_widget) {
		g_object_get (G_OBJECT (mplay->video_widget), "scale_factor", &scale_factor, NULL);
	}
	return scale_factor;
}

GstElementState
gst_media_play_get_state (GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	return gst_play_get_state (mplay->play);
}

GstMediaPlayMode
gst_media_play_get_display_mode (GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	return mplay->display_mode;
}

static void
gst_media_play_stream_end (GstPlay *play, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	gst_media_play_set_state (mplay, GST_STATE_READY);
}

static void
gst_media_play_time_tick (GstPlay *play, gint64 time_nanos, GstMediaPlay *mplay)
{
	gdouble seek_pos;
	gint seconds;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	seek_pos = ((gdouble) time_nanos) / ((gdouble) mplay->length_nanos);
	seconds = (gint) (time_nanos / GST_SECOND);
	gst_control_set_seek_pos (mplay->control, seek_pos);
	gst_status_area_set_time(GST_STATUS_AREA(mplay->statusarea), time_nanos);
}

static void
gst_media_play_got_length (GstPlay *play, gint64 length_nanos, GstMediaPlay *mplay)
{
	gint seconds;

	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	seconds = (gint) (length_nanos / GST_SECOND);
	mplay->length_nanos = length_nanos;
	gst_control_set_enable_seek (GST_CONTROL (mplay->control), TRUE);
	gst_status_area_set_length(GST_STATUS_AREA(mplay->statusarea), length_nanos);
}

static void
gst_media_play_state_change (GstPlay *play, GstElementState old_state, GstElementState new_state, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	gst_control_set_state (GST_CONTROL (mplay->control), new_state);
	gst_status_area_set_state (GST_STATUS_AREA (mplay->statusarea), old_state, new_state);
}

static void
gst_media_play_have_xid (GstPlay *play, gint xid, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));


	if (mplay->video_widget)
	{
		gtk_widget_show (mplay->video_widget);
		gst_video_widget_set_xembed_xid(GST_VIDEO_WIDGET(mplay->video_widget), xid);
		gst_control_allow_fullscreen (GST_CONTROL(mplay->control), TRUE);
	}

}

static void
gst_media_play_have_video_size (GstPlay *play, gint width, gint height, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));

	if (mplay->video_widget)
	{
		gst_video_widget_set_source_size(GST_VIDEO_WIDGET(mplay->video_widget), width, height);
	}
}

static void
gst_media_play_information (GstPlay *play, GstElement* element, GParamSpec *param, GstMediaPlay *mplay)
{
	g_return_if_fail(GST_IS_MEDIA_PLAY(mplay));
	gst_status_area_set_information(GST_STATUS_AREA(mplay->statusarea), element, param);
}

static GladeXML*
gst_media_play_get_glade_xml(const char *fname, const char *root, const char *domain)
{
	GladeXML *xml = NULL;
	gchar *full_fname;
	
	full_fname = g_strjoin("/", GLADEUI_UNINSTALLED_DIR, fname, NULL);
	if (g_file_test(full_fname, G_FILE_TEST_IS_REGULAR)){
		/* g_print("loading %s\n", full_fname); */
		xml = glade_xml_new(full_fname, root, domain);
		g_free(full_fname);
		return xml;
	}

	g_free(full_fname);
	full_fname = g_strjoin("/", GLADEUI_DIR, fname, NULL);
	if (g_file_test(full_fname, G_FILE_TEST_IS_REGULAR)){
		/* g_print("loading %s\n", full_fname); */
		xml = glade_xml_new(full_fname, root, domain);
		g_free(full_fname);
		return xml;
	}
	g_free(full_fname);
	return glade_xml_new(fname, root, domain);
}

/* FIXME: move this function ? */
gboolean
gst_media_gconf_get_boolean (const gchar *key, gboolean *value)
{
  gchar *full_key;
  GError *error = NULL;

  static GConfClient *_gst_media_gconf_client = NULL;

  if (!_gst_media_gconf_client)
    _gst_media_gconf_client = gconf_client_get_default ();

  full_key = g_strdup_printf ("%s/%s", GST_MEDIA_GCONF_DIR, key);
  *value = gconf_client_get_bool (_gst_media_gconf_client, full_key, &error);
  g_free (full_key);
  if (error)
  {
    g_warning ("GConf: %s\n", error->message);
    g_error_free (error);
    return FALSE;
  }
  return TRUE;
}
