/*
 * A GNOME front end for LinuxConf
 *
 * (C) 1997 the Free Software Foundation
 * Author: Miguel de Icaza (miguel@gnu.org)
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "gnome.h"

#define BSIZE 512
#define NFORM_STACK 16
#define GNOME_WINDOW "GNOME-Window"

static struct {
	GtkWidget  *table;
	int        column;
	int        line;
	GHashTable *hash;
} form_stack [NFORM_STACK];

int stack_top = 0;

#define current_table  form_stack [stack_top].table
#define current_column form_stack [stack_top].column
#define current_line   form_stack [stack_top].line
#define current_hash   form_stack [stack_top].hash
GHashTable *toplevels;
GtkWidget *current_window, *current_book;
GtkWidget *last_widget;

#define bind_sym(x,y) g_hash_table_insert (current_hash, x, y);

char **
split_string (char *str, int items)
{
	char **result;
	int  i;
	
	result = g_malloc (sizeof (char *) * items + 1);

	for (i = 0; i < items; i++){
		result [i] = str;

		while (*str && (!isspace (*str)))
			str++;
		
		if (*str){
			*str = 0;
			str++;
		}

		while (isspace (*str)){
			*str++;
		}
	}
	result [i] = str;
	return result;
}

GtkWidget *
new_table (void)
{
	return gtk_table_new (100, 100, 0);
}

void
maybe_skipit (void *data, void *data2)
{
	GtkTableChild *child = data;

	if (!((current_line >= child->top_attach) &&
	      (current_line < child->bottom_attach)))
		return;
	
	if ((current_column >= child->left_attach) &&
	    (current_column < child->right_attach))
		current_column = child->right_attach;
}


void
walk_over_allocated_space ()
{
	GtkTableChild *child;
	GtkTable *table = GTK_TABLE(current_table);
	GList *p;

	p = table->children;
	if (!p)
		return;

	g_list_foreach (table->children, maybe_skipit, NULL);
	return;
}

void
place (GtkWidget *widget)
{
	walk_over_allocated_space ();
	last_widget = widget;
	gtk_table_attach_defaults (GTK_TABLE (current_table), widget,
				   current_column, current_column + 1,
				   current_line,   current_line + 1);
	current_column++;
}

void
create_form (char *str)
{
	GtkWidget  *window, *table;
	GHashTable *wtab;
	char **strs;

	if (!form_stack [0].table){
		strs = split_string (str, 1);
		window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title (GTK_WINDOW(window), strs [1]);
		table = new_table ();
		gtk_widget_show (table);
		gtk_container_add (GTK_CONTAINER(window), table);
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		g_hash_table_insert (toplevels, strs [0], wtab);
		g_hash_table_insert (wtab, GNOME_WINDOW, window);
		
		current_hash   = wtab;
		current_window = window;
	} else {
		strs = split_string (str, 0);
		table = new_table ();
		gtk_widget_show (table);
		bind_sym (strs [0], table);
		place (table);

		/* nest */
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		gtk_object_set_user_data (GTK_OBJECT (table), wtab);
		stack_top++;
	}
	current_table  = table;
	current_column = current_line = 0;
}

void
create_book (char *str)
{
	GtkWidget *book;
	char **strs;

	strs = split_string (str, 1);
	book = gtk_notebook_new ();
	gtk_widget_show (book);
	bind_sym (strs [0], book);
	place (book);
	current_book = book;
}

void
create_button (char *str)
{
	GtkWidget *button;
	char **strs;

	strs = split_string (str, 1);
	button = gtk_button_new_with_label (strs [1]);
	gtk_widget_show (button);
	bind_sym (strs [0], button);
	place (button);
}

void
create_combo (char *str)
{
	
}

void
create_checkbox (char *str)
{
	GtkWidget *check;
	char **strs;

	strs = split_string (str, 2);
	check = gtk_check_button_new_with_label (strs [2]);
	gtk_widget_show (check);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(check), *strs [1] == '0' ? 0 : 1);
	bind_sym (strs [0], check);
	place (check);
}

void
create_group (char *str)
{
	GtkWidget *group, *table;
	GHashTable *wtab;
	char **strs;

	strs = split_string (str, 1);
	group = gtk_frame_new (strs [1]);
	gtk_widget_show (group);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	place (group);
	gtk_container_add (GTK_CONTAINER(group), table);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT(table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
}

void
create_label (char *str)
{
	GtkWidget *label;
	char **strs;

	label = gtk_label_new (str);
	gtk_widget_show (label);
	place (label);
}

void
create_page (char *str)
{
	GtkWidget *label, *table;
	GHashTable *wtab; 
	char **strs;

	strs = split_string (str, 1);
	label = gtk_label_new (strs [1]);
	gtk_widget_show (label);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	place (table);
	gtk_notebook_append_page (GTK_NOTEBOOK(current_book), table, label);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT (table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
}

void
create_string (char *str)
{
	GtkWidget *entry;
	char **strs;
	int  maxlen;
	
	strs = split_string (str, 2);
	maxlen = atoi (strs [1]);
	entry = gtk_entry_new ();
	gtk_widget_show (entry);
	gtk_entry_set_text (GTK_ENTRY(entry), strs [2]);
	place (entry);
	bind_sym (strs [0], entry);
}

void
create_hline (char *str)
{
	GtkWidget *sep;
	
	sep = gtk_hseparator_new ();
	gtk_widget_show (sep);
	place (sep);
}

void
newline (char *str)
{
	current_line++;
	current_column = 0;
}

void
end (char *str)
{
	if (stack_top)
		stack_top--;
	else {
		gtk_widget_show (current_window);
		/* bad hack: force MainForm  :-) */
		form_stack [0].table = 0;
	}
}

void
skip (char *str)
{
	current_column += atoi (str);
}

float
h_align (char *str)
{
	if (*str == 'l')
		return 0.0;
	if (*str == 'r')
		return 1.0;
	return 0.5;
}

float
v_align (char *str)
{
	if (*str == 't')
		return 0.0;
	if (*str == 'b')
		return 1.0;
	return 0.5;
}

/*
 * When we receive this command, it means that we have to change the layout of
 * the last inserted widget.
 * This also provides alignement contraints.
 */
void
dispolast (char *str)
{
	GtkWidget *align;
	char **strs;
	int  column;

	column = current_column - 1;
	strs = split_string (str, 4);
	gtk_container_remove (GTK_CONTAINER(current_table), last_widget);
	
	align = gtk_alignment_new (h_align (strs [0]), v_align (strs [2]), 1.0, 1.0);
	gtk_widget_show (align);
	gtk_table_attach_defaults (GTK_TABLE (current_table), align,
				   column,       column + atoi (strs [1]),
				   current_line, current_line + atoi (strs [3]));
	gtk_container_add (GTK_CONTAINER(align), last_widget);
}

struct {
	char *name;
	void (*cmd)(char *str);
} cmds [] = {
	{ "Book",      create_book },
	{ "Button",    create_button },
	{ "Checkbox",  create_checkbox },
	{ "Combo",     create_combo },
	{ "Dispolast", dispolast },
	{ "End",       end },
	{ "Form",      create_form },
	{ "Group",     create_group },
	{ "Hline",     create_hline },
	{ "Label",     create_label },
	{ "MainForm",  create_form  },
	{ "Newline",   newline },
	{ "Page",      create_page },
	{ "Skip",      skip },
	{ "String",    create_string },
	{ NULL, NULL }
};

void
process (char *str)
{
	int i;
		
	while (*str && (*str == ' ' || *str == '\t'))
		str++;

	for (i = 0; cmds [i].name; i++){
		int cmd_len = strlen (cmds [i].name);
		
		if (strncmp (cmds [i].name, str, cmd_len))
			continue;

		printf ("comando: %s\n", cmds [i].name);
		(*cmds [i].cmd) (str + cmd_len + 1);
	}
}

int
main (int argc, char *argv [])
{
	char buffer [BSIZE];
	char *s;
	
	gnome_init (&argc, &argv);
	toplevels = g_hash_table_new (g_string_hash, g_string_equal);
	
	while (fgets (buffer, BSIZE, stdin)){
		if ((s = strchr (buffer, '\n')))
			*s = 0;
		process (buffer);
	}
		gtk_main ();
	return 0;
}
