/*
 * $Id: g_dispatch.c,v 1.16 1998/08/06 19:14:54 jochen Exp $
 * GXSNMP -- An snmp management application
 * Copyright (C) 1998 Gregory McLean & Jochen Friedrich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Implementation of a SNMP dispatcher as of RFC2271
 */

#include "config.h"
#include "g_snmp.h"

/* int hash functions needed for old glibs not containing it.
 */

#ifndef HAVE_INT_HASH
gint
g_int_equal (gpointer v, gpointer v2)
{
  return *((const gint*) v) == *((const gint*) v2);
}

guint
g_int_hash (gpointer v)
{
  return *(const gint*) v;
}
#endif

/* This module defines the API to the SNMP RFC layer. Requests are routed
 * to the appropriate transport (e.g. IPv4 or IPv6 or IPX) by using the
 * message processing compatible with the given PDU version (V1, V2C,
 * or V3). Applications will prefer to use the sync or async event loop
 * API presented by the g_session layer.
 */

/* RFC2271 defines some dispatcher primitives as standard SNMPv3 API.
 * These names do not match GNU conventions. RFC2272 defines what exactly
 * these primitives are supposed to do.
 *
 * It appears to me these primitives are mainly designed to describe
 * interfaces between objects. I.e. the "modules" of RFC2271 translate
 * to "objects" and the "primitives" translate to "public methods".
 * In this case there should be a global registry (IR?) which keeps track
 * of all known "modules" and loads them at init time. The dispatcher
 * "module" (which should be guaranteed to exist only once -> singleton)
 * must be constructed first, so the other "modules" can register themselves.
 *
 * statusInformation =
 * sendPdu(
 *         IN   transportDomain                 (guint)
 *         IN   transportAddress                (struct sockaddr *)
 *         IN   messageProcessingModel          INTEGER (0..2147483647)
 *         IN   securityModel                   INTEGER (0..2147483647)
 *         IN   securityName                    (gpointer)
 *         IN   securityLevel                   INTEGER (1,2,3)
 *         IN   contextEngineID                 OCTET STRING (SIZE(1..32))
 *         IN   contextName                     SnmpAdminString (SIZE(0..32))
 *         IN   pduVersion                      (guint)
 *         IN   PDU                             (SNMP_PDU *)
 *         IN   expectResponse                  (gboolean)
 * )
 *
 * returnResponsePdu(
 *              messageProcessingModel          INTEGER (0..2147483647)
 *              securityModel                   INTEGER (0..2147483647)
 *              securityName                    (gpointer)
 *              securityLevel                   INTEGER (1,2,3)
 *              contextEngineID                 OCTET STRING (SIZE(1..32))
 *              contextName                     SnmpAdminString (SIZE(0..32))
 *              pduVersion                      (guint)
 *              PDU                             (SNMP_PDU *)
 *              maxSizeResponseScopedPDU        INTEGER (0..2147483647)
 *              stateReference                  (gpointer)
 *              statusInformation               (gint)
 * )
 *
 * statusInformation =
 * registerContextEngineID(
 *              contextEngineID                 OCTET STRING (SIZE(1..32))
 *              pduType                         INTEGER (1,2,3,4,5,6,7)
 * )
 *
 * unregisterContextID(
 *              contextEngineID                 OCTET STRING (SIZE(1..32))
 *              pduType                         INTEGER (1,2,3,4,5,6,7)
 * )
 *
 * Callback Functions. These must be provided by the application:
 *
 * processPdu(
 *              messageProcessingModel          INTEGER (0..2147483647)
 *              securityModel                   INTEGER (0..2147483647)
 *              securityName                    (gpointer)
 *              securityLevel                   INTEGER (1,2,3)
 *              contextEngineID                 OCTET STRING (SIZE(1..32))
 *              contextName                     SnmpAdminString (SIZE(0..32))
 *              pduVersion                      (guint)
 *              PDU                             (SNMP_PDU *)
 *              maxSizeResponseScopedPDU        INTEGER (0..2147483647)
 *              stateReference                  (gpointer)
 * )
 *
 * processResponsePdu(
 *              messageProcessingModel          INTEGER (0..2147483647)
 *              securityModel                   INTEGER (0..2147483647)
 *              securityName                    (gpointer)
 *              securityLevel                   INTEGER (1,2,3)
 *              contextEngineID                 OCTET STRING (SIZE(1..32))
 *              contextName                     SnmpAdminString (SIZE(0..32))
 *              pduVersion                      (guint)
 *              PDU                             (SNMP_PDU *)
 *              statusInformation               (gint)
 *              sendPduHandle                   (int)
 * )
 */

/* FIXME: Should be moved into a transport module with an init function
 *        For this it would be nice, it glib already had the event loop :)
 */

/* static registration hash tables. */

static GHashTable *message_models   = NULL;
static GHashTable *security_models  = NULL;
static GHashTable *access_models    = NULL;
static GHashTable *transport_models = NULL;

int 
sendPdu(guint transportDomain, struct sockaddr *transportAddress,
        guint messageProcessingModel, guint securityModel,
        GString *securityName, int securityLevel,
        GString *contextEngineID, GString *contextName,
        guint pduVersion, SNMP_PDU *PDU, gboolean expectResponse)
{
  gboolean            result;
  guint               destTransportDomain;
  struct sockaddr    *destTransportAddress;  
  gpointer            outgoingMessage;
  guint               outgoingMessageLength;
  guint               sendPduHandle;
  struct g_message   *msg_model;
  struct g_transport *trp_model;

/* RFC2272 explain how to send SNMP messages in 4.1.1
 *
 * Step 1) application calls our primitive
 * Step 2) we check for a known Message Processing Model
 */

  if (!message_models)
    return FALSE;

  if (!(msg_model = g_hash_table_lookup(message_models, 
      &messageProcessingModel))) 
    return FALSE;

/* Step 3) we generate a sendPduHandle
 * FIXME: don't do it such stupid! SNMPv1 and SNMPv2 don't use this value
 * so currently we just fake it
 */

  sendPduHandle=1;

/* Step 4) sending request to version-specific Message Processing module.
 */

  result = msg_model->prepareOutgoingMessage(transportDomain, transportAddress,
             messageProcessingModel, securityModel, securityName, 
             securityLevel, contextEngineID, contextName,
             pduVersion, PDU, expectResponse, sendPduHandle,
             &destTransportDomain, &destTransportAddress,
             &outgoingMessage, &outgoingMessageLength);

/* Step 5) if the statusInformation indicates an error, return.
 */

  if (!result) return -1;

/* Step 6) send message via indicated transport.
 */

  if (!transport_models)
    return FALSE;

  if (!(trp_model = g_hash_table_lookup(transport_models,
      &destTransportDomain)))
    return FALSE;

  if (!trp_model->sendMessage(destTransportAddress, outgoingMessage, 
                              outgoingMessageLength))
    {
      g_free(outgoingMessage);
      return FALSE;
    }
  g_free(outgoingMessage);
  return sendPduHandle;
}

gboolean
returnResponsePdu(guint messageProcessingModel, guint securityModel,
                  GString *securityName, int securityLevel, 
                  GString *contextEngineID, GString *contextName, 
                  guint pduVersion, SNMP_PDU *PDU, int maxSizeResponseScopedPDU,
                  gpointer stateReference, int statusInformation)
{
  gboolean            result;
  guint               destTransportDomain;
  struct sockaddr    *destTransportAddress;
  gpointer            outgoingMessage;
  guint               outgoingMessageLength;
  struct g_message   *msg_model;
  struct g_transport *trp_model;

/* RFC2272 explain how to send SNMP response messages in 4.1.2
 *
 * Step 1) application calls our primitive
 * Step 2) we check for a known Message Processing Model then call model
 */

  if (!message_models)
    return FALSE;

  if (!(msg_model = g_hash_table_lookup(message_models,
      &messageProcessingModel)))
    return FALSE;

  result = msg_model->prepareResponseMessage(messageProcessingModel, 
             securityModel, securityName, securityLevel, contextEngineID, 
             contextName, pduVersion, PDU, maxSizeResponseScopedPDU,
             stateReference, statusInformation,
             &destTransportDomain, &destTransportAddress,
             &outgoingMessage, &outgoingMessageLength);

/* Step 3) if the statusInformation indicates an error, return.
 */

  if (!result) return FALSE;

/* Step 4) send message via indicated transport.
 */

  if (!transport_models)
    return FALSE;

  if (!(trp_model = g_hash_table_lookup(transport_models,
      &destTransportDomain)))
    return FALSE;

  if (!trp_model->sendMessage(destTransportAddress, outgoingMessage, 
                              outgoingMessageLength))
    {
      g_free(outgoingMessage);
      return FALSE;
    }
  g_free(outgoingMessage);
  return TRUE;
}

gboolean
g_register_message(guint model_nr, struct g_message *msg)
{
  guint *ptr;

  if (g_hash_table_lookup(message_models, &model_nr)) return FALSE;
  ptr = g_malloc(sizeof(guint));
  *ptr = model_nr;
  g_hash_table_insert(message_models, ptr, msg);
  return TRUE;
}

gboolean
g_register_security(guint model_nr, struct g_security *sec)
{
  guint *ptr;

  if (g_hash_table_lookup(security_models, &model_nr)) return FALSE;
  ptr = g_malloc(sizeof(guint));
  *ptr = model_nr;
  g_hash_table_insert(security_models, ptr, sec);
  return TRUE;
}

gboolean
g_register_access(guint model_nr, struct g_access *acc)
{
  guint *ptr;

  if (g_hash_table_lookup(access_models, &model_nr)) return FALSE;
  ptr = g_malloc(sizeof(guint));
  *ptr = model_nr;
  g_hash_table_insert(access_models, ptr, acc);
  return TRUE;
}

gboolean
g_register_transport(guint model_nr, struct g_transport *tpt)
{
  guint *ptr;

  if (g_hash_table_lookup(transport_models, &model_nr)) return FALSE;
  ptr = g_malloc(sizeof(guint));
  *ptr = model_nr;
  g_hash_table_insert(transport_models, ptr, tpt);
  return TRUE;
}

void
g_receive_message(int transportDomain, struct sockaddr *transportAddress,
                  gpointer wholeMsg, guint wholeMsgLength)
{
  ASN1_SCK asn1;
  guint cls, con, tag, version, snmp_len;
  guchar *eoc, *end;
  gpointer snmp;
  struct g_message *msg_model;
  guint messageProcessingModel;
  guint securityModel;
  GString *securityName;
  int securityLevel;
  GString *contextEngineID;
  GString *contextName;
  guint pduVersion;
  SNMP_PDU PDU;
  guint pduType;
  int sendPduHandle;
  guint maxSizeResponseScopedPDU;
  guint statusInformation;
  gpointer stateReference;
  gint result;

/* 
 * Step 1
 * increment snmpInPkts
 */

/* not yet */

/*
 * Step 2
 * determine SNMP version
 */

  g_asn1_open (&asn1, wholeMsg, wholeMsgLength, ASN1_DEC);
  if (!g_asn1_header_decode (&asn1, &eoc, &cls, &con, &tag))
    return;
  if (cls != ASN1_UNI || con != ASN1_CON || tag != ASN1_SEQ)
    return;
  if (!g_asn1_header_decode (&asn1, &end, &cls, &con, &tag))
    return;
  if (cls != ASN1_UNI || con != ASN1_PRI || tag != ASN1_INT)
    return;
  if (!g_asn1_uint_decode (&asn1, end, &version))
    return;
  g_asn1_close (&asn1, snmp, &snmp_len);

/*
 * Step 3
 * (how can this be after step 2? Shouldn't address and domain be
 *  obvious after the recvfrom() call?)
 * This obviously must be yet another proxy thing...
 */

/*
 * Step 4
 * Pass message to Message Processing Model
 */

  if (!message_models)
    return;

  if (!(msg_model = g_hash_table_lookup(message_models, 
      &version))) 
    return;

  result = msg_model->prepareDataElements(transportDomain, transportAddress, 
             wholeMsg, wholeMsgLength, &messageProcessingModel, &securityModel,
             &securityName, &securityLevel, &contextEngineID, &contextName,
             &pduVersion, &PDU, &pduType, &sendPduHandle, 
             &maxSizeResponseScopedPDU, &statusInformation, &stateReference);

/*
 * Step 5
 * test for FAILURE errorIndication and discard if yes
 */

/*
 * Step 6
 * Now dispatch PDU (back) to application dependent on sendPduHandle
 */ 

if (sendPduHandle)
  {
    /* Using processResponsePDU primitive to dispatch message */
    g_session_response_pdu(messageProcessingModel, securityModel,
      securityName, securityLevel, contextEngineID, contextName, 
      pduVersion, &PDU);
  }
}

gboolean
g_lookup_address (guint model_nr, guchar *hostname, struct sockaddr **address)
{
  struct g_transport *trp_model;

  if (!transport_models)
    return FALSE;

  if (!(trp_model = g_hash_table_lookup(transport_models, &model_nr)))
    return FALSE;

  return trp_model->resolveAddress(hostname, address);
}

static void
fe_transport(gpointer key, gpointer value, gpointer userdata)
{
  struct g_transport *trp_model;
  GXINITCB initcb;

  trp_model = (struct g_transport *) value;
  initcb    = (GXINITCB) userdata;
  initcb(trp_model->getSocket(), trp_model->receiveMessage);
}

gboolean
g_snmp_init(gboolean dobind, GXINITCB initcb)
{
  message_models   = g_hash_table_new (g_int_hash, g_int_equal);
  security_models  = g_hash_table_new (g_int_hash, g_int_equal);
  access_models    = g_hash_table_new (g_int_hash, g_int_equal);
  transport_models = g_hash_table_new (g_int_hash, g_int_equal);

/* Init all required models by RFC2271. Any private model should be
 * initialized after calling snmp_init().
 */ 

  if (!g_message_init())
    return FALSE;

  if (!g_transport_init(dobind))
    return FALSE;

  if (initcb)
    g_hash_table_foreach(transport_models, fe_transport, (gpointer) initcb);

  return TRUE;
#if 0
  g_security_init() &&
  g_access_init();
#endif
}

void g_snmp_reinit(GXINITCB initcb)
{
  if (initcb)
    g_hash_table_foreach(transport_models, fe_transport, (gpointer) initcb);
}

/* EOF */
