/*
 * GXSNMP -- An snmp mangament application
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>   
#include <getopt.h>
#include <strings.h>
#include <signal.h>
#include <libintl.h>

#include "collector.h"

char *cnffile = "collector.conf";
char *outfile = "collector.data";
int debug=0;
int pdusize=10;
int maxreq=10;
int defer=300;
int retry=5;
int timeout=5;

void flush_output(void);
void init_output(void);

void hupsig(int sig) {
  flush_output();
  reget_static();
  signal(SIGHUP, hupsig);
}

void init_env(void)
{
  int child;

  if (!debug) {
    child = fork();
    if (child == -1) {
      exit(1);
    }
    if (child == 0) {
      close(0);
      close(1);
      close(2);
      setsid();
    } else {
      exit(0);
    }
  }
/* init_mib(); */
  init_output();
  signal(SIGHUP, hupsig);
}

void usage(void)
{
  printf(gettext("\n"
         "Usage: collector [-c config] [-o outfile] [-p pdusize] [-m maxreq]\n"
         "                 [-t timeout] [-r retry] [-d defer] [-x] [-h]\n"
         "\n"
         "-c config:    configuration file to be used.\n"
         "-o outfile:   file where collected data should be written to.\n"
         "-p pdusize:   max. number of variables requested in one PDU.\n"
         "-m maxreq:    max. number of requests sent to network agents. The higher\n"
         "              this number the better the performance, but also the higher\n"
         "              the impact on the network\n"
         "-t timeout:   time to wait for a SNMP response\n"
         "-r retry:     number of retries until a node is considered unreachable.\n"
         "-d defer:     time until an unreachable node is asked again.\n"
         "-x            debug flag.\n"
         "-h            print this message.\n\n"));
  exit(1);
}

int main(int argc, char *argv[])
{
  static const struct option long_options[] = {
    {"config",  required_argument, 0, 'c'},
    {"outfile", required_argument, 0, 'o'},
    {"pdusize", required_argument, 0, 'p'},
    {"maxreq",  required_argument, 0, 'm'},
    {"retry",   required_argument, 0, 'r'},
    {"timeout", required_argument, 0, 't'},
    {"defer",   required_argument, 0, 'd'},
    {"debug",   no_argument,       0, 'x'},
    {"help",    no_argument,       0, 'h'},
    {0, 0, 0, 0}
  };
  int oi = 0;
  int c;
  int tim=900;

  while (1) {
    c = getopt_long(argc, argv, "c:o:p:m:t:r:d:xh?", long_options,
                    &oi);

    if (c == -1)
      break;

    switch(c) {
      case 'c':
        cnffile = optarg;
        break;
      case 'o':
        outfile = optarg;
        break;
      case 'p':
        pdusize = atoi(optarg);
        break;
      case 'm':
        maxreq = atoi(optarg);
        break;
      case 't':
        timeout = atoi(optarg);
        break;
      case 'r':
        retry = atoi(optarg);
        break;
      case 'd':
        defer = atoi(optarg);
        break;
      case 'x':
        debug++;
        break;
      default:
        usage();
        break;
    }
  }
  init_env();
  read_config();
  process_table();
  return 0;
}

