/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <glib.h>

#include "calc.h"
#include "eval.h"
#include "util.h"
#include "funclib.h"

int yyparse(void);

extern FILE *yyin;
extern char *yytext;

/* stack ... has to be global:-( */
evalstack_t evalstack={NULL,0,-1};

/*error .. global as well*/
error_t error_num;

/*the current state of the calculator*/
calcstate_t calcstate;

/*error reporting function*/
void (*errorout)(char *)=NULL;

/*round off the number at some digits*/
void
makemaxdigits(char *s,int digits)
{
	int i;
	int sd=0; /*digit where the number starts*/

	if(s[0]=='-')
		sd=1;

	if(!s || digits<=0)
		return;

	digits+=sd;

	if(strlen(s)<=digits)
		return;

	if(s[digits]<'5') {
		s[digits]='\0';
		return;
	}
	s[digits]='\0';

	for(i=digits-1;i>=sd;i--) {
		if(s[i]<'9') {
			s[i]++;
			return;
		}
		s[i]='\0';
	}
	shiftstr(s,1);
	s[sd]='1';
}


/*formats a floating point with mantissa in p and exponent in e*/
char *
formatfloat(char *p,long int e)
{
	long int len;
	int i;
	if(((e-1)<-8 || (e-1)>8) ||
		calcstate.scientific_notation) {
		p=my_realloc(p,strlen(p)+1,
			strlen(p)+1+((int)log10(abs(e))+2)+1);
		if(p[0]=='-') {
			if(strlen(p)>2) {
				shiftstr(p+2,1);
				p[2]='.';
			}
		} else {
			if(strlen(p)>1) {
				shiftstr(p+1,1);
				p[1]='.';
			}
		}
		sprintf(p,"%se%ld",p,e-1);
	} else if(e>0) {
		len=strlen(p);
		if(p[0]=='-')
			len--;
		if(e>len) {
			p=my_realloc(p,strlen(p)+1,
				strlen(p)+1+e-len);
			for(i=0;i<e-len;i++)
				strcat(p,"0");
		} else if(e<len) {
			if(p[0]=='-') {
				shiftstr(p+1+e,1);
				p[e+1]='.';
			} else {
				shiftstr(p+e,1);
				p[e]='.';
			}
		}
	} else { /*e<=0*/
		p=my_realloc(p,strlen(p)+1,
			strlen(p)+1+(-e)+2);
		if(p[0]=='-') {
			shiftstr(p+1,2+(-e));
			p[1]='0';
			p[2]='.';
			for(i=0;i<(-e);i++)
				p[i+3]='0';
		} else {
			shiftstr(p,2+(-e));
			p[0]='0';
			p[1]='.';
			for(i=0;i<(-e);i++)
				p[i+2]='0';
		}
	}
	return p;
}

char *
getnumstring_f(mpf_t num)
{
	char *p;
	long e;

	p=mpf_get_str(NULL,&e,10,0,num);
	makemaxdigits(p,calcstate.max_digits);
	p=formatfloat(p,e);

	return p;
}

char *
getnumstring_z(mpz_t num)
{
	char *p,*p2;
	mpf_t fr;

	p=mpz_get_str(NULL,10,num);
	if(calcstate.max_digits>0 &&
		calcstate.max_digits<strlen(p)) {
		mpf_init(fr);
		mpf_set_z(fr,num);
		p2=getnumstring_f(fr);
		mpf_clear(fr);
		if(strlen(p2)>=strlen(p)) {
			g_free(p2);
			return p;
		} else  {
			g_free(p);
			return p2;
		}
	}
	return p;
}

char *
getnumstring_q(mpq_t num)
{
	char *p,*p2;
	mpf_t fr;

	p=mpz_get_str(NULL,10,mpq_numref(num));
	p=appendstr(p,"/");
	p2=mpz_get_str(NULL,10,mpq_denref(num));
	p=appendstr(p,p2);
	g_free(p2);
	if(calcstate.max_digits>0 &&
		calcstate.max_digits<strlen(p)) {
		mpf_init(fr);
		mpf_set_q(fr,num);
		p2=getnumstring_f(fr);
		mpf_clear(fr);
		if(strlen(p2)>=strlen(p)) {
			g_free(p2);
			return p;
		} else  {
			g_free(p);
			return p2;
		}
	}
	return p;
}

/*make a string representation of an expression*/
char *
makeexprstr(char *s,tree_t *n)
{
	char *p;
	char tmp[20];
	int i;
	mpf_t fr;

	if(!n)
		return s;

	if(n->type==NUMBER_NODE) {
		if(n->data.number.type==FLOAT_TYPE) {
			p=getnumstring_f(n->data.number.data.fval);
		} else if(n->data.number.type==INTEGER_TYPE) {
			if(calcstate.results_as_floats) {
				mpf_init(fr);
				mpf_set_z(fr,n->data.number.data.ival);
				p=getnumstring_f(fr);
				mpf_clear(fr);
			} else
				p=getnumstring_z(n->data.number.data.ival);
		} else { /*type==RATIONAL_TYPE*/
			if(calcstate.results_as_floats) {
				mpf_init(fr);
				mpf_set_q(fr,n->data.number.data.rval);
				p=getnumstring_f(fr);
				mpf_clear(fr);
			} else
				p=getnumstring_q(n->data.number.data.rval);
		}
		if(!s)
			return p;
		s=appendstr(s,p);
		g_free(p);
		return s;
	} else if(n->type==ACTION_NODE) {
		if(n->data.action.type==PRIMITIVE_TYPE) {
			s=appendstr(s,"(");
			s=makeexprstr(s,n->left);

			primstr(tmp,n->data.action.data.primitive);
			s=appendstr(s,tmp);

			s=makeexprstr(s,n->right);
			s=appendstr(s,")");
			return s;
		} else { /*type==FUNCTION_TYPE*/
			s=appendstr(s,n->data.action.data.function.func->id);
			if(n->data.action.data.function.func->args <= 0)
				return s;
			s=appendstr(s,"(");
			s=makeexprstr(s,
				n->data.action.data.function.args[0]);
			for(i=1;i<n->data.action.data.function.func->args;i++) {
				s=appendstr(s,",");
				s=makeexprstr(s,
					n->data.action.data.function.args[i]);
			}

			s=makeexprstr(s,n->right);
			s=appendstr(s,")");
			return s;
		}
	}
	s=appendstr(s,"(???)");
	return s;
}

char *
evalexp(char * str, calcstate_t state,void (*errorfunc)(char *))
{
	int fd[2];
	char * p;
	static int first_time=TRUE;

	if(first_time)
		funclib_addall();
	first_time=FALSE;

	errorout=errorfunc;

	error_num=NO_ERROR;

	/*set the state variable for calculator*/
	calcstate=state;
	
	mpf_set_default_prec(state.float_prec);
	mp_set_memory_functions(g_malloc,my_realloc,my_free);

	pipe(fd);
	yyin=fdopen(fd[0],"r");

	if(calcstate.notation_in==INFIX_NOTATION)
		write(fd[1],"infix ",6);
	else if(calcstate.notation_in==POSTFIX_NOTATION)
		write(fd[1],"postfix ",8);
	else
		write(fd[1],"prefix ",7);
	write(fd[1],str,strlen(str));
	close(fd[1]);

	while(yyparse() && !feof(yyin))
		;

	close(fd[0]);

	/*catch parsing errors*/
	if(error_num!=NO_ERROR) {
		while(evalstack.top>-1)
			freetree(t_pop(&evalstack));
		return NULL;
	}

	/*stack is supposed to have only ONE entry*/
	if(evalstack.top!=0) {
		while(evalstack.top>-1)
			freetree(t_pop(&evalstack));
		(*errorout)("ERROR: Probably corrupt stack!");
		return NULL;
	}

	evalstack.stack[0]=evalnode(evalstack.stack[0]);

	/*catch evaluation errors*/
	if(error_num!=NO_ERROR) {
		while(evalstack.top>-1)
			freetree(t_pop(&evalstack));
		return NULL;
	}

	p=makeexprstr(NULL,evalstack.stack[0]);

	freetree(t_pop(&evalstack));

	return p;
}

void
yyerror(char *s)
{
	char *out=NULL;
	out=appendstr(out,"ERROR: ");
	out=appendstr(out,s);
	out=appendstr(out," at '");
	out=appendstr(out,yytext);
	out=appendstr(out,"'");
	
	(*errorout)(out);
	g_free(out);
	error_num=PARSE_ERROR;
}
