/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Type;

/**
 * A single RadioButton performs the same basic function as a {@link CheckButton},
 * as it's position in the object hierarchy reflects.  It is only when
 * multiple RadioButtons are grouped together that they become a different
 * user interface component.
 * <p>
 * Every RadioButton is a member of some group of RadioButtons.  When
 * one is selected, all of the other RadioButtons in the same group are
 * deselected.
 * <p>
 * A RadioButton is created by the constructor passing a <code>null</code>
 * value for the RadioButton objection for the first object, and the First
 * object as a parameter for the remaining objects.
 */
public class RadioButton extends CheckButton {
	
	
	
	public RadioButton(RadioMenuItem[] group) {
		super(init1(group));
	}
	
	private static int init1(RadioMenuItem[] group) {
		int[] groupHandle;
		if (null != group) {
			groupHandle = new int[group.length];
			for (int i = 0; i < group.length; i++) {
				groupHandle[i] = group[i].getHandle();
			}
		} else
			groupHandle = null;
		return gtk_radio_button_new(groupHandle);
	}

	
	public RadioButton(RadioButton[] group, String label, boolean hasMnemonic) {
		super(init2(group, label, hasMnemonic));
	}
	
	private static int init2(RadioButton[] group, String label, boolean hasMnemonic) {
		int[] groupHandle;
		if (null != group) {
			groupHandle = new int[group.length];
			for (int i = 0; i < group.length; i++) {
				groupHandle[i] = group[i].getHandle();
			}
		} else
			groupHandle = null;
		if (hasMnemonic) {
			return gtk_radio_button_new_with_mnemonic(groupHandle, label);
		} else {
			return gtk_radio_button_new_with_label(groupHandle, label);
		}
	}
	
	/**
	 * Create a new RadioButton object adding it to the same <i>group</i>
	 * as the provided RadioButton.  If this is the first RadioButton
	 * pass <code>null</code> for this parameter.
	 * 
	 * @param group A RadioButton that belongs to the group that we wish
	 * to add this newly constructed RadioButton to.  If this is the first
	 * RadioButton in the group just pass <code>null</code>.
	 */
	public RadioButton(RadioButton group) {
		super(init3(group));
	}
	
	private static int init3(RadioButton group) {
		int hndl;
		if (null == group)
			hndl =0;
		else
			hndl = group.getHandle();
		return RadioButton.gtk_radio_button_new_from_widget(hndl);
	}
	
	/**
	 * Create a new RadioButton object adding it to the same <i>group</i>
	 * as the provided RadioButton.  If this is the first RadioButton
	 * pass <code>null</code> for this parameter.
	 * 
	 * @param group A RadioButton that belongs to the group that we wish
	 * to add this newly constructed RadioButton to.  If this is the first
	 * RadioButton in the group just pass <code>null</code>.
	 * @param label The text label to assign to this RadioButton.
	 * @param hasMnemonic An indicator to inform the widget if the label
	 * contains a mnemonic.
	 */
	public RadioButton(RadioButton group, String label, boolean hasMnemonic) {
		super(init4(group, label, hasMnemonic));
	}
	
	private static int init4(RadioButton group, String label, boolean hasMnemonic) {
		int hndl;
		if (null == group)
			hndl = 0;
		else
			hndl = group.getHandle();
		if (hasMnemonic) {
			return gtk_radio_button_new_with_mnemonic_from_widget(hndl, label);
		} else {
			return gtk_radio_button_new_with_label_from_widget(hndl, label);
		}
	}
	
	/**
	 * Construct a radio button using a handle to a native resource.
	 */
	public RadioButton(int handle) {
	    super(handle);
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_radio_button_get_type());
	}


	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_radio_button_get_type();
	native static final protected int gtk_radio_button_new(int[] group);
	native static final protected int gtk_radio_button_new_from_widget(int group);
	native static final protected int gtk_radio_button_new_with_label(int[] group, String label);
	native static final protected int gtk_radio_button_new_with_label_from_widget(int group, String label);
	native static final protected int gtk_radio_button_new_with_mnemonic(int[] group, String label);
	native static final protected int gtk_radio_button_new_with_mnemonic_from_widget(int group, String label);
	native static final protected int[] gtk_radio_button_get_group(int button);
	native static final protected void gtk_radio_button_set_group(int button, int[] group);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
