/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.pango;
import org.gnu.glib.GObject;
/** 
 * While complete access to the layout capabilities of Pango is provided using
 * the detailed interfaces for itemization and shaping, using that functionality
 * directly involves writing a fairly large amount of code. The objects and
 * functions in this structure provide a high-level driver for formatting entire
 * paragraphs of text at once.
 *
 * <p>The PangoLayout structure represents and entire paragraph of text. It is
 * initialized with a PangoContext, UTF-8 string and set of attributes for that
 * string. Once that is done, the set of formatted lines can be extracted from
 * the object, the layout can be rendered, and conversion between logical
 * character positions within the layout's text, and the physical position of
 * the resulting glyphs can be made.
 */
public class Layout extends GObject 
{
	
	/**
	 * Construct a new Layout with the given Context.
	 * 
	 * @param The context to use for the Layout construction.
	 */
	public Layout(Context context) {
		super(Layout.pango_layout_new(context.getHandle()));
	}
	
	/**
	 * Construct a new Layout with a handle that has been returned
	 * from a native call.
	 * 
	 * @param handle The handle to the native resource.
	 */
	public Layout(int handle) {
		super(handle);
	}

	/**
	 * Set the text of the layout.
	 */
	public void setText(String text){
		pango_layout_set_text(getHandle(), text, text.length());
	}

	/**
	 * Gets the text in the layout.
	 */
	public String getText(){
		return pango_layout_get_text(getHandle());
	}

	/**
	 * Sets the layout text and attribute list from marked-up text (see markup
	 * format). Replaces the current text and attribute list.
	 *
	 * <p>If accelMarker is nonzero, the given character will mark the character
	 * following it as an accelerator. For example, the accel marker might be an
	 * ampersand or underscore. All characters marked as an accelerator will
	 * receive a {@link Underline#LOW} attribute. Two accelMarker characters
	 * following each other produce a single literal accelMarker character.
	 *
	 * @param markup some marked-up text
	 * @param accelMarker : marker for accelerators in the text
	 */
	public void setMarkup(String markup, char accelMarker){
		pango_layout_set_markup_with_accel(getHandle(), markup, markup.length(), (byte) accelMarker, (byte) '0');
	}

	/**
	 * Same as {@link #setMarkup(String, char)}, but the markup text isn't
	 * scanned for accelerators.
	 */
	public void setMarkup(String markup){
		pango_layout_set_markup(getHandle(), markup, markup.length());
	}


	/**
	 * Sets the text attributes for a layout object
	 */
	public void setAttributes(AttrList attributes){
		pango_layout_set_attributes(getHandle(), attributes.getHandle());
	}

	/**
	 * Gets the attribute list for the layout, if any
	 */
	public AttrList getAttributes(){
		return new AttrList( pango_layout_get_attributes (getHandle()) );
	}

	/**
	 * Set the default font description for the layout. If no font description
	 * is set on the layout, the font description from the layout's context is
	 * used.
	 */
	public void setFontDescription(FontDescription desc){
		pango_layout_set_font_description(getHandle(), desc.getHandle());
	}

	/**
	 * Sets the width to which the lines of the PangoLayout should be wrapped.
	 */
	public void setWidth(int width){
		pango_layout_set_width(getHandle(), width);
	}

	/**
	 * Gets the width to which the lines of the PangoLayout should be wrapped.
	 */
	public int getWidth(){
		return pango_layout_get_width(getHandle());
	}

	/**
	 * Sets the wrap style; the wrap style only has an effect if a width is set
	 * on the layout with {@link #setWidth(int)}To turn off wrapping, set
	 * the width to -1.
	 */
	public void setWrapStyle(WrapMode wrap){
		pango_layout_set_wrap(getHandle(), wrap.getValue());
	}

	/**
	 * Get the wrap mode for the layout.
	 */
	public WrapMode getWrapMode(){
		return WrapMode.intern( pango_layout_get_wrap(getHandle()) );
	}

	/**
	 * Sets the amount by which the first line should be shorter than the rest
	 * of the lines. This may be negative, in which case the subsequent lines
	 * will be shorter than the first line. (However, in either case, the entire
	 * width of the layout will be given by the value
	 */
	public void setIndent(int indent){
		pango_layout_set_indent(getHandle(), indent);
	}

	/**
	 * Gets the amount by which the first line should be shorter than the rest
	 * of the lines.
	 */
	public int getIndent(){
		return pango_layout_get_indent(getHandle());
	}

	/**
	 * Gets the amount of spacing between the lines of the layout.
	 * @return  the spacing (in thousandths of a device unit)
	 */
	public int getSpacing(){
		return pango_layout_get_spacing(getHandle());
	}

	/**
	 * Sets the amount of spacing between the lines of the layout.
	 */
	public void setSpacing(int spacing){
		pango_layout_set_spacing(getHandle(), spacing);
	}

	/**
	 * Sets whether or not each complete line should be stretched to fill the
	 * entire width of the layout. This stretching is typically done by adding
	 * whitespace, but for some scripts (such as Arabic), the justification is
	 * done by extending the characters.
	 */
	public void setJustification(boolean justify){
		pango_layout_set_justify(getHandle(), justify);
	}

	/**
	 * Gets whether or not each complete line should be stretched to fill the
	 * entire width of the layout.
	 */
	public boolean getJustified(){
		return pango_layout_get_justify(getHandle());
	}

	/**
	 * Sets the alignment for the layout (how partial lines are positioned
	 * within the horizontal space available.)
	 */
	public void setAlignment(Alignment alignment){
		pango_layout_set_alignment(getHandle(), alignment.getValue());
	}

	/**
	 * Gets the alignment for the layout (how partial lines are positioned
	 * within the horizontal space available.)
	 */
	public Alignment getAlignment(){
		return Alignment.intern( pango_layout_get_alignment(getHandle()) );
	}

	/**
	 * Sets the tabs to use for layout, overriding the default tabs (by default,
	 * tabs are every 8 spaces).
	 */
	public void setTabs(TabArray tabs){
		pango_layout_set_tabs(getHandle(), tabs.getHandle());
	}

	/**
	 * If setting is TRUE, do not treat newlines and similar characters as
	 * paragraph separators; instead, keep all text in a single paragraph, and
	 * display a glyph for paragraph separator characters. Used when you want to
	 * allow editing of newlines on a single text line.
	 */
	public void setSingleParagraphMode(boolean setting){
		pango_layout_set_single_paragraph_mode(getHandle(), setting);
	}

	/**
	 * Obtains the value set by {@link #setSingleParagraphMode}.
	 */
	public boolean getSingleParagraphMode(){
		return pango_layout_get_single_paragraph_mode(getHandle());
	}

	/**
	 * Retrieve the count of lines for the layout
	 */
	public int getLineCount(){
		return pango_layout_get_line_count(getHandle());
	}

	/**
	 * Retrieves a particular line from a PangoLayout.
	 * @param line  the index of a line, which must be between 0 and
	 * pango_layout_get_line_count(layout) - 1, inclusive.
	 * @return  the requested PangoLayoutLine.
	 */
	public LayoutLine getLine(int line){
		return new LayoutLine(pango_layout_get_line(getHandle(), line));
	}

	



	 
	/****************************************
     * BEGINNING OF GENERATED CODE
     ****************************************/
    native static final protected int pango_layout_get_type ();
    native static final protected int pango_layout_new (int context);
    native static final protected int pango_layout_copy (int src);
    native static final protected int pango_layout_get_context (int layout);
    native static final protected void pango_layout_set_attributes (int layout, int attrs);
    native static final protected int pango_layout_get_attributes (int layout);
    native static final protected void pango_layout_set_text (int layout, String text, int length);
    native static final protected String pango_layout_get_text (int layout);
    native static final protected void pango_layout_set_markup (int layout, String markup, int 
        length);
    native static final protected void pango_layout_set_markup_with_accel (int layout, String 
        markup, int length, byte accelMarker, byte accelChar);
    native static final protected void pango_layout_set_font_description (int layout, int desc);
    native static final protected void pango_layout_set_width (int layout, int width);
    native static final protected int pango_layout_get_width (int layout);
    native static final protected void pango_layout_set_wrap (int layout, int wrap);
    native static final protected int pango_layout_get_wrap (int layout);
    native static final protected void pango_layout_set_indent (int layout, int indent);
    native static final protected int pango_layout_get_indent (int layout);
    native static final protected void pango_layout_set_spacing (int layout, int spacing);
    native static final protected int pango_layout_get_spacing (int layout);
    native static final protected void pango_layout_set_justify (int layout, boolean justify);
    native static final protected boolean pango_layout_get_justify (int layout);
    native static final protected void pango_layout_set_alignment (int layout, int alignment);
    native static final protected int pango_layout_get_alignment (int layout);
    native static final protected void pango_layout_set_tabs (int layout, int tabs);
    native static final protected int pango_layout_get_tabs (int layout);
    native static final protected void pango_layout_set_single_paragraph_mode (int layout, boolean 
        setting);
    native static final protected boolean pango_layout_get_single_paragraph_mode (int layout);
    native static final protected void pango_layout_context_changed (int layout);
    native static final protected void pango_layout_index_to_pos (int layout, int index, int pos);
    native static final protected void pango_layout_get_cursor_pos (int layout, int index, int [] 
        strongPos, int [] weakPos);
    native static final protected void pango_layout_move_cursor_visually (int layout, boolean 
        strong, int oldIndex, int oldTrailing, int direction, int [] newIndex, int [] newTrailing);
    native static final protected boolean pango_layout_xy_to_index (int layout, int x, int y, int []
        index, int [] trailing);
    native static final protected void pango_layout_get_extents (int layout, int [] inkRect, int [] 
        logicalRect);
    native static final protected void pango_layout_get_pixel_extents (int layout, int [] inkRect, 
        int [] logicalRect);
    native static final protected void pango_layout_get_size (int layout, int [] width, int [] 
        height);
    native static final protected void pango_layout_get_pixel_size (int layout, int [] width, int []
        height);
    native static final protected int pango_layout_get_line_count (int layout);
    native static final protected int pango_layout_get_line (int layout, int line);
    /****************************************
     * END OF GENERATED CODE
     ****************************************/
}

