/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.gtk.event.FontButtonEvent;
import org.gnu.gtk.event.FontButtonListener;
import org.gnu.gtk.event.GtkEventType;

/**
 */
public class FontButton extends Button {
	
	/**
	 * Construct a new FontButton widget
	 */
	public FontButton(){
		if (-1 == handle) {
			handle = gtk_font_button_new();
			initializeEventHandlers();
		}
	}

	/**
	 * Create a FontButton widget initialized with the
	 * provided font.
	 * 
	 * @param fontName Name of the font to display in the
	 * font selection dialog.
	 */
	public FontButton(String fontName) {
		handle = gtk_font_button_new_with_font(fontName);
		initializeEventHandlers();
	}
	
	/**
	 * Set the title for the font selection dialog.
	 * 
	 * @param title The title for the dialog.
	 */
	public void setTitle(String title) {
		gtk_font_button_set_title(handle, title);
	}
	
	/**
	 * Get the title for the font selection dialog.
	 * 
	 * @return The title for the font selection dialog.
	 */
	public String getTitle() {
		return gtk_font_button_get_title(handle);
	}
	
	/**
	 * Set whether the selected font is used in the label.
	 * 
	 * @param useFont If ture the font name will be written
	 * using the chosen font.
	 */
	public void setUseFont(boolean useFont) {
		gtk_font_button_set_use_font(handle, useFont);
	}
	
	/**
	 * Returns whether the selected font is used in the label.
	 * 
	 * @return true if the selected font is used in the label.
	 */
	public boolean getUseFont() {
		return gtk_font_button_get_use_font(handle);
	}
	
	/**
	 * Set whether the selected size is used in the label.
	 * 
	 * @param useSize If ture the font name will be written
	 * using the chosen font size.
	 */
	public void setUseSize(boolean useSize) {
		gtk_font_button_set_use_size(handle, useSize);
	}
	
	/**
	 * Returns whether the selected size is used in the label.
	 * 
	 * @return true if the selected size is used in the label.
	 */
	public boolean getUseSize(){
		return gtk_font_button_get_use_size(handle);
	}
	
	/**
	 * Sets whether the name of the font style will be shown in the label.
	 * 
	 * @param showStyle 
	 */
	public void setShowStyle(boolean showStyle) {
		gtk_font_button_set_show_style(handle, showStyle);
	}
	
	/**
	 * Returns whether the name of the font style will be shown in the label.
	 * 
	 * @return true if the font style will be shown in the label.
	 */
	public boolean getShowStyle() {
		return gtk_font_button_get_show_style(handle);
	}
	
	/**
	 * Sets whether the size of the font style will be shown in the label.
	 * 
	 * @param showSize
	 */
	public void setShowSize(boolean showSize) {
		gtk_font_button_set_show_size(handle, showSize);
	}
	
	/**
	 * Returns whether the name of the font style will be shown in the label.
	 * 
	 * @return true if the font style will be shown in the label.
	 */
	public boolean getShowSize() {
		return gtk_font_button_get_show_size(handle);
	}
	
	/**
	 * Sets or changes the currently selected font.
	 * 
	 * @param fontName The name of the font to be selected.
	 */
	public void setFont(String fontName) {
		gtk_font_button_set_font_name(handle, fontName);
	}
	
	/**
	 * Returns the name of the currently selected font.
	 * 
	 * @return The name of the currently selected font.
	 */
	public String getFont() {
		return gtk_font_button_get_font_name(handle);
	}
	

	/***************************************
	 * EVENT LISTENERS
	 ****************************************/

	/**
	 * Listeners for handling dialog events
	 */
	private Vector fbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see FontButtonListener
	 */
	public void addListener(FontButtonListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(fbListeners, listener);
		if (i == -1) {
			if (null == fbListeners) {
				fbListeners = new Vector();
			}
			fbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(FontButtonListener)
	 */
	public void removeListener(FontButtonListener listener) {
		int i = findListener(fbListeners, listener);
		if (i > -1) {
			fbListeners.remove(i);
		}
		if (0 == fbListeners.size()) {
			fbListeners = null;
		}
	}

	protected void fireFontButtonEvent(FontButtonEvent event) {
		if (null == fbListeners) {
			return;
		}
		int size = fbListeners.size();
		int i = 0;
		while (i < size) {
			FontButtonListener fbl = (FontButtonListener)fbListeners.elementAt(i);
			fbl.fontButtonEvent(event);
			i++;
		}
	}

	private void handleFontSet() {
		fireFontButtonEvent(new FontButtonEvent(this, FontButtonEvent.Type.FONT_SET));
	}

	protected void initializeEventHandlers() {
		super.initializeEventHandlers();
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		return evtMap.getListenerClass(signal);
	}

	public GtkEventType getEventType(String signal) {
		return evtMap.getEventType(signal);
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("font_set", "handleFontSet", FontButtonEvent.Type.FONT_SET, FontButtonListener.class);
	}
	
	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_font_button_get_type ();
	native static final protected int gtk_font_button_new();
	native static final protected int gtk_font_button_new_with_font(String fontname);
	native static final protected String gtk_font_button_get_title(int fontButton);
	native static final protected void gtk_font_button_set_title(int fontButton, String title);
	native static final protected boolean gtk_font_button_get_use_font(int fontButton);
	native static final protected void gtk_font_button_set_use_font(int fontButton, boolean useFont);
	native static final protected boolean gtk_font_button_get_use_size(int fontButton);
	native static final protected void gtk_font_button_set_use_size(int fontButton, boolean useSize);
	native static final protected String gtk_font_button_get_font_name(int fontButton);
	native static final protected boolean gtk_font_button_set_font_name(int fontButton, String fontname);
	native static final protected boolean gtk_font_button_get_show_style(int fontButton);
	native static final protected void gtk_font_button_set_show_style(int fontButton, boolean showStyle);
	native static final protected boolean gtk_font_button_get_show_size(int fontButton);
	native static final protected void gtk_font_button_set_show_size(int fontButton, boolean showSize);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
